/* Copyright(C) 2004 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifndef _SEN_IO_H
#define _SEN_IO_H

#ifndef _SENNA_H
#include "senna_in.h"
#endif /* _SENNA_H */

#ifdef	__cplusplus
extern "C" {
#endif

#ifdef WIN32
#define SEN_IO_FILE_SIZE               134217728L
#else /* WIN32 */
#define SEN_IO_FILE_SIZE               1073741824UL
#endif /* WIN32 */

typedef enum {
  sen_io_rdonly,
  sen_io_wronly,
  sen_io_rdwr
} sen_io_rw_mode;

typedef enum {
  sen_io_auto,
  sen_io_manual
} sen_io_mode;

typedef struct _sen_io sen_io;

typedef struct {
  int32_t segno;
  void *addr;
} sen_io_seginfo;

typedef struct {
  sen_io *io;
  sen_io_rw_mode mode;
  uint32_t segment;
  uint32_t offset;
  uint32_t size;
  uint32_t nseg;
  off_t pos;
  void *addr;
  uint32_t diff;
  int cached;
} sen_io_win;

sen_io *sen_io_create(const char *path, uint32_t header_size, uint32_t segment_size,
		      uint32_t max_segment, sen_io_mode mode, unsigned int cache_size);
sen_io *sen_io_open(const char *path, sen_io_mode mode, uint32_t cache_size);
sen_rc sen_io_close(sen_io *io);
sen_rc sen_io_remove(const char *path);
sen_rc sen_io_size(sen_io *io, off_t *size);
sen_rc sen_io_rename(const char *old_name, const char *new_name);
void *sen_io_header(sen_io *io);

void *sen_io_win_map(sen_io *io, sen_io_win *iw, uint32_t segment,
		     uint32_t offset, uint32_t size, sen_io_rw_mode mode);
sen_rc sen_io_win_mapv(sen_io_win **list, int nent);
sen_rc sen_io_win_unmap(sen_io_win *iw);

void * sen_io_seg_map(sen_io *io, int32_t segno);
sen_rc sen_io_seg_unmap(sen_io *io, int32_t segno);

sen_rc sen_io_seg_bind(sen_io *io, sen_io_seginfo *si);
sen_rc sen_io_seg_unbind(sen_io *io, sen_io_seginfo *si);

uint32_t sen_io_base_seg(sen_io *io);
const char *sen_io_path(sen_io *io);

#ifdef __cplusplus
}
#endif

/* encode/decode */

#define SEN_B_ENC(v,p) \
{ \
  uint8_t *_p = (uint8_t *)p; \
  uint32_t _v = v; \
  if (_v < 0x8f) { \
    *_p++ = _v; \
  } else if (_v < 0x408f) { \
    _v -= 0x8f; \
    *_p++ = 0xc0 + (_v >> 8); \
    *_p++ = _v & 0xff; \
  } else if (_v < 0x20408f) { \
    _v -= 0x408f; \
    *_p++ = 0xa0 + (_v >> 16); \
    *_p++ = (_v >> 8) & 0xff; \
    *_p++ = _v & 0xff; \
  } else if (_v < 0x1020408f) { \
    _v -= 0x20408f; \
    *_p++ = 0x90 + (_v >> 24); \
    *_p++ = (_v >> 16) & 0xff; \
    *_p++ = (_v >> 8) & 0xff; \
    *_p++ = _v & 0xff; \
  } else { \
    *_p++ = 0x8f; \
    *((uint32_t *)_p) = _v; \
    _p += sizeof(uint32_t); \
  } \
  p = _p; \
}

#define SEN_B_DEC(v,p) \
{ \
  uint8_t *_p = (uint8_t *)p; \
  uint32_t _v = *_p++; \
  switch (_v >> 4) { \
  case 0x08 : \
    if (_v == 0x8f) { \
      _v = *((uint32_t *)_p); \
      _p += sizeof(uint32_t); \
    } \
    break; \
  case 0x09 : \
    _v = (_v - 0x90) * 0x100 + *_p++; \
    _v = _v * 0x100 + *_p++; \
    _v = _v * 0x100 + *_p++ + 0x20408f; \
    break; \
  case 0x0a : \
  case 0x0b : \
    _v = (_v - 0xa0) * 0x100 + *_p++; \
    _v = _v * 0x100 + *_p++ + 0x408f; \
    break; \
  case 0x0c : \
  case 0x0d : \
  case 0x0e : \
  case 0x0f : \
    _v = (_v - 0xc0) * 0x100 + *_p++ + 0x8f; \
    break; \
  } \
  v = _v; \
  p = _p; \
}

#define SEN_B_SKIP(p) \
{ \
  uint8_t *_p = (uint8_t *)p; \
  uint32_t _v = *_p++; \
  switch (_v >> 4) { \
  case 0x08 : \
    if (_v == 0x8f) { \
      _p += sizeof(uint32_t); \
    } \
    break; \
  case 0x09 : \
    _p += 3; \
    break; \
  case 0x0a : \
  case 0x0b : \
    _p += 2; \
    break; \
  case 0x0c : \
  case 0x0d : \
  case 0x0e : \
  case 0x0f : \
    _p += 1; \
    break; \
  } \
  p = _p; \
}

#define SEN_B_COPY(p2,p1) \
{ \
  uint32_t size = 0, _v = *p1++; \
  *p2++ = _v; \
  switch (_v >> 4) { \
  case 0x08 : \
    size = (_v == 0x8f) ? 4 : 0; \
    break; \
  case 0x09 : \
    size = 3; \
    break; \
  case 0x0a : \
  case 0x0b : \
    size = 2; \
    break; \
  case 0x0c : \
  case 0x0d : \
  case 0x0e : \
  case 0x0f : \
    size = 1; \
    break; \
  } \
  while (size--) { *p2++ = *p1++; } \
}

#endif /* _SEN_IO_H */
