/*
 * Copyright(C) 2007 Jiro Nishiguchi <jiro@cpan.org>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "senna_java.h"
#include "senna_Index.h"

static sen_index *
this_sen_index(JNIEnv *env, jobject obj)
{
    jclass clazz;
    jfieldID id;
    clazz = (*env)->FindClass(env, "senna/Index");
    id = (*env)->GetFieldID(env, clazz, "ptr", "I");
    return (sen_index *) (*env)->GetIntField(env, obj, id);
}
static void
set_sen_index(JNIEnv *env, jobject obj, sen_index *index)
{
    jclass clazz;
    jfieldID id;
    clazz = (*env)->FindClass(env, "senna/Index");
    id = (*env)->GetFieldID(env, clazz, "ptr", "I");
    (*env)->SetIntField(env, obj, id, (jint) index);
}

JNIEXPORT jobject JNICALL
Java_senna_Index_create(JNIEnv *env, jclass clazz, jstring path_str,
        jint key_size, jint flags, jint initial_n_segments, jint encoding)
{
    const char *path;
    sen_index *index;
    jmethodID constructor;
    jobject obj;
    path = (*env)->GetStringUTFChars(env, path_str, NULL);
    index = sen_index_create(path, key_size, flags, initial_n_segments, encoding);
    (*env)->ReleaseStringUTFChars(env, path_str, path);
    if (index == NULL)
        ThrowNewSennaException(env, "Failed to call sen_index_create().", 0);
    constructor = (*env)->GetMethodID(env, clazz, "<init>", "()V");
    obj = (*env)->NewObject(env, clazz, constructor);
    set_sen_index(env, obj, index);
    return obj;
}

JNIEXPORT void JNICALL
Java_senna_Index_rename(JNIEnv *env, jclass clazz, jstring old_name_str, jstring new_name_str)
{
    const char *old_name, *new_name;
    sen_rc rc;
    old_name = (*env)->GetStringUTFChars(env, old_name_str, NULL);
    new_name = (*env)->GetStringUTFChars(env, new_name_str, NULL);
    rc = sen_index_rename(old_name, new_name);
    (*env)->ReleaseStringUTFChars(env, old_name_str, old_name);
    (*env)->ReleaseStringUTFChars(env, new_name_str, new_name);
    if (rc != sen_success)
        ThrowNewSennaException(env, "Failed to call sen_index_rename().", rc);
}

JNIEXPORT void JNICALL
Java_senna_Index_remove(JNIEnv *env, jclass clazz, jstring path_str)
{
    const char *path;
    sen_rc rc;
    path = (*env)->GetStringUTFChars(env, path_str, NULL);
    rc = sen_index_remove(path);
    (*env)->ReleaseStringUTFChars(env, path_str, path);
    if (rc != sen_success)
        ThrowNewSennaException(env, "Failed to call sen_index_remove().", rc);
}

JNIEXPORT void JNICALL
Java_senna_Index_open(JNIEnv *env, jobject obj, jstring path_str)
{
    const char *path;
    sen_index *index;
    path = (*env)->GetStringUTFChars(env, path_str, NULL);
    index = sen_index_open(path);
    (*env)->ReleaseStringUTFChars(env, path_str, path);
    if (index == NULL)
        ThrowNewSennaException(env, "Failed to call sen_index_open().", 0);
    set_sen_index(env, obj, index);
}

JNIEXPORT void JNICALL
Java_senna_Index_insert(JNIEnv *env, jobject obj, jstring key_str, jstring val_str)
{
    sen_index *index;
    const char *key, *val;
    jsize len;
    sen_rc rc;
    index = this_sen_index(env, obj);
    key = (*env)->GetStringUTFChars(env, key_str, NULL);
    val = (*env)->GetStringUTFChars(env, val_str, NULL);
    len = (*env)->GetStringUTFLength(env, val_str);
    rc = sen_index_upd(index, key, NULL, 0, val, len);
    (*env)->ReleaseStringUTFChars(env, key_str, key);
    (*env)->ReleaseStringUTFChars(env, val_str, val);
    if (rc != sen_success)
        ThrowNewSennaException(env, "Failed to call sen_index_upd().", rc);
}

JNIEXPORT void JNICALL
Java_senna_Index_delete(JNIEnv *env, jobject obj, jstring key_str,
        jstring old_str)
{
    sen_index *index;
    const char *key, *old;
    jsize len;
    sen_rc rc;
    index = this_sen_index(env, obj);
    key = (*env)->GetStringUTFChars(env, key_str, NULL);
    old = (*env)->GetStringUTFChars(env, old_str, NULL);
    len = (*env)->GetStringUTFLength(env, old_str);
    rc = sen_index_upd(index, key, old, len, NULL, 0);
    (*env)->ReleaseStringUTFChars(env, key_str, key);
    (*env)->ReleaseStringUTFChars(env, old_str, old);
    if (rc != sen_success)
        ThrowNewSennaException(env, "Failed to call sen_index_upd().", rc);
}

JNIEXPORT void JNICALL
Java_senna_Index_upd(JNIEnv *env, jobject obj, jstring key_str,
        jstring old_val_str, jstring new_val_str)
{
    sen_index *index;
    const char *key, *old_val, *new_val;
    jsize old_len, new_len;
    sen_rc rc;
    index = this_sen_index(env, obj);
    key = (*env)->GetStringUTFChars(env, key_str, NULL);
    old_val = (*env)->GetStringUTFChars(env, old_val_str, NULL);
    new_val = (*env)->GetStringUTFChars(env, new_val_str, NULL);
    old_len = (*env)->GetStringUTFLength(env, old_val_str);
    new_len = (*env)->GetStringUTFLength(env, new_val_str);
    rc = sen_index_upd(index, key, old_val, old_len, new_val, new_len);
    (*env)->ReleaseStringUTFChars(env, key_str, key);
    (*env)->ReleaseStringUTFChars(env, old_val_str, old_val);
    (*env)->ReleaseStringUTFChars(env, new_val_str, new_val);
    if (rc != sen_success)
        ThrowNewSennaException(env, "Failed to call sen_index_upd().", rc);
}

JNIEXPORT void JNICALL
Java_senna_Index_printInfo(JNIEnv *env, jobject obj)
{
    sen_index *index;
    int key_size;
    int flags;
    int initial_n_segments;
    sen_encoding encoding;
    unsigned int nrecords_keys;
    unsigned int file_size_keys;
    unsigned int nrecords_lexicon;
    unsigned int file_size_lexicon;
    unsigned int inv_seg_size;
    unsigned int inv_chunk_size;
    sen_rc rc;

    index = this_sen_index(env, obj);
    rc = sen_index_info(index,
        &key_size, &flags, &initial_n_segments, &encoding,
        &nrecords_keys, &file_size_keys, &nrecords_lexicon,
        &file_size_lexicon, &inv_seg_size, &inv_chunk_size
    );
    if (rc != sen_success)
        ThrowNewSennaException(env, "Failed to call sen_index_info().", rc);

    printf("key_size           %24d\n", key_size);
    printf("flags              %24d\n", flags);
    printf("initial_n_segments %24d\n", initial_n_segments);
    printf("encoding           %24d\n", encoding);
    printf("nrecords_keys      %24u\n", nrecords_keys);
    printf("file_size_keys     %24u\n", file_size_keys);
    printf("nrecords_lexicon   %24u\n", nrecords_lexicon);
    printf("file_size_lexicon  %24u\n", file_size_lexicon);
    printf("inv_segment_size   %24u\n", inv_seg_size);
    printf("inv_chunk_size     %24u\n", inv_chunk_size);
}

JNIEXPORT void JNICALL
Java_senna_Index_close(JNIEnv *env, jobject obj)
{
    sen_index *index;
    sen_rc rc;
    index = this_sen_index(env, obj);
    rc = sen_index_close(index);
    if (rc != sen_success)
        ThrowNewSennaException(env, "Failed to call sen_index_close().", rc);
}


JNIEXPORT jobject JNICALL
Java_senna_Index_sel(JNIEnv *env, jobject obj, jstring query_str)
{
    sen_index *index;
    sen_records *records;
    const char *query;
    jsize len;
    jclass records_class;
    jmethodID records_constructor;
    jobject records_obj;
    query = (*env)->GetStringUTFChars(env, query_str, NULL);
    len = (*env)->GetStringUTFLength(env, query_str);
    index = this_sen_index(env, obj);
    records = sen_index_sel(index, query, len);
    (*env)->ReleaseStringUTFChars(env, query_str, query);
    if (records == NULL)
        return NULL;
    records_class = (*env)->FindClass(env, "senna/Records");
    records_constructor = (*env)->GetMethodID(env, records_class, "<init>", "(I)V");
    records_obj = (*env)->NewObject(env, records_class, records_constructor, (jint) records);
    return records_obj;
}

