/*
 * Copyright(C) 2007 Jiro Nishiguchi <jiro@cpan.org>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "senna_java.h"
#include "senna_Snippet.h"

static sen_snip *
this_sen_snip(JNIEnv *env, jobject obj)
{
    jclass clazz;
    jfieldID id;
    clazz = (*env)->FindClass(env, "senna/Snippet");
    id = (*env)->GetFieldID(env, clazz, "ptr", "I");
    return (sen_snip *) (*env)->GetIntField(env, obj, id);
}

JNIEXPORT void JNICALL
Java_senna_Snippet_open(JNIEnv *env, jobject obj, jint encoding, jint flags, jint width,
        jint max_results, jstring defaultopentag, jstring defaultclosetag, jint mapping)
{
    sen_snip *snip;
    const char *open_str, *close_str;
    unsigned int open_len, close_len;
    jclass clazz;
    jfieldID fieldID;
    open_str = (*env)->GetStringUTFChars(env, defaultopentag, NULL);
    close_str = (*env)->GetStringUTFChars(env, defaultclosetag, NULL);
    open_len = (*env)->GetStringUTFLength(env, defaultopentag);
    close_len = (*env)->GetStringUTFLength(env, defaultclosetag);
    snip = sen_snip_open(encoding, flags, width, max_results, open_str,
            open_len, close_str, close_len, (sen_snip_mapping *) mapping);
    (*env)->ReleaseStringUTFChars(env, defaultopentag, open_str);
    (*env)->ReleaseStringUTFChars(env, defaultclosetag, close_str);
    if (snip == NULL)
        ThrowNewSennaException(env, "Failed to call sen_snip_open().", 0);
    clazz = (*env)->FindClass(env, "senna/Snippet");
    fieldID = (*env)->GetFieldID(env, clazz, "ptr", "I");
    (*env)->SetIntField(env, obj, fieldID, (jint) snip);
}

JNIEXPORT void JNICALL
Java_senna_Snippet_addCond(JNIEnv *env, jobject obj, jstring keyword_str, jstring opentag_str,
        jstring closetag_str)
{
    sen_snip *snip;
    const char *keyword, *opentag, *closetag;
    unsigned int keyword_len, opentag_len, closetag_len;
    sen_rc rc;
    snip = this_sen_snip(env, obj);
    keyword = (*env)->GetStringUTFChars(env, keyword_str, NULL);
    opentag = (*env)->GetStringUTFChars(env, opentag_str, NULL);
    closetag = (*env)->GetStringUTFChars(env, closetag_str, NULL);
    keyword_len = (*env)->GetStringUTFLength(env, keyword_str);
    opentag_len = (*env)->GetStringUTFLength(env, opentag_str);
    closetag_len = (*env)->GetStringUTFLength(env, closetag_str);
    rc = sen_snip_add_cond(snip, keyword, keyword_len, opentag, opentag_len, closetag,
            closetag_len);
    (*env)->ReleaseStringUTFChars(env, keyword_str, keyword);
    (*env)->ReleaseStringUTFChars(env, opentag_str, opentag);
    (*env)->ReleaseStringUTFChars(env, closetag_str, closetag);
    if (rc != sen_success)
        ThrowNewSennaException(env, "Failed to call sen_snip_add_cond().", rc);
}

JNIEXPORT jobjectArray JNICALL
Java_senna_Snippet_exec(JNIEnv *env, jobject obj, jstring jstr)
{
    sen_snip *snip;
    const char *string;
    unsigned int string_len, nresults, max_tagged_len;
    sen_rc rc;
    jclass string_class;
    jobjectArray array;
    int i;
    snip = this_sen_snip(env, obj);
    string = (*env)->GetStringUTFChars(env, jstr, NULL);
    string_len = (*env)->GetStringUTFLength(env, jstr);
    rc = sen_snip_exec(snip, string, string_len, &nresults, &max_tagged_len);
    (*env)->ReleaseStringUTFChars(env, jstr, string);
    if (rc != sen_success)
        ThrowNewSennaException(env, "Failed to call sen_snip_exec().", rc);
    string_class = (*env)->FindClass(env, "java/lang/String");
    array = (*env)->NewObjectArray(env, nresults, string_class, NULL);
    if (array == NULL)
        ThrowNewSennaException(env, "Failed to create Result Array.", 0);
    for (i = 0; i < nresults; i++) {
        char result[max_tagged_len];
        unsigned int result_len;
        jstring result_str;
        rc = sen_snip_get_result(snip, i, result, &result_len);
        if (rc != sen_success)
            ThrowNewSennaException(env, "Failed to call sen_snip_get_result().", rc);
        result_str = (*env)->NewStringUTF(env, result);
        (*env)->SetObjectArrayElement(env, array, i, result_str);
    }
    return array;
}

JNIEXPORT void JNICALL
Java_senna_Snippet_close(JNIEnv *env, jobject obj)
{
    sen_snip *snip;
    sen_rc rc;
    snip = this_sen_snip(env, obj);
    rc = sen_snip_close(snip);
    if (rc != sen_success)
        ThrowNewSennaException(env, "Failed to call sen_snip_close().", rc);
}

