/*
#! SELinux Policy Editor, a simple editor for SELinux policies
#! Copyright (C) 2005-2007 Yuichi Nakamura
#! Copyright (C) 2003-2007 Hitachi Software
#! 
#! This program is free software; you can redistribute it and/or modify
#! it under the terms of the GNU General Public License as published by
#! the Free Software Foundation; either version 2 of the License, or
#! (at your option) any later version.
#! 
#! This program is distributed in the hope that it will be useful,
#! but WITHOUT ANY WARRANTY; without even the implied warranty of
#! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#! GNU General Public License for more details.
#! 
#! You should have received a copy of the GNU General Public License
#! along with this program; if not, write to the Free Software
#! Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include "action.h"
#include "convert.h"
#include "initial_policy.h"
#include "security_class.h"
#include "preprocess_include.h"
#include "out_macro.h"

extern FILE *yyin;
extern int yyparse(void);

#define DEFAULT_INFILE		"test.conf"
#define DEFAULT_POLICY		"policy.conf"
#define DEFAULT_FILE_CONTEXT	"file_context"

const char usage[]= "Usage : seedit-converter [-t] [-p] -i <infile> -b <base policy dir> -o <output dir> --profile-data <attribute profile file>\n" \
	"	-t <number>: Tiny. Performs size optimization. Must be used with -a option.<number> is optimize level\n" \
	"	-p: Profile. Generate profile data to be used -t option\n" \
	"	-i <infile>: input SPDL file\n" \
	"	-b: base policy dir. Usually it is /usr/share/seedit/base_policy\n" \
	"	-c: <file>: Input label-file relationship for files that are labeled by file_type trans" \
	"	-o <outdir>: Result is generated in <outdir>\n"		\
	"	--profile-data <attribute profile file>: Used with -t option, specifies profile generated by -p option.\n" \
	"	--more-warning: Show more warning useful to avoid misconfiguration.\n"\
	"	--disable-boolean: Do not output conditional policy\n" \
	"	--out-file_type_trans-context: Search local file tree and find files that are labeled by file_type_trans, and output list to file_contexts file. \n";

int gDiet_by_attr = 0;
int gProfile = 0;
int gDir_search = 1;
int gNoBool = 0;
int gOutFileTypeTransContext = 0;
int gMoreWarning = 0;
char *gInFileTypeTransContext = NULL;

int main(int argc, char **argv){
  char *outdir=NULL;
  char *basedir = NULL;
  char *include_dir =NULL;
  FILE *attribute_profile = NULL;
  int ch;
  FILE *tmp;
  FILE *input=stdin;
  int option_index = 0;
  int level;
  static struct option long_options[] = {
	  {"profile", 0, 0, 'p'},
	  {"profile-data",1, 0, '\xfe'},
	  {"more-warning",0, 0, '\xfc'},
	  {"disable-boolean",0, 0, '\xfd'},
	  {"out-file_type_trans-context", 0, 0 ,'\xff'},
	  {0, 0, 0, 0}

  };

  if (argc == 1) {
	  fprintf(stderr, "%s\n", usage);
	  exit(1);
  }
  while ((ch = getopt_long(argc, argv, "\xfe:t:po:i:b:I:c:\xfd\xff\xfc", long_options, 
			   &option_index)) != EOF) {
	  switch (ch) {
	  case 't':
		  level = atoi(optarg);
		  if (level >= 1) {
			  gDiet_by_attr = 1;
			  fprintf(stderr, "#I'll try to reduce policy size by using attribute.\n");
		  }
		  if (level >= 2) {
			  gDir_search = 0;
			  fprintf(stderr, "#I'll try to reduce policy size by not supporting dir search permission. However, it will reduce security too.\n");
		  }
		  break;
		  
	  case 'p':
		  gProfile = 1;
		  break;
	  case 'o':
		  outdir = strdup(optarg);
		  break;
	  case '\xfe':
		  if((attribute_profile = fopen(optarg, "r")) == NULL) {
			  perror(optarg);
			  exit(1);
		  }
		  break;
	  case '\xfc':
		  gMoreWarning = 1;
		  break;
	  case '\xfd':
		  gNoBool = 1;
		  break;
	  case '\xff':
		  gOutFileTypeTransContext = 1;
		  break;
	  case 'i':
		  if ((input = fopen(optarg, "r")) == NULL){
			  perror(optarg);
			  exit(1);
		  }	
		  break;
		  
	  case 'b':		  
		  if (optarg == NULL){
			  basedir = DEFAULT_BASE_POLICY_DIR;
		  }else{
			  basedir = optarg;
		  }
		  set_base_policy_files(basedir);
		  break; 

	  case 'I':
		  include_dir = strdup(optarg);
		  break;
	  case 'c':
		  gInFileTypeTransContext = strdup(optarg);
		  break;
	  default:
		  fprintf(stderr, "%s\n", usage);
		  exit(1);
		  break;
	  }
  }
  if(basedir == NULL) { 
	  basedir = DEFAULT_BASE_POLICY_DIR;
	  set_base_policy_files(basedir);	  
  }
  
  tmp = tmpfile();
  if (!tmp) {
	  fprintf(stderr,"Error opening tmpfile\n");
	  exit(1);
  }
  if (gProfile)
	  fprintf(stderr, "Taking profile data.\n");
  /*do preprocess */
  preprocess_include(input, tmp,include_dir);
  rewind(tmp);
  yyin = tmp;

  /* create hash table for label table */
  defined_label_table = create_hash_table(LABEL_LIST_SIZE);
  
  /* initialize default class arrray */
  init_classes(get_base_policy_files()->security_class);
  
  parse_converter_conf();
  
  if (yyparse() != 0)
    exit(1);
 
  register_dummy_home_rule();
  append_homedir_rule();
  
  /**/
  if(attribute_profile) { 
	  init_tobe_used_attribute_table(attribute_profile);
	  fclose(attribute_profile);
  }
 
  /* convert SPDL to SELinux configuration language */
  convert(outdir);

  fclose(tmp);
  return 0;
}
