/*
 * ohci for superh
 */

#define SH7760_OHCI_NAME	"sh7760-ohci"

#define SH7760_HCD_DEBUG 10
#ifdef SH7760_HCD_DEBUG
#define DPRINTK(args...)        printk("<1>" SH7760_OHCI_NAME ": " args)
#else
#define DPRINTK(args...)
#endif


#include <linux/dma-mapping.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <asm/cpu-sh4/port.h>

static long dma_mask_limit=0xffffffff;

int usb_hcd_sh_probe(const struct hc_driver *driver,
			  struct platform_device *pdev)
{
	int retval;
	struct usb_hcd *hcd;
	struct ohci_hcd	*ohci;
	struct resource *res;
	int irq;

	DPRINTK("initializing SH7760 USB Controller\n");
	PORT_CONTROL_BIT(PORT_PHCR, 0, PMOD);
	PORT_CONTROL_BIT(PORT_PHCR, 1, PMOD);
	PORT_CONTROL_BIT(PORT_PHCR, 2, PMOD);
	ctrl_outl(0x00020000,0xFE3C008C); /* DMAUCR */

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	irq = platform_get_irq(pdev, 0);


	hcd = usb_create_hcd(driver, &pdev->dev, "SH7760 USB");
	if (!hcd)
		return -ENOMEM;
	hcd->rsrc_start = res->start;
	hcd->rsrc_len = res->end - res->start + 1;
	pdev->dev.dma_mask = &dma_mask_limit;
	if (!request_mem_region(hcd->rsrc_start, hcd->rsrc_len, hcd_name)) {
		pr_debug(__FILE__ ": request_mem_region failed\n");
		retval = -EBUSY;
		goto err1;
	}
	hcd->regs = hcd->rsrc_start;
	if (!hcd->regs) {
		pr_debug(__FILE__ ": ioremap failed\n");
		retval = -ENOMEM;
		goto err2;
	}
	ohci = hcd_to_ohci(hcd);
	ohci_hcd_init(ohci);

	retval = usb_add_hcd(hcd, irq, IRQF_DISABLED);
	if (retval == 0){
		// Good sequence
		return retval;
	}
	
	// error sequence
	DPRINTK("Removing SH7760 USB Controller\n");

 err2:
	release_mem_region(hcd->rsrc_start, hcd->rsrc_len);
 err1:
 	usb_put_hcd(hcd);
	return retval;
}


/* may be called without controller electrically present */
/* may be called with controller, bus, and devices active */

/**
 * usb_hcd_ppc_soc_remove - shutdown processing for On-Chip HCDs
 * @pdev: USB Host Controller being removed
 * Context: !in_interrupt()
 *
 * Reverses the effect of usb_hcd_ppc_soc_probe().
 * It is always called from a thread
 * context, normally "rmmod", "apmd", or something similar.
 *
 */
void usb_hcd_sh_remove(struct usb_hcd *hcd,
		struct platform_device *pdev)
{
	usb_remove_hcd(hcd);

	DPRINTK("stopping SH7760 USB Controller\n");

	release_mem_region(hcd->rsrc_start, hcd->rsrc_len);
	usb_put_hcd(hcd);
}

int __devinit ohci_sh_start(struct usb_hcd *hcd)
{
	struct ohci_hcd	*ohci = hcd_to_ohci(hcd);
	int		ret;

	if ((ret = ohci_init(ohci)) < 0)
		return ret;

	if ((ret = ohci_run(ohci)) < 0) {
		err("can't start %s", ohci_to_hcd(ohci)->self.bus_name);
		ohci_stop(hcd);
		return ret;
	}

	return 0;
}


static const struct hc_driver ohci_sh_hc_driver = {
	.description =		hcd_name,
	.product_desc =         "SH7760 OHCI",
	.hcd_priv_size =	sizeof(struct ohci_hcd),

	/*
	 * generic hardware linkage
	 */
	.irq =			ohci_irq,		//
	.flags =		HCD_USB11 | HCD_MEMORY,

	/*
	 * basic lifecycle operations
	 */
	.start =		ohci_sh_start,
	.stop =			ohci_stop,
	.shutdown               = ohci_shutdown,
	/*
	 * managing i/o requests and associated device resources
	 */
	.urb_enqueue =		ohci_urb_enqueue,
	.urb_dequeue =		ohci_urb_dequeue,
	.endpoint_disable =	ohci_endpoint_disable,

	/*
	 * scheduling support
	 */
	.get_frame_number =	ohci_get_frame,

	/*
	 * root hub support
	 */
	.hub_status_data =	ohci_hub_status_data,
	.hub_control =		ohci_hub_control,
	.hub_irq_enable         = ohci_rhsc_enable,
	.start_port_reset =	ohci_start_port_reset,
};

/* ------------------------------------------------------------------ */

int ohci_sh_drv_probe(struct platform_device *pdev)
{
	int ret;

	if (usb_disabled())
		return -ENODEV;

	ret = usb_hcd_sh_probe(&ohci_sh_hc_driver, pdev);
	return ret;
}

int ohci_sh_drv_remove(struct platform_device *pdev)
{
	struct usb_hcd *hcd = platform_get_drvdata(pdev);

	usb_hcd_sh_remove(hcd, pdev);
	return 0;
}

static struct platform_driver usb_hcd_sh7760_driver = {
	.driver = {
		.owner  = THIS_MODULE,
		.name   = SH7760_OHCI_NAME,
		.bus	= &platform_bus_type
	},
	.probe          = ohci_sh_drv_probe,
	.remove         = ohci_sh_drv_remove,
};

#define SH7760_USB_IRQ		64
#define SH7760_USB_BASE         (0xFE340000)
#define SH7760_USB_MEM          (0xFE341000)
#define SH7760_USB_MEM_SIZE     (0x2000)

static struct resource sh7760_hcd_res[] = {
        [0] = {
                .start  = SH7760_USB_BASE,
                .end    = SH7760_USB_BASE + 0xfff,
                .flags  = IORESOURCE_MEM,
        },
        [1] = {
                .start  = SH7760_USB_IRQ,
                .end    = SH7760_USB_IRQ,
                .flags  = IORESOURCE_IRQ,
        },
};

static struct platform_device usb_hcd_sh_dev = {
        .name           = SH7760_OHCI_NAME,
        .id             = 0,
        .resource       = sh7760_hcd_res,
        .num_resources  = ARRAY_SIZE(sh7760_hcd_res),
};

/* ------------------------------------------------------------------ */

int __init ohci_hcd_sh_init(void)
{
	DPRINTK(DRIVER_INFO " (SH7760)\n");
	DPRINTK("block sizes: ed %d td %d\n",
		sizeof(struct ed),sizeof(struct td));

	// EBIHARA onchip_mem_init();
#if 0
	platform_driver_register(&usb_hcd_sh7760_driver);
#endif
	platform_device_register(&usb_hcd_sh_dev);
	return 0;
}

void __exit ohci_hcd_sh_cleanup(void)
{
	platform_device_unregister(&usb_hcd_sh_dev);
#if 0
	platform_driver_unregister(&usb_hcd_sh7760_driver);
#endif
}

module_init(ohci_hcd_sh_init);
module_exit(ohci_hcd_sh_cleanup);
