/*
 * All Rights Reserved, Copyright (C) 2003, Hitachi Software Engineering Co., Ltd.
 */
/* $Id: out_file_acl.c,v 1.1.1.1 2003/01/20 02:39:36 ueno Exp $ */

#include <stdio.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <flask.h>
#include <selinux.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "global.h"
#include "debug.h"
#include "hashtab.h"
#include "action.h"

/**
 *  this is used in "print_allow_consider_child".
 *  if the parentname is the same,
 *  distance between filename and parentname is the nearest will survive.
 */
typedef struct child_allow
{
	char	*filename;	/* target file name	*/
	char	*parentname;	/* parent file name	*/
	int	perm;		/* permittion		*/
} CHILD_ALLOW_BUF;

/**
 *  hash table the element if CHILD_ALLOW_BUF
 */
static HASH_TABLE *child_buf_table = NULL;
#define CHILD_BUF_TABLE_SIZE 1024

/**
 *  @name:	chk_child_dir
 *  @about:	check wether target path includes parent path 
 *  @args:	s (char *) -> parent path
 *  @args:	t (char *) -> child path
 *  @return:	if t is the child directory of s,then return 1
 *		s must not end with "/"
 */
int
chk_child_dir(char *s, char *t)
{
	int len_s;
	int len_t;

	if (strcmp(s, "/") == 0)
	{
		return 1;
	}

	len_s = strlen(s);
	len_t = strlen(t);

	if (len_s >= len_t)
		return 0;

	if (strncmp(s, t, len_s) != 0)
		return 0;

	if (t[len_s] == '/')
		return 1;

	return 0;
}

/**
 *  @name:	chop_slash	
 *  @about:	chop slash from target string 
 *  @args:	s (char *) -> target string
 *  @return:	none
 */
void
chop_slash(char *s)
{
	int len;

	if (strcmp(s, "/") == 0)
	{
		return ;
	}

	len = strlen(s);

	if (s[len-1] == '/')
	{
		s[len-1] = '\0';
	}
}

/**
 *  @name:	check_child_file	
 *  @about:	check child file name 
 *  @args:	old_s (char *) -> old file name
 *  @args:	t (char *) -> target file name
 *  @return:	if t is the file under s return 1
 */
int
chk_child_file(char *old_s, char *t)
{
	int len_s;
	int len_t;
	char *s;
	DIR *dp;
	char *fullname;
	int fullnamelen;
	struct dirent *p;
	struct stat buf;

	s = strdup(old_s);
	chop_slash(s);

	len_s = strlen(s);
	len_t = strlen(t);

	if (len_s >= len_t)
	{
		free(s);
		return 0;
	}

	if (strncmp(s, t, len_s) != 0)
	{
		free(s);
		return 0;
	}

	stat(t, &buf);
	if (buf.st_mode & S_IFDIR)
	{
		free(s);
		return 0;
	}

	if ((dp = opendir(s)) == NULL)
	{
		return 0;
	}

	while ((p = readdir(dp)) != NULL)
	{
		if(strcmp(p->d_name, "..") == 0 ||
		   strcmp(p->d_name, ".") ==0)
			continue;

		fullnamelen = strlen(s) + strlen(p->d_name);
		fullname = (char *)malloc(sizeof(char)*(2+fullnamelen));
		if (strcmp(s, "/") == 0)
		{
			sprintf(fullname, "%s%s", s, p->d_name);
		}
		else
		{
			sprintf(fullname, "%s/%s", s, p->d_name);
		}

		if (strcmp(fullname, t) == 0)
		{
			free(fullname);
			free(s);
			closedir(dp);
			return 1;
		}
		free(fullname);
	}

	free(s);
	closedir(dp);

	return 0;
}

/**
 *  @name:	distance	
 *  @about:	return the distance between filename and parent.
 *		filename must be the child of parent 
 *  @args:	filename (char *) -> target file name
 *  @args:	parent (char *) -> parent directory
 *  @return:	distance
 */
int
distance(char *filename, char *parent)
{
	int len;
	int l, k, i;
	l = k = 0;

	if (strcmp(filename, "/") == 0)
	{
		k++;
	}
	if (strcmp(parent, "/") == 0)
	{
		l++;
	}

	len = strlen(filename);
	for (i = 0; i < len; i++)
	{
		if (filename[i] == '/')
			l++;
	}

	len = strlen(parent);
	for (i = 0; i < len; i++)
	{
		if (filename[i] == '/')
			k++;
	}

	return l-k;
}

/**
 *  @name:	clear_child_buf_table	
 *  @about:	clear child buffer table 
 *  @args:	none
 *  @return:	none
 */
static void
clear_child_buf_table()
{
	int i,num;
	HASH_NODE **array;
	CHILD_ALLOW_BUF *tmp;

	num = child_buf_table->element_num;
	array = create_hash_array(child_buf_table);

	for (i = 0; i < num; i++)
	{
		tmp = (CHILD_ALLOW_BUF *)array[i]->data;
		free(tmp->filename);
		free(tmp->parentname);
		free(tmp);
	}

	delete_hash_table(child_buf_table);
	child_buf_table = NULL;
}

/**
 *  @name:	registar_child_buf_table	
 *  @about:	set value with child_buf hash table. if key:filename already exists, 
 *		register the one whose the distance to parent directory is smaller.
 *  @args:	filename (char *) -> file name
 *  @args:	parentname (char *) -> parent file name
 *  @args:	allowed (int) -> permission
 *  @return: 	none
 */
static void
registar_child_buf_table(char *filename, char *parentname, int allowed)
{
	CHILD_ALLOW_BUF *tmp;
	CHILD_ALLOW_BUF *old;
	CHILD_ALLOW_BUF work;
	int s;
	int l1,l2;

	if (child_buf_table == NULL)
	{
		child_buf_table = create_hash_table(CHILD_BUF_TABLE_SIZE);
		if (child_buf_table == NULL)
		{
			perror("malloc");
			exit(1);
		}
	}

	work.filename = strdup(filename);
	work.parentname = strdup(parentname);
	work.perm = allowed;

	if ((tmp = (CHILD_ALLOW_BUF *)malloc(sizeof(CHILD_ALLOW_BUF))) == NULL)
	{
		perror("malloc");
		exit(1);
	}
	*tmp = work;

	s = insert_element(child_buf_table,tmp,filename);
	if (s == -2)
	{
		/**
		 *  the one whose  distance to parent directory is smaller  remain
		 */
		old = search_element(child_buf_table, tmp->filename);
		if (old == NULL)
		{
			error_print(__FILE__, __LINE__, "bug!");
			exit(1);
		}

		l1 = distance(filename, parentname);
		l2 = distance(old->filename, old->parentname);

		/**
 	       	 debug_print(__FILE__, __LINE__, "%s:old:%s,new:%s,%d,%d",
			   filename, old->parentname, parentname, l2, l1);
		 */
		if (l1 < l2)
		{
			free(old->parentname);
			old->parentname = strdup(parentname);
			old->perm = allowed;
		}
	}

	return;
}

static void print_allow(char *, char *,int, FILE *);

/**
 *  @name:	print_allow_consider_child	
 *  @about:	print "allow" considering the label of child directory
 *              if the child of "filename" is labeled in file-label table,
 *              print "allow" to the label of the child. 
 *  @args:	domain (char *) -> domain name
 *  @args:	filename (char *) -> file name
 *  @args:	allowed (int) -> permission
 *  @args:	outfp (FILE *) -> output file descripter
 *  @args:	only_flag (int) -> flag
 *  @return:	none
 */
static void
print_allow_consider_child(char *domain, char *filename, int allowed, FILE *outfp, int only_flag)
{
	FILE_LABEL *label;
	HASH_NODE **file_label_array;
	DOMAIN *domain_info;
	int i;
	int child_flag;

	print_allow(domain, filename, allowed, outfp);

	file_label_array = create_hash_array(file_label_table);

	for (i = 0; i < file_label_table->element_num; i++)
	{
		label = (FILE_LABEL *)file_label_array[i]->data;

		if (only_flag == 1)
		{
			child_flag = chk_child_file(filename, label->filename);
		}
		else
		{
			child_flag = chk_child_dir(filename, label->filename);
		}

		if (child_flag == 1)
		{
			domain_info=(DOMAIN *)search_element(domain_hash_table,domain);

			if (domain_info == NULL)
			{
				fprintf(stderr, "this must be bug\n");
				exit(1);
			}

			if (search_element(domain_info->file_acl, label->filename) == NULL)
			{
				/* register it with child_buf_table and print later	*/
				registar_child_buf_table(label->filename, filename, allowed);
			}
		}
	}

	free(file_label_array);
}

/**
 *  @name:	print_allow		
 *  @about:	print "allow" based on "domain,filename,allowed" 
 *  @args:	domain (char *) -> domain name
 *  @args:	filename (char *) -> filename
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 */
static void
print_allow(char *domain, char *filename, int allowed, FILE *outfp)
{
	FILE_LABEL *label;
	DOMAIN *d;
	ADMIN_RULE adm;

	label = (FILE_LABEL *)search_element(file_label_table, filename);

	d = (DOMAIN *)search_element(domain_hash_table, domain);
	adm = d->admin_rule;

	if (allowed == DENY_PRM)
	{
		return;
	}
	fprintf(outfp, "\n#%s:%s:%s\n",label->labelname, filename, perm_to_str(allowed));

	/* common to file and directory */
	if (allowed & READ_PRM)
	{
		fprintf(outfp, "allow %s %s:dir_file_class_set ", domain,label->labelname);
		fprintf(outfp, "{ read getattr ioctl lock  };\n");
	}

	if (allowed & WRITE_PRM)
	{
		fprintf(outfp, "allow %s %s:dir_file_class_set ", domain,label->labelname);
		fprintf(outfp, "{ write setattr append create unlink link rename };\n");

		/* allowadm part_relabel	*/
		if (adm.part_relabel)
		{
			fprintf(outfp, "#allowadm relabel\n");
			fprintf(outfp, "allow %s %s:dir_file_class_set { relabelfrom relabelto };\n", domain,label->labelname);
		}
	}

	if(allowed & EXECUTE_PRM)
	{
		fprintf(outfp, "allow %s %s:dir_file_class_set ", domain, label->labelname);
		fprintf(outfp, "{ execute };\n");
	}

	/* specific to file	*/
	if (allowed & EXECUTE_PRM)
	{
		fprintf(outfp,"allow %s %s:file execute_no_trans;\n", domain, label->labelname);
	}

	/* specific to directory */
	if (allowed & READ_PRM)
	{
		;
	}

	if (allowed & SEARCH_PRM)
	{
		fprintf(outfp, "allow %s %s:dir { search getattr read  };\n", domain,label->labelname);
		fprintf(outfp, "allow %s %s:dir_file_class_set { getattr };\n", domain, label->labelname);
		fprintf(outfp, "allow %s %s:lnk_file { read };\n", domain, label->labelname);
	}

	if (allowed & WRITE_PRM)
	{
		fprintf(outfp, "allow %s %s:dir {add_name remove_name reparent rmdir};\n", domain, label->labelname);
	}
}

/**
 *  @name:	print_child_allow		
 *  @about:	print "allow" based on child_buf_table which is constructed by "print_allow_consider_child". 
 *  @args:	domain (char *) -> domain name
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return: 	none
 */
static void
print_child_allow(char *domain, FILE *outfp)
{
	HASH_NODE **array;
	CHILD_ALLOW_BUF *tmp;
	int i,num;

	if (child_buf_table == NULL)
		return;

	/* fprintf(outfp, "########Childallow\n"); */

	num = child_buf_table->element_num;
	array = create_hash_array(child_buf_table);

	for (i = 0; i < num; i++)
	{
		tmp = (CHILD_ALLOW_BUF *)array[i]->data;
		print_allow(domain, tmp->filename, tmp->perm, outfp);
	}
	clear_child_buf_table();
}

/**
 *  @name:	print_domain_allow
 *  @about:	 
 *  @args:	a (vlid *) -> File acl rule
 *  @return:	return 0
 */
static FILE *file_out_fp;
static int
print_domain_allow(void *a)
{
	FILE_ACL_RULE *acl = a;

	print_allow_consider_child(acl->domain->name, acl->path,acl->allowed, file_out_fp,acl->only_flag);
	return 0;


/*
	  if (acl->only_flag == 0)
	  {
		  print_allow_consider_child(acl->domain->name, acl->path,acl->allowed, file_out_fp);
	  }
	  else
	  {
		  print_allow(acl->domain->name, acl->path, acl->allowed, file_out_fp);
	  }
	  return 0;
*/
}

/**
 *  @name:	save_prev_label
 *  @about:	Labels labeled by "file_type_auto_trans" are overwritten 
 *              by setfiles. So,save such labels in file.
 *  @args:	path (char *) -> directory
 *  @args:	exec_label (char *) -> label name
 *  @return:	none
 */
static void
save_prev_label(char *path, char *exc_label)
{
	struct stat buf;
	security_context_t context=NULL;
	DIR *fp;
	struct dirent *dent;
	char *fullpath;
	char *type;

	if (is_selinux_enabled() != 1)
	{				/*do nothing if SELinux is off */
		return;
	}

	stat(path, &buf);
	if (!S_ISDIR(buf.st_mode))
	{
		return;
	}

	if ((fp=opendir(path)) == NULL)
	{
		fprintf(stderr,"Directory open error %s\n", path);
	}

	while ((dent = readdir(fp)) != NULL)
	{
		//    printf("%s\n", dent->d_name);
		fullpath = malloc(sizeof(char)*(strlen(path) + strlen(dent->d_name) + 4));
		sprintf(fullpath, "%s/%s", path, dent->d_name);
		stat(fullpath, &buf);

		if (lgetfilecon(fullpath, &context) == -1)
		{
			free(fullpath);
			return;
		}

		type = strrchr((char *)context, ':');

		if (type == NULL)
		{
			error_print(__FILE__, __LINE__, "bug\n");
			printf("%s,%s\n", fullpath, (char *)context);
			exit(1);
		}

		type++;
		if (strcmp(type, exc_label) == 0)
		{
			//      fprintf(stdout, "%s\t%s\n", fullpath, type);
			if (S_ISDIR(buf.st_mode))
			{
				fprintf(TMP_fp, "%s(|/.*)\tsystem_u:object_r:%s\n", fullpath, type);
			}
			else
			{
				fprintf(TMP_fp, "%s\tsystem_u:object_r:%s\n", fullpath, type);
			}
		}

		freecon(context);

		free(fullpath);
	}
}

/**
 *  @name:	out_file_type_trans
 *  @about:	print "file_type_auto_trans"
 *  @args:	outfp (FILE *) -> output file
 *  @args:	d (DOMAIN *) -> domain buffer list
 *  @return:	none
 */
static void
out_file_type_trans(FILE *outfp, DOMAIN *d)
{
	FILE_EXC_RULE e;
	int i;
	FILE_LABEL *l;

	if (d->exc_rule_array_num == 0)
		return;

	fprintf(outfp, "\n####file_type_auto_trans rule\n");

	for (i = 0; i < d->exc_rule_array_num; i++)
	{
		e = d->exc_rule_array[i];
		l = (FILE_LABEL *)search_element(file_label_table, e.path);
		save_prev_label(e.path, e.name);

		if (l == NULL)
		{
			fprintf(stderr, "bug\n");
			exit(1);
		}
		fprintf(outfp, "file_type_auto_trans(%s,%s,%s)\n", d->name, l->labelname, e.name);
	}

	fprintf(outfp, "####\n");
}

/**
 *  @name:	out_file_acl
 *  @about:	output file acl
 *  @args:	outfp (FILE *) -> output file
 *  @args:	domain (DOMAIN *) -> domain buffer list
 *  @return:	none
 */
void
out_file_acl(FILE *outfp, DOMAIN *domain)
{
	file_out_fp = outfp;
	handle_all_element(domain->file_acl, print_domain_allow);

	print_child_allow(domain->name, outfp);
	out_file_type_trans(outfp, domain);
}

