/*
 * All Rights Reserved, Copyright (C) 2003, Hitachi Software Engineering Co., Ltd.
 */
/* $Id: global.h,v 1.1.1.1 2003/01/20 02:39:36 ueno Exp $ */

#ifndef GLOBAL_H
#define GLOBAL_H
#include "hashtab.h"

/**
 *  max length of security context
 */
#define MAX_CONTEXT_LENi	4048

/**
 *  permissions used against file
 */
#define DENY_PRM		0x00
#define READ_PRM		0x01
#define WRITE_PRM		0x02
#define EXECUTE_PRM		0x04
#define APPEND_PRM		0x08
#define SEARCH_PRM		0x10
#define CHANGE_PRM		0x20

#define READ_STR		"r"
#define WRITE_STR		"w"
#define EXECUTE_STR		"x"
#define APPEND_STR		"a"
#define SEARCH_STR		"s"

/**
 *  default path and keyword
 */
#define PSID_LABEL_FILE		"/...security"
#define SECURITY_CONF_FILE	"/etc/security/selinux"
#define ROOT_LABEL		"rootdir_t"
#define DUMMY_DOMAIN_NAME	"dummy_domain_t"
#define FILE_ACL_TABLE_SIZE	1024

#define FILE_PERM		0
#define IPC_PERM		1
#define SIG_PERM		2
#define ADM_PERM		3
#define RW_PERM			4

/**
 *  permission of SIGNAL
 */
#define CHID_PRM		0x01
#define KILL_PRM		0x02
#define STOP_PRM		0x04
#define OTHERSIG_PRM		0x08

#define CHID_STR		"c"
#define KILL_STR		"k"
#define STOP_STR		"s"
#define OTHERSIG_STR		"o"

/**
 * permission of allowadm
 */
#define RELABEL_STR		"relabel"	
#define GETSECURITY_STR		"getsecurity" // replace "chsid" to "compute_relabel"
#define SETENFORCE_STR		"setenforce"  // replace "av_toggle" to "setenforce" 
#define LOAD_POLICY_STR		"load_policy"
#define NET_STR			"net"
#define BOOT_STR		"boot"
#define INSMOD_STR		"insmod"
#define QUOTAON_STR		"quotaon"
#define SWAPON_STR		"swapon"
#define MOUNT_STR		"mount"
#define RAW_IO_STR		"raw_io"
#define PTRACE_STR		"ptrace"
#define CHROOT_STR		"chroot"
#define SEARCH_ALL_STR		"search"
#define UNLABEL_STR		"unlabel"
#define READ_ALL_STR		"read"
#define WRITE_ALL_STR		"write"
#define PART_RELABEL_STR	"part_relabel"

/**
 *  DOMAIN structure
 */
typedef struct domain_t DOMAIN;

/**
 *  RBAC structure
 */
typedef struct rbac_domain_t RBAC;

/**
 *  this stores File ACL
 */
typedef struct file_acl_rule_t
{
	DOMAIN		*domain;	/* domain                */
	char		*path;		/* path name             */
	int		allowed;	/* toggle for permission */
	int		only_flag;	/* allowonly:1           */
} FILE_ACL_RULE;

/**
 *  allow exclusive
 */
typedef struct file_exc_rule_t
{
	DOMAIN		*domain;	/* domain                */
	char		*path;		/* path name             */
	char		*name;		/* file name		 */
} FILE_EXC_RULE;

/**
 *  Network permission
 */
#define CAN_TCP			0x01
#define CAN_UDP			0x02
#define CAN_RAW			0x04
#define CAN_WELLKNOWN		0x08
#define CAN_WELLKNOWN_TCP	0x10
#define CAN_WELLKNOWN_UDP	0x20
#define NONE			0x00

/**
 *  network ACL
 */
typedef struct net_rule_t
{
	DOMAIN		*domain;
	char		perm;		/*socket type allowed to use*/
					/*Well-known ports allowed to use*/
	char		tcp_port[1024+1];
	char		udp_port[1024+1];
	char		flag;		/*whether doing access control to well-known port or not */
} NET_RULE;

#define TCP_ACL		1		/* TCP socket		*/
#define UDP_ACL		2		/* UDP socket		*/
#define UNIX_ACL	3		/* UNIX domain socket	*/
#define SEM_ACL		4		/* semaphore		*/
#define MSG_ACL		5		/* message		*/
#define MSGQ_ACL	6		/* message queue	*/
#define SHM_ACL		7		/*pshared memory	*/
#define PIPE_ACL	8		/* pipe			*/
#define SIG_ACL		9		/* signal		*/
#define TMPFS_ACL	10		/* tmpfs		*/

/**
 *  IPC ACL
 */
typedef struct com_rule_t
{
	DOMAIN		*domain;
	char		flag;		/* kind of IPC ,kinds are described in #define */
	char		*domain_name;	/* the destination domain */
	int		perm;		/* allowed permission (r,w) */
} COM_ACL_RULE;

/**
 * access rights allowed by "allowadm"
 * if bit is on,the operation is allowed
 */
typedef struct admin_rule_t
{
	unsigned int	relabel:1;
	unsigned int	getsecurity:1;
	unsigned int	setenforce:1;
	unsigned int	load_policy:1;
	unsigned int	net:1;
	unsigned int	boot:1;
	unsigned int	insmod:1;
	unsigned int	quotaon:1;
	unsigned int	swapon:1;
	unsigned int	mount:1;
	unsigned int	raw_io:1;
	unsigned int	ptrace:1;
	unsigned int	chroot:1;
	unsigned int	search:1;
	unsigned int	unlabel:1;
	unsigned int	read:1;
	unsigned int	write:1;	/* experimental */
	unsigned int	part_relabel:1;	/* experimental */
} ADMIN_RULE;

/**
 *  TTY ACL
 */
typedef struct allow_tty_t
{
	DOMAIN		*domain;
	//  RBAC	*role;
	char		*rolename;	/* destination */
	int		perm;		/* allowed permission (r,w,CHANGE) */
} TTY_RULE;

/**
 *  PTS ACL
 *  access rights of domain to pts of "domainname"
 */
typedef struct allow_pts_t
{
	DOMAIN		*domain;
	char		*domain_name;	/* destination */
	// char		*change_name;
	int		perm;		/* allowed permission(r,w,CHANGE) */
} PTS_RULE;

/**
 *  ACL related to "allowproc"
 */
typedef struct allow_proc_t
{
	unsigned int	self:1;
	unsigned int	other:1;
	unsigned int	system:1;
	unsigned int	kmsg:1;
	DOMAIN		*domain;
	int		type;
	int		perm;
} PROC_RULE;

#define PROC_SELF	1
#define PROC_OTHER	2
#define PROC_SYSTEM	3
#define PROC_KMSG	4
#define PROC_PROC	5

/**
 *  The main structure.
 *  Information about domain.
 */
struct domain_t {
	char		*name;			/* name of domain		*/
	int		roleflag;		/* if "name" is role,then 1	*/

	/* File */
	HASH_TABLE	*file_acl;		/* File ACL			*/
	FILE_EXC_RULE	*exc_rule_array;	/* "allow exclusive"		*/
	int		exc_rule_array_num;

	/* net, ipc */
	NET_RULE	*net_acl;		/* network			*/
	COM_ACL_RULE	*com_acl_array;		/* IPC				*/
	int		com_acl_array_num;

	ADMIN_RULE	admin_rule;		/* allowadm			*/

	/* allowtty */
	char		tty_create_flag;	/* allowtty -create		*/
	TTY_RULE	*tty_acl_array;
	int		tty_acl_array_num;

	/* allowpts */
	char		pts_create_flag;
	PTS_RULE	*pts_acl_array;
	int		pts_acl_array_num;
	//  char *pts_change_name;

	/* allowproc */
	PROC_RULE	*proc_acl_array;
	int		proc_acl_array_num;

	/* allowtmpfs*//*permission is described in "com_acl_rule"*/
	int		tmpfs_create_flag;
};

/**
 *  domain transition 
 */
typedef struct trans_rule_t
{
	char		*parent;
	char		*path;
	char		*child;
	int		auto_flag;	/* if the transition is "domain_auto_trans",then 1 */
} TRANS_RULE;

/**
 *  association of file with label
 */
typedef struct file_label_t
{
	char		*filename;
	char		*labelname;
	char		rec_flag;	/* if the label is inherited by child directory, this is 1,else 0 */
} FILE_LABEL;


/**
 *     rbac     
 */

/**
 *  relationship between  role and domain
 */
struct rbac_domain_t {
	char *rolename;
	DOMAIN *default_domain;
	//DOMAIN **domain_array;
	//int domain_array_num;
};

/**
 *  relation between user and role
 */
typedef struct user_role_t
{
	char *username;
	char **role_name_array;
	int role_name_array_num;
} USER_ROLE;

/**
 *  global value 
 */
extern HASH_TABLE *domain_hash_table;		/* Hash table of DOMAIN structure
						   the body is in action.c			*/
extern HASH_TABLE *file_label_table;		/* relationship between file and label,
						   the body is in file_label.c			*/
extern HASH_TABLE *defined_label_table;		/* registers defined label lists
						   key:labelname,value:1(int)			*/
extern HASH_TABLE *exc_label_table;		/* registers defined allow exclusive labels
						   key:labelname,value:1(int)			*/


#define LABEL_LIST_SIZE 10000

/**
 *  domain transition
 */
extern TRANS_RULE *rulebuf;			/* the body is in action.c			*/
extern int domain_trans_rule_num;

/**
 *  relationship between user adn role
 */
extern HASH_TABLE *user_hash_table;		/* element is USER_ROLE structure		*/
extern HASH_TABLE *rbac_hash_table;		/* element is RBAC structure			*/

/**
 *  allownet
 */
extern char used_tcp_ports[1024+1];		/* body is in action.c				*/
extern char used_udp_ports[1024+1];		/* body is in action.c				*/

/**
 *  this stores labels named by "file_type_auto_trans",because these labels are overwritten by "setfiles".
 */
#define TMP_FILE_NAME "exc_label.tmp"
FILE *TMP_fp;

int yyerror(char *);
int yywarn(char *);

#endif
