// Aqsis
// Copyright (c) 1997 - 2001, Paul C. Gregory
//
// Contact: pgregory@aqsis.com
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/**
 * Copyright (C) 2006-2007  NTT DATA CORPORATION
 * 
 * Version: 1.0.0 2007/04/01
 *  
 */
package net.cellcomputing.himawari.library;

import java.lang.reflect.Method;

import net.cellcomputing.himawari.library.types.CqBitVector;
import net.cellcomputing.himawari.library.types.CqMatrix;

/**
 * Interface to shader execution environment.<br>
 * The shader execution environment is responsible for providing functionality for accessing shader data on a
 * 'shadable' item, and providing shadeops to process that data.<br>
 * 
 * @author NTT DATA Corporation
 */
public abstract strictfp class IqShaderExecEnv {

	abstract public	void	Initialise( final int uGridRes, final int vGridRes, IqAttributes pAttr, IqTransform pTrans, IqShader pShader, int Uses );
    /** Get grid size in u
     */
	abstract public	int	uGridRes();
    /** Get grid size in v
     */
	abstract public	int	vGridRes();
    /** Get total grid size.
     */
	abstract public	int	GridSize();
    /** Get the matrix which describes the transformation from Object space to World space for the surface related to this execution environment.
     */
	abstract public	CqMatrix	matObjectToWorld();
    /** Get a pointer to the associated attributes.
     */
	abstract public	IqAttributes	pAttributes();
    /** Get a pointer to the associated transform.
     */
	abstract public	IqTransform	pTransform();
	/** Set the pointer to the currently being lit surface
	 */
	abstract public void SetCurrentSurface(IqSurface pEnv);

	/** Get the pointer to the currently being lit surface
	 */
	abstract public IqSurface GetCurrentSurface();
    /** Update all cached lighting results.
     */
	abstract public	void	ValidateIlluminanceCache( IqShaderData pP, IqShaderData pN, IqShader pShader );
    /** Reset the illuminance cache.
     */
	abstract public	void	InvalidateIlluminanceCache();
    /** Get the current execution state. Bits in the vector indicate which SIMD indexes have passed the current condition.
     */
	abstract public	CqBitVector CurrentState();
    /** Get the running execution state. Bits in the vector indicate which SIMD indexes are valid.
     */
	abstract public	CqBitVector RunningState();
    /** Transfer the current state into the running state.
     */
	abstract public	void	GetCurrentState();
    /** Clear the current state ready for a new condition.
     */
	abstract public	void	ClearCurrentState();
    /** Push the running state onto the stack.
     */
	abstract public	void	PushState();
    /** Pop the running state from the stack.
     */
	abstract public	void	PopState();
    /** Invert the bits in the running state, to perform the opposite to the condition, i.e. else.
     */
	abstract public	void	InvertRunningState();
    /** Find a named standard variable in the list.
     * \param pname Character pointer to the name.
     * \return IqShaderData pointer or 0.
     */
	abstract public	IqShaderData FindStandardVar( final String pname );
    /** Find a named standard variable in the list.
     * \param pname Character pointer to the name.
     * \return Integer index in the list or -1.
     */
	abstract public	int	FindStandardVarIndex( final String pname );
    /** Get a standard variable pointer given an index.
     * \param Index The integer index returned from FindStandardVarIndex.
     * \return IqShaderData pointer.
     */
	abstract public	IqShaderData	pVar( int Index );
    /** Delete an indexed variable from the list.
     * \param Index The integer index returned from FindStandardVarIndex.
     */
	abstract public	void	DeleteVariable( int Index );
    /** Get a reference to the Cs standard variable.
     */
	abstract public	IqShaderData Cs();
    /** Get a reference to the Os standard variable.
     */
	abstract public	IqShaderData Os();
    /** Get a reference to the Ng standard variable.
     */
	abstract public	IqShaderData Ng();
    /** Get a reference to the du standard variable.
     */
	abstract public	IqShaderData du();
    /** Get a reference to the dv standard variable.
     */
	abstract public	IqShaderData dv();
    /** Get a reference to the L standard variable.
     */
	abstract public	IqShaderData L();
    /** Get a reference to the Cl standard variable.
     */
	abstract public	IqShaderData Cl();
    /** Get a reference to the Ol standard variable.
     */
	abstract public IqShaderData Ol();
    /** Get a reference to the P standard variable.
     */
	abstract public IqShaderData P();
    /** Get a reference to the dPdu standard variable.
     */
	abstract public IqShaderData dPdu();
    /** Get a reference to the dPdv standard variable.
     */
	abstract public IqShaderData dPdv();
    /** Get a reference to the N standard variable.
     */
	abstract public IqShaderData N();
    /** Get a reference to the u standard variable.
     */
	abstract public IqShaderData u();
    /** Get a reference to the v standard variable.
     */
	abstract public IqShaderData v();
    /** Get a reference to the s standard variable.
     */
	abstract public IqShaderData s();
    /** Get a reference to the t standard variable.
     */
	abstract public IqShaderData t();
    /** Get a reference to the I standard variable.
     */
	abstract public IqShaderData I();
    /** Get a reference to the Ci standard variable.
     */
	abstract public IqShaderData Ci();
    /** Get a reference to the Oi standard variable.
     */
	abstract public IqShaderData Oi();
    /** Get a reference to the Ps standard variable.
     */
	abstract public IqShaderData Ps();
    /** Get a reference to the E standard variable.
     */
	abstract public IqShaderData E();
    /** Get a reference to the ncomps standard variable.
     */
	abstract public IqShaderData ncomps();
    /** Get a reference to the time standard variable.
     */
	abstract public IqShaderData time();
    /** Get a reference to the alpha standard variable.
     */
	abstract public IqShaderData alpha();
    /** Get a reference to the Ns standard variable.
     */
	abstract public IqShaderData Ns();

	abstract public	boolean	SO_init_illuminance();
	abstract public	boolean	SO_advance_illuminance();

    // ShadeOps
	abstract public void	SO_radians( IqShaderData degrees, IqShaderData Result, IqShader pShader );
    final public void	SO_radians( IqShaderData degrees, IqShaderData Result ){
    	SO_radians( degrees, Result, null );
    }
    abstract public void	SO_degrees( IqShaderData radians, IqShaderData Result, IqShader pShader );
    final public void	SO_degrees( IqShaderData radians, IqShaderData Result ){
    	SO_degrees( radians, Result, null );
    }
    abstract public void	SO_sin( IqShaderData a, IqShaderData Result, IqShader pShader );
    final public void SO_sin( IqShaderData a, IqShaderData Result ){
    	SO_sin( a, Result, null );
    }
    abstract public void	SO_asin( IqShaderData a, IqShaderData Result, IqShader pShader );
    final public void SO_asin( IqShaderData a, IqShaderData Result ){
    	SO_asin( a, Result, null );
    }
    abstract public void	SO_cos( IqShaderData a, IqShaderData Result, IqShader pShader );
    final public void	SO_cos( IqShaderData a, IqShaderData Result ){
    	SO_cos( a, Result, null );
    }
    abstract public void	SO_acos( IqShaderData a, IqShaderData Result, IqShader pShader );
    final public void	SO_acos( IqShaderData a, IqShaderData Result ){
        SO_acos( a, Result, null );
    }
    abstract public void	SO_tan( IqShaderData a, IqShaderData Result, IqShader pShader );
    final public void	SO_tan( IqShaderData a, IqShaderData Result ){
        SO_tan( a, Result, null );
    }
    abstract public void	SO_atan( IqShaderData yoverx, IqShaderData Result, IqShader pShader );
    final public void	SO_atan( IqShaderData yoverx, IqShaderData Result ){
        SO_atan( yoverx, Result, (IqShaderData)null );
    }
    abstract public void	SO_atan( IqShaderData y, IqShaderData x, IqShaderData Result, IqShader pShader );
    final public void	SO_atan( IqShaderData y, IqShaderData x, IqShaderData Result ){
        SO_atan( y, x, Result, null );
    }
    abstract public void	SO_pow( IqShaderData x, IqShaderData y, IqShaderData Result, IqShader pShader );
    final public void	SO_pow( IqShaderData x, IqShaderData y, IqShaderData Result ){
        SO_pow( x, y, Result, null );
    }
    abstract public void	SO_exp( IqShaderData x, IqShaderData Result, IqShader pShader );
    final public void	SO_exp( IqShaderData x, IqShaderData Result ){
        SO_exp( x, Result, null );
    }
    abstract public void	SO_sqrt( IqShaderData x, IqShaderData Result, IqShader pShader );
    final public void	SO_sqrt( IqShaderData x, IqShaderData Result ){
        SO_sqrt( x, Result, null );
    }
    abstract public void	SO_log( IqShaderData x, IqShaderData Result, IqShader pShader );
    final public void	SO_log( IqShaderData x, IqShaderData Result ){
        SO_log( x, Result, (IqShaderData)null );
    }
    abstract public void	SO_log( IqShaderData x, IqShaderData base, IqShaderData Result, IqShader pShader );
    final public void	SO_log( IqShaderData x, IqShaderData base, IqShaderData Result ){
        SO_log( x, base, Result, null );
    }
    abstract public void	SO_mod( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader );
    final public void	SO_mod( IqShaderData a, IqShaderData b, IqShaderData Result ){
        SO_mod( a, b, Result, null );
    }
    abstract public void	SO_abs( IqShaderData x, IqShaderData Result, IqShader pShader );
    final public void	SO_abs( IqShaderData x, IqShaderData Result ){
        SO_abs( x, Result, null );
    }
    abstract public void	SO_sign( IqShaderData x, IqShaderData Result, IqShader pShader );
    final public void	SO_sign( IqShaderData x, IqShaderData Result ){
        SO_sign( x, Result, null );
    }
    abstract public void	SO_min( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_min( IqShaderData a, IqShaderData b, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_min( a, b, Result, null, cParams, apParams );
    }
    abstract public void	SO_max( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_max( IqShaderData a, IqShaderData b, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_max( a, b, Result, null, cParams, apParams );
    }
    abstract public void	SO_pmin( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_pmin( IqShaderData a, IqShaderData b, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_pmin( a, b, Result, null, cParams, apParams );
    }
    abstract public void	SO_pmax( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_pmax( IqShaderData a, IqShaderData b, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_pmax( a, b, Result, null, cParams, apParams );
    }
    abstract public void	SO_cmin( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_cmin( IqShaderData a, IqShaderData b, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_cmin( a, b, Result, null, cParams, apParams );
    }
    abstract public void	SO_cmax( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_cmax( IqShaderData a, IqShaderData b, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_cmax( a, b, Result, null, cParams, apParams );
    }
    abstract public void	SO_clamp( IqShaderData a, IqShaderData _min, IqShaderData _max, IqShaderData Result, IqShader pShader );
    final public void	SO_clamp( IqShaderData a, IqShaderData _min, IqShaderData _max, IqShaderData Result ){
        SO_clamp( a, _min, _max, Result, null );
    }
    abstract public void	SO_pclamp( IqShaderData a, IqShaderData _min, IqShaderData _max, IqShaderData Result, IqShader pShader );
    final public void	SO_pclamp( IqShaderData a, IqShaderData _min, IqShaderData _max, IqShaderData Result ){
        SO_pclamp( a, _min, _max, Result, null );
    }
    abstract public void	SO_cclamp( IqShaderData a, IqShaderData _min, IqShaderData _max, IqShaderData Result, IqShader pShader );
    final public void	SO_cclamp( IqShaderData a, IqShaderData _min, IqShaderData _max, IqShaderData Result ){
        SO_cclamp( a, _min, _max, Result, null );
    }
    abstract public void	SO_floor( IqShaderData x, IqShaderData Result, IqShader pShader );
    final public void	SO_floor( IqShaderData x, IqShaderData Result ){
        SO_floor( x, Result, null );
    }
    abstract public void	SO_ceil( IqShaderData x, IqShaderData Result, IqShader pShader );
    final public void	SO_ceil( IqShaderData x, IqShaderData Result ){
        SO_ceil( x, Result, null );
    }
    abstract public void	SO_round( IqShaderData x, IqShaderData Result, IqShader pShader );
    final public void	SO_round( IqShaderData x, IqShaderData Result ){
        SO_round( x, Result, null );
    }
    abstract public void	SO_step( IqShaderData _min, IqShaderData value, IqShaderData Result, IqShader pShader );
    final public void	SO_step( IqShaderData _min, IqShaderData value, IqShaderData Result ){
        SO_step( _min, value, Result, null );
    }
    abstract public void	SO_smoothstep( IqShaderData _min, IqShaderData _max, IqShaderData value, IqShaderData Result, IqShader pShader );
    final public void	SO_smoothstep( IqShaderData _min, IqShaderData _max, IqShaderData value, IqShaderData Result ){
        SO_smoothstep( _min, _max, value, Result, null );
    }
    abstract public void	SO_fspline( IqShaderData value, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_fspline( IqShaderData value, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_fspline( value, Result, null, cParams, apParams );
    }
    abstract public void	SO_cspline( IqShaderData value, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_cspline( IqShaderData value, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_cspline( value, Result, null, cParams, apParams );
    }
    abstract public void	SO_pspline( IqShaderData value, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_pspline( IqShaderData value, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_pspline( value, Result, null, cParams, apParams );
    }
    abstract public void	SO_sfspline( IqShaderData basis, IqShaderData value, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_sfspline( IqShaderData basis, IqShaderData value, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_sfspline( basis, value, Result, null, cParams, apParams );
    }
    abstract public void	SO_scspline( IqShaderData basis, IqShaderData value, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_scspline( IqShaderData basis, IqShaderData value, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_scspline( basis, value, Result, null, cParams, apParams );
    }
    abstract public void	SO_spspline( IqShaderData basis, IqShaderData value, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_spspline( IqShaderData basis, IqShaderData value, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_spspline( basis, value, Result, null, cParams, apParams );
    }
    abstract public void	SO_fDu( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_fDu( IqShaderData p, IqShaderData Result ){
        SO_fDu( p, Result, null );
    } 
    abstract public void	SO_fDv( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_fDv( IqShaderData p, IqShaderData Result ){
        SO_fDv( p, Result, null );
    }
    abstract public void	SO_fDeriv( IqShaderData p, IqShaderData den, IqShaderData Result, IqShader pShader );
    final public void	SO_fDeriv( IqShaderData p, IqShaderData den, IqShaderData Result ){
        SO_fDeriv( p, den, Result, null );
    }
    abstract public void	SO_cDu( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_cDu( IqShaderData p, IqShaderData Result ){
        SO_cDu( p, Result, null );
    }
    abstract public void	SO_cDv( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_cDv( IqShaderData p, IqShaderData Result ){
        SO_cDv( p, Result, null );
    }
    abstract public void	SO_cDeriv( IqShaderData p, IqShaderData den, IqShaderData Result, IqShader pShader );
    final public void	SO_cDeriv( IqShaderData p, IqShaderData den, IqShaderData Result ){
        SO_cDeriv( p, den, Result, null );
    }
    abstract public void	SO_pDu( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_pDu( IqShaderData p, IqShaderData Result ){
        SO_pDu( p, Result, null );
    }
    abstract public void	SO_pDv( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_pDv( IqShaderData p, IqShaderData Result ){
        SO_pDv(  p, Result, null );
    }
    abstract public void	SO_pDeriv( IqShaderData p, IqShaderData den, IqShaderData Result, IqShader pShader );
    final public void	SO_pDeriv( IqShaderData p, IqShaderData den, IqShaderData Result ){
        SO_pDeriv( p, den, Result, null );
    }
    abstract public void	SO_frandom( IqShaderData Result, IqShader pShader );
    final public void	SO_frandom( IqShaderData Result ){
        SO_frandom( Result, null );
    }
    abstract public void	SO_crandom( IqShaderData Result, IqShader pShader );
    final public void	SO_crandom( IqShaderData Result ){
        SO_crandom( Result, null );
    }
    abstract public void	SO_prandom( IqShaderData Result, IqShader pShader );
    final public void	SO_prandom( IqShaderData Result ){
        SO_prandom( Result, null );
    }
    abstract public void	SO_fnoise1( IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_fnoise1( IqShaderData v, IqShaderData Result ){
        SO_fnoise1( v, Result, null );
    }
    abstract public void	SO_fnoise2( IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_fnoise2( IqShaderData u, IqShaderData v, IqShaderData Result ){
        SO_fnoise2( u, v, Result, null );
    }
    abstract public void	SO_fnoise3( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_fnoise3( IqShaderData p, IqShaderData Result ){
        SO_fnoise3( p, Result, null );
    }
    abstract public void	SO_fnoise4( IqShaderData p, IqShaderData t, IqShaderData Result, IqShader pShader );
    final public void	SO_fnoise4( IqShaderData p, IqShaderData t, IqShaderData Result ){
        SO_fnoise4( p, t, Result, null );
    }
    abstract public void	SO_cnoise1( IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_cnoise1( IqShaderData v, IqShaderData Result ){
        SO_cnoise1( v, Result, null );
    }
    abstract public void	SO_cnoise2( IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_cnoise2( IqShaderData u, IqShaderData v, IqShaderData Result ){
        SO_cnoise2( u, v, Result, null );
    }
    abstract public void	SO_cnoise3( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_cnoise3( IqShaderData p, IqShaderData Result ){
        SO_cnoise3( p, Result, null );
    }
    abstract public void	SO_cnoise4( IqShaderData p, IqShaderData t, IqShaderData Result, IqShader pShader );
    final public void	SO_cnoise4( IqShaderData p, IqShaderData t, IqShaderData Result ){
        SO_cnoise4( p, t, Result, null );
    }
    abstract public void	SO_pnoise1( IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_pnoise1( IqShaderData v, IqShaderData Result ){
        SO_pnoise1( v, Result, null );
    }
    abstract public void	SO_pnoise2( IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_pnoise2( IqShaderData u, IqShaderData v, IqShaderData Result ){
        SO_pnoise2( u, v, Result, null );
    }
    abstract public void	SO_pnoise3( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_pnoise3( IqShaderData p, IqShaderData Result ){
        SO_pnoise3( p, Result, null );
    }
    abstract public void	SO_pnoise4( IqShaderData p, IqShaderData t, IqShaderData Result, IqShader pShader );
    final public void	SO_pnoise4( IqShaderData p, IqShaderData t, IqShaderData Result ){
        SO_pnoise4( p, t, Result, null );
    }
    abstract public void	SO_setcomp( IqShaderData p, IqShaderData i, IqShaderData v, IqShader pShader );
    final public void	SO_setcomp( IqShaderData p, IqShaderData i, IqShaderData v ){
        SO_setcomp( p, i, v, null );
    }
    abstract public void	SO_setxcomp( IqShaderData p, IqShaderData v, IqShader pShader );
    final public void	SO_setxcomp( IqShaderData p, IqShaderData v ){
        SO_setxcomp( p, v, null );
    }
    abstract public void	SO_setycomp( IqShaderData p, IqShaderData v, IqShader pShader );
    final public void	SO_setycomp( IqShaderData p, IqShaderData v ){
        SO_setycomp( p, v, null );
    }
    abstract public void	SO_setzcomp( IqShaderData p, IqShaderData v, IqShader pShader );
    final public void	SO_setzcomp( IqShaderData p, IqShaderData v ){
        SO_setzcomp( p, v, null );
    }
    abstract public void	SO_length( IqShaderData V, IqShaderData Result, IqShader pShader );
    final public void	SO_length( IqShaderData V, IqShaderData Result ){
        SO_length( V, Result, null );
    }
    abstract public void	SO_distance( IqShaderData P1, IqShaderData P2, IqShaderData Result, IqShader pShader );
    final public void	SO_distance( IqShaderData P1, IqShaderData P2, IqShaderData Result ){
        SO_distance( P1, P2, Result, null );
    }
    abstract public void	SO_area( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_area( IqShaderData p, IqShaderData Result ){
        SO_area( p, Result, null );
    }
    abstract public void	SO_normalize( IqShaderData V, IqShaderData Result, IqShader pShader );
    final public void	SO_normalize( IqShaderData V, IqShaderData Result ){
        SO_normalize( V, Result, null );
    }
    abstract public void	SO_faceforward( IqShaderData N, IqShaderData I, IqShaderData Result, IqShader pShader );
    final public void	SO_faceforward( IqShaderData N, IqShaderData I, IqShaderData Result ){
        SO_faceforward( N, I, Result, null );
    }
    abstract public void	SO_faceforward2( IqShaderData N, IqShaderData I, IqShaderData Nref, IqShaderData Result, IqShader pShader );
    final public void	SO_faceforward2( IqShaderData N, IqShaderData I, IqShaderData Nref, IqShaderData Result ){
        SO_faceforward2( N, I, Nref, Result, null );
    }
    abstract public void	SO_reflect( IqShaderData I, IqShaderData N, IqShaderData Result, IqShader pShader );
    final public void	SO_reflect( IqShaderData I, IqShaderData N, IqShaderData Result ){
        SO_reflect( I, N, Result, null );
    }
    abstract public void	SO_refract( IqShaderData I, IqShaderData N, IqShaderData eta, IqShaderData Result, IqShader pShader );
    final public void	SO_refract( IqShaderData I, IqShaderData N, IqShaderData eta, IqShaderData Result ){
        SO_refract( I, N, eta, Result, null );
    }
    abstract public void	SO_fresnel( IqShaderData I, IqShaderData N, IqShaderData eta, IqShaderData Kr, IqShaderData Kt, IqShader pShader );
    final public void	SO_fresnel( IqShaderData I, IqShaderData N, IqShaderData eta, IqShaderData Kr, IqShaderData Kt ){
        SO_fresnel( I, N, eta, Kr, Kt, null );
    }
    abstract public void	SO_fresnel( IqShaderData I, IqShaderData N, IqShaderData eta, IqShaderData Kr, IqShaderData Kt, IqShaderData R, IqShaderData T, IqShader pShader );
    final public void	SO_fresnel( IqShaderData I, IqShaderData N, IqShaderData eta, IqShaderData Kr, IqShaderData Kt, IqShaderData R, IqShaderData T ){
        SO_fresnel( I, N, eta, Kr, Kt, R, T, null );
    }
    abstract public void	SO_transform( IqShaderData fromspace, IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_transform( IqShaderData fromspace, IqShaderData tospace, IqShaderData p, IqShaderData Result ){
        SO_transform( fromspace, tospace, p, Result, null );
    }
    abstract public void	SO_transform( IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_transform( IqShaderData tospace, IqShaderData p, IqShaderData Result ){
        SO_transform( tospace, p, Result, (IqShaderData)null );
    }
    abstract public void	SO_transformm( IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_transformm( IqShaderData tospace, IqShaderData p, IqShaderData Result ){
        SO_transformm( tospace, p, Result, null );
    }
    abstract public void	SO_vtransform( IqShaderData fromspace, IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_vtransform( IqShaderData fromspace, IqShaderData tospace, IqShaderData p, IqShaderData Result ){
        SO_vtransform( fromspace, tospace, p, Result, null );
    }
    abstract public void	SO_vtransform( IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_vtransform( IqShaderData tospace, IqShaderData p, IqShaderData Result ){
        SO_vtransform( tospace, p, Result, (IqShaderData)null );
    }
    abstract public void	SO_vtransformm( IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_vtransformm( IqShaderData tospace, IqShaderData p, IqShaderData Result ){
        SO_vtransformm( tospace, p, Result, null );
    }
    abstract public void	SO_ntransform( IqShaderData fromspace, IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_ntransform( IqShaderData fromspace, IqShaderData tospace, IqShaderData p, IqShaderData Result ){
        SO_ntransform( fromspace, tospace, p, Result, null );
    }
    abstract public void	SO_ntransform( IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_ntransform( IqShaderData tospace, IqShaderData p, IqShaderData Result ){
        SO_ntransform( tospace, p, Result, (IqShaderData)null );
    }
    abstract public void	SO_ntransformm( IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_ntransformm( IqShaderData tospace, IqShaderData p, IqShaderData Result ){
        SO_ntransformm( tospace, p, Result, null );
    }
    abstract public void	SO_depth( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_depth( IqShaderData p, IqShaderData Result ){
        SO_depth( p, Result, null );
    }
    abstract public void	SO_calculatenormal( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_calculatenormal( IqShaderData p, IqShaderData Result ){
        SO_calculatenormal( p, Result, null );
    }
    abstract public void	SO_cmix( IqShaderData color0, IqShaderData color1, IqShaderData value, IqShaderData Result, IqShader pShader );
    final public void	SO_cmix( IqShaderData color0, IqShaderData color1, IqShaderData value, IqShaderData Result ){
        SO_cmix( color0, color1, value, Result, null );
    }
    abstract public void	SO_fmix( IqShaderData f0, IqShaderData f1, IqShaderData value, IqShaderData Result, IqShader pShader );
    final public void	SO_fmix( IqShaderData f0, IqShaderData f1, IqShaderData value, IqShaderData Result ){
        SO_fmix( f0, f1, value, Result, null );
    }
    abstract public void	SO_pmix( IqShaderData p0, IqShaderData p1, IqShaderData value, IqShaderData Result, IqShader pShader );
    final public void	SO_pmix( IqShaderData p0, IqShaderData p1, IqShaderData value, IqShaderData Result ){
        SO_pmix( p0, p1, value, Result, null );
    }
    abstract public void	SO_vmix( IqShaderData v0, IqShaderData v1, IqShaderData value, IqShaderData Result, IqShader pShader );
    final public void	SO_vmix( IqShaderData v0, IqShaderData v1, IqShaderData value, IqShaderData Result ){
        SO_vmix( v0, v1, value, Result, null );
    }
    abstract public void	SO_nmix( IqShaderData n0, IqShaderData n1, IqShaderData value, IqShaderData Result, IqShader pShader );
    final public void	SO_nmix( IqShaderData n0, IqShaderData n1, IqShaderData value, IqShaderData Result ){
        SO_nmix( n0, n1, value, Result, null );
    }
    abstract public void	SO_ambient( IqShaderData Result, IqShader pShader );
    final public void	SO_ambient( IqShaderData Result ){
        SO_ambient( Result, null );
    }
    abstract public void	SO_diffuse( IqShaderData N, IqShaderData Result, IqShader pShader );
    final public void	SO_diffuse( IqShaderData N, IqShaderData Result ){
        SO_diffuse( N, Result, null );
    }
    abstract public void	SO_specular( IqShaderData N, IqShaderData V, IqShaderData roughness, IqShaderData Result, IqShader pShader );
    final public void	SO_specular( IqShaderData N, IqShaderData V, IqShaderData roughness, IqShaderData Result ){
        SO_specular( N, V, roughness, Result, null );
    }
    abstract public void	SO_phong( IqShaderData N, IqShaderData V, IqShaderData size, IqShaderData Result, IqShader pShader );
    final public void	SO_phong( IqShaderData N, IqShaderData V, IqShaderData size, IqShaderData Result ){
        SO_phong( N, V, size, Result, null );
    }
    abstract public void	SO_trace( IqShaderData P, IqShaderData R, IqShaderData Result, IqShader pShader );
    final public void	SO_trace( IqShaderData P, IqShaderData R, IqShaderData Result ){
        SO_trace( P, R, Result, null );
    }
    abstract public void	SO_ftexture1( IqShaderData name, IqShaderData channel, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_ftexture1( IqShaderData name, IqShaderData channel, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_ftexture1( name, channel, Result, null, cParams, apParams );
    }
    abstract public void	SO_ftexture2( IqShaderData name, IqShaderData channel, IqShaderData s, IqShaderData t, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_ftexture2( IqShaderData name, IqShaderData channel, IqShaderData s, IqShaderData t, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_ftexture2( name, channel, s, t, Result, null, cParams, apParams );
    }
    abstract public void	SO_ftexture3( IqShaderData name, IqShaderData channel, IqShaderData s1, IqShaderData t1, IqShaderData s2, IqShaderData t2, IqShaderData s3, IqShaderData t3, IqShaderData s4, IqShaderData t4, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_ftexture3( IqShaderData name, IqShaderData channel, IqShaderData s1, IqShaderData t1, IqShaderData s2, IqShaderData t2, IqShaderData s3, IqShaderData t3, IqShaderData s4, IqShaderData t4, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_ftexture3( name, channel, s1, t1, s2, t2, s3, t3, s4, t4, Result, null, cParams, apParams );
    }
    abstract public void	SO_ctexture1( IqShaderData name, IqShaderData channel, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_ctexture1( IqShaderData name, IqShaderData channel, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_ctexture1( name, channel, Result, null, cParams, apParams );
    }
    abstract public void	SO_ctexture2( IqShaderData name, IqShaderData channel, IqShaderData s, IqShaderData t, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_ctexture2( IqShaderData name, IqShaderData channel, IqShaderData s, IqShaderData t, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_ctexture2( name, channel, s, t, Result, null, cParams, apParams );
    }
    abstract public void	SO_ctexture3( IqShaderData name, IqShaderData channel, IqShaderData s1, IqShaderData t1, IqShaderData s2, IqShaderData t2, IqShaderData s3, IqShaderData t3, IqShaderData s4, IqShaderData t4, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_ctexture3( IqShaderData name, IqShaderData channel, IqShaderData s1, IqShaderData t1, IqShaderData s2, IqShaderData t2, IqShaderData s3, IqShaderData t3, IqShaderData s4, IqShaderData t4, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_ctexture3( name, channel, s1, t1, s2, t2, s3, t3, s4, t4, Result, null, cParams, apParams );
    }
    abstract public void	SO_fenvironment2( IqShaderData name, IqShaderData channel, IqShaderData R, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_fenvironment2( IqShaderData name, IqShaderData channel, IqShaderData R, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_fenvironment2( name, channel, R, Result, null, cParams, apParams );
    }
    abstract public void	SO_fenvironment3( IqShaderData name, IqShaderData channel, IqShaderData R1, IqShaderData R2, IqShaderData R3, IqShaderData R4, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_fenvironment3( IqShaderData name, IqShaderData channel, IqShaderData R1, IqShaderData R2, IqShaderData R3, IqShaderData R4, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_fenvironment3( name, channel, R1, R2, R3, R4, Result, null, cParams, apParams );
    }
    abstract public void	SO_cenvironment2( IqShaderData name, IqShaderData channel, IqShaderData R, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_cenvironment2( IqShaderData name, IqShaderData channel, IqShaderData R, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_cenvironment2( name, channel, R, Result, null, cParams, apParams );
    }
    abstract public void	SO_cenvironment3( IqShaderData name, IqShaderData channel, IqShaderData R1, IqShaderData R2, IqShaderData R3, IqShaderData R4, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_cenvironment3( IqShaderData name, IqShaderData channel, IqShaderData R1, IqShaderData R2, IqShaderData R3, IqShaderData R4, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_cenvironment3( name, channel, R1, R2, R3, R4, Result, null, cParams, apParams );
    }
    abstract public void	SO_bump1( IqShaderData name, IqShaderData channel, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_bump1( IqShaderData name, IqShaderData channel, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_bump1( name, channel, Result, null, cParams, apParams );
    }
    abstract public void	SO_bump2( IqShaderData name, IqShaderData channel, IqShaderData s, IqShaderData t, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_bump2( IqShaderData name, IqShaderData channel, IqShaderData s, IqShaderData t, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_bump2( name, channel, s, t, Result, null, cParams, apParams );
    }
    abstract public void	SO_bump3( IqShaderData name, IqShaderData channel, IqShaderData s1, IqShaderData t1, IqShaderData s2, IqShaderData t2, IqShaderData s3, IqShaderData t3, IqShaderData s4, IqShaderData t4, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_bump3( IqShaderData name, IqShaderData channel, IqShaderData s1, IqShaderData t1, IqShaderData s2, IqShaderData t2, IqShaderData s3, IqShaderData t3, IqShaderData s4, IqShaderData t4, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_bump3( name, channel, s1, t1, s2, t2, s3, t3, s4, t4, Result, null, cParams, apParams );
    }
    abstract public void	SO_shadow( IqShaderData name, IqShaderData channel, IqShaderData P, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_shadow( IqShaderData name, IqShaderData channel, IqShaderData P, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_shadow( name, channel, P, Result, null, cParams, apParams );
    }
    abstract public void	SO_shadow1( IqShaderData name, IqShaderData channel, IqShaderData P1, IqShaderData P2, IqShaderData P3, IqShaderData P4, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_shadow1( IqShaderData name, IqShaderData channel, IqShaderData P1, IqShaderData P2, IqShaderData P3, IqShaderData P4, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_shadow1( name, channel, P1, P2, P3, P4, Result, null, cParams, apParams );
    }
    abstract public void	SO_illuminance( IqShaderData P, IqShaderData nsamples, IqShader pShader );
    final public void	SO_illuminance( IqShaderData P, IqShaderData nsamples ){
        SO_illuminance( P, nsamples, null );
    }
    abstract public void	SO_illuminance( IqShaderData P, IqShaderData Axis, IqShaderData Angle, IqShaderData nsamples, IqShader pShader );
    final public void	SO_illuminance( IqShaderData P, IqShaderData Axis, IqShaderData Angle, IqShaderData nsamples ){
        SO_illuminance( P, Axis, Angle, nsamples, null );
    }
    abstract public void	SO_illuminate( IqShaderData P, IqShaderData Axis, IqShaderData Angle, IqShader pShader );
    final public void	SO_illuminate( IqShaderData P, IqShaderData Axis, IqShaderData Angle ){
        SO_illuminate( P, Axis, Angle, null );
    }
    abstract public void	SO_illuminate( IqShaderData P, IqShader pShader );
    final public void	SO_illuminate( IqShaderData P ){
        SO_illuminate( P, null );
    }
    abstract public void	SO_solar( IqShaderData Axis, IqShaderData Angle, IqShader pShader );
    final public void	SO_solar( IqShaderData Axis, IqShaderData Angle ){
        SO_solar( Axis, Angle, null );
    }
    abstract public void	SO_solar( IqShader pShader );
    final public void	SO_solar(){
        SO_solar( null );
    }
    abstract public void	SO_printf( IqShaderData str, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_printf( IqShaderData str, int cParams, IqShaderData[] apParams ){
        SO_printf( str, null, cParams, apParams );
    }
    abstract public void	SO_format( IqShaderData str, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_format( IqShaderData str, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_format( str, Result, null, cParams, apParams );
    }
    abstract public void	SO_concat( IqShaderData stra, IqShaderData strb, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_concat( IqShaderData stra, IqShaderData strb, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_concat( stra, strb, Result, null, cParams, apParams );
    }
    abstract public void	SO_atmosphere( IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader );
    final public void	SO_atmosphere( IqShaderData name, IqShaderData pV, IqShaderData Result ){
        SO_atmosphere( name, pV, Result, null );
    }
    abstract public void	SO_displacement( IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader );
    final public void	SO_displacement( IqShaderData name, IqShaderData pV, IqShaderData Result ){
        SO_displacement( name, pV, Result, null );
    }
    abstract public void	SO_lightsource( IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader );
    final public void	SO_lightsource( IqShaderData name, IqShaderData pV, IqShaderData Result ){
        SO_lightsource( name, pV, Result, null );
    }
    abstract public void	SO_surface( IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader );
    final public void	SO_surface( IqShaderData name, IqShaderData pV, IqShaderData Result ){
        SO_surface( name, pV, Result, null );
    }
    abstract public void	SO_attribute( IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader );
    final public void	SO_attribute( IqShaderData name, IqShaderData pV, IqShaderData Result ){
        SO_attribute( name, pV, Result, null );
    }
    abstract public void	SO_option( IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader );
    final public void	SO_option( IqShaderData name, IqShaderData pV, IqShaderData Result ){
        SO_option( name, pV, Result, null );
    }
    abstract public void	SO_rendererinfo( IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader );
    final public void	SO_rendererinfo( IqShaderData name, IqShaderData pV, IqShaderData Result ){
        SO_rendererinfo( name, pV, Result, null );
    }
    abstract public void	SO_incident( IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader );
    final public void	SO_incident( IqShaderData name, IqShaderData pV, IqShaderData Result ){
        SO_incident( name, pV, Result, null );
    }
    abstract public void	SO_opposite( IqShaderData name, IqShaderData pV, IqShaderData Result, IqShader pShader );
    final public void	SO_opposite( IqShaderData name, IqShaderData pV, IqShaderData Result ){
        SO_opposite( name, pV, Result, null );
    }
    abstract public void	SO_fcellnoise1( IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_fcellnoise1( IqShaderData v, IqShaderData Result ){
        SO_fcellnoise1( v, Result, null );
    }
    abstract public void	SO_fcellnoise2( IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_fcellnoise2( IqShaderData u, IqShaderData v, IqShaderData Result ){
        SO_fcellnoise2( u, v, Result, null );
    }
    abstract public void	SO_fcellnoise3( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_fcellnoise3( IqShaderData p, IqShaderData Result ){
        SO_fcellnoise3( p, Result, null );
    }
    abstract public void	SO_fcellnoise4( IqShaderData p, IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_fcellnoise4( IqShaderData p, IqShaderData v, IqShaderData Result ){
        SO_fcellnoise4( p, v, Result, null );
    }
    abstract public void	SO_ccellnoise1( IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_ccellnoise1( IqShaderData v, IqShaderData Result ){
        SO_ccellnoise1( v, Result, null );
    }
    abstract public void	SO_ccellnoise2( IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_ccellnoise2( IqShaderData u, IqShaderData v, IqShaderData Result ){
        SO_ccellnoise2( u, v, Result, null );
    }
    abstract public void	SO_ccellnoise3( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_ccellnoise3( IqShaderData p, IqShaderData Result ){
        SO_ccellnoise3( p, Result, null );
    }
    abstract public void	SO_ccellnoise4( IqShaderData p, IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_ccellnoise4( IqShaderData p, IqShaderData v, IqShaderData Result ){
        SO_ccellnoise4( p, v, Result, null );
    }
    abstract public void	SO_pcellnoise1( IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_pcellnoise1( IqShaderData v, IqShaderData Result ){
        SO_pcellnoise1( v, Result, null );
    }
    abstract public void	SO_pcellnoise2( IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_pcellnoise2( IqShaderData u, IqShaderData v, IqShaderData Result ){
        SO_pcellnoise2( u, v, Result, null );
    }
    abstract public void	SO_pcellnoise3( IqShaderData p, IqShaderData Result, IqShader pShader );
    final public void	SO_pcellnoise3( IqShaderData p, IqShaderData Result ){
        SO_pcellnoise3( p, Result, null );
    }
    abstract public void	SO_pcellnoise4( IqShaderData p, IqShaderData v, IqShaderData Result, IqShader pShader );
    final public void	SO_pcellnoise4( IqShaderData p, IqShaderData v, IqShaderData Result ){
        SO_pcellnoise4( p, v, Result, null );
    }
    abstract public void	SO_fpnoise1( IqShaderData v, IqShaderData period, IqShaderData Result, IqShader pShader );
    final public void	SO_fpnoise1( IqShaderData v, IqShaderData period, IqShaderData Result ){
        SO_fpnoise1( v, period, Result, null );
    }
    abstract public void	SO_fpnoise2( IqShaderData u, IqShaderData v, IqShaderData uperiod, IqShaderData vperiod, IqShaderData Result, IqShader pShader );
    final public void	SO_fpnoise2( IqShaderData u, IqShaderData v, IqShaderData uperiod, IqShaderData vperiod, IqShaderData Result ){
        SO_fpnoise2( u, v, uperiod, vperiod, Result, null );
    }
    abstract public void	SO_fpnoise3( IqShaderData p, IqShaderData pperiod, IqShaderData Result, IqShader pShader );
    final public void	SO_fpnoise3( IqShaderData p, IqShaderData pperiod, IqShaderData Result ){
        SO_fpnoise3( p, pperiod, Result, null );
    }
    abstract public void	SO_fpnoise4( IqShaderData p, IqShaderData t, IqShaderData pperiod, IqShaderData tperiod, IqShaderData Result, IqShader pShader );
    final public void	SO_fpnoise4( IqShaderData p, IqShaderData t, IqShaderData pperiod, IqShaderData tperiod, IqShaderData Result ){
        SO_fpnoise4( p, t, pperiod, tperiod, Result, null );
    }
    abstract public void	SO_cpnoise1( IqShaderData v, IqShaderData period, IqShaderData Result, IqShader pShader );
    final public void	SO_cpnoise1( IqShaderData v, IqShaderData period, IqShaderData Result ){
        SO_cpnoise1( v, period, Result, null );
    }
    abstract public void	SO_cpnoise2( IqShaderData u, IqShaderData v, IqShaderData uperiod, IqShaderData vperiod, IqShaderData Result, IqShader pShader );
    final public void	SO_cpnoise2( IqShaderData u, IqShaderData v, IqShaderData uperiod, IqShaderData vperiod, IqShaderData Result ){
        SO_cpnoise2( u, v, uperiod, vperiod, Result, null );
    }
    abstract public void	SO_cpnoise3( IqShaderData p, IqShaderData pperiod, IqShaderData Result, IqShader pShader );
    final public void	SO_cpnoise3( IqShaderData p, IqShaderData pperiod, IqShaderData Result ){
        SO_cpnoise3( p, pperiod, Result, null );
    }
    abstract public void	SO_cpnoise4( IqShaderData p, IqShaderData t, IqShaderData pperiod, IqShaderData tperiod, IqShaderData Result, IqShader pShader );
    final public void	SO_cpnoise4( IqShaderData p, IqShaderData t, IqShaderData pperiod, IqShaderData tperiod, IqShaderData Result ){
        SO_cpnoise4( p, t, pperiod, tperiod, Result, null );
    }
    abstract public void	SO_ppnoise1( IqShaderData v, IqShaderData period, IqShaderData Result, IqShader pShader );
    final public void	SO_ppnoise1( IqShaderData v, IqShaderData period, IqShaderData Result ){
        SO_ppnoise1( v, period, Result, null );
    }
    abstract public void	SO_ppnoise2( IqShaderData u, IqShaderData v, IqShaderData uperiod, IqShaderData vperiod, IqShaderData Result, IqShader pShader );
    final public void	SO_ppnoise2( IqShaderData u, IqShaderData v, IqShaderData uperiod, IqShaderData vperiod, IqShaderData Result ){
        SO_ppnoise2( u, v, uperiod, vperiod, Result, null );
    }
    abstract public void	SO_ppnoise3( IqShaderData p, IqShaderData pperiod, IqShaderData Result, IqShader pShader );
    final public void	SO_ppnoise3( IqShaderData p, IqShaderData pperiod, IqShaderData Result ){
        SO_ppnoise3( p, pperiod, Result, null );
    }
    abstract public void	SO_ppnoise4( IqShaderData p, IqShaderData t, IqShaderData pperiod, IqShaderData tperiod, IqShaderData Result, IqShader pShader );
    final public void	SO_ppnoise4( IqShaderData p, IqShaderData t, IqShaderData pperiod, IqShaderData tperiod, IqShaderData Result ){
        SO_ppnoise4( p, t, pperiod, tperiod, Result, null );
    }
    abstract public void	SO_ctransform( IqShaderData fromspace, IqShaderData tospace, IqShaderData c, IqShaderData Result, IqShader pShader );
    final public void	SO_ctransform( IqShaderData fromspace, IqShaderData tospace, IqShaderData c, IqShaderData Result ){
        SO_ctransform( fromspace, tospace, c, Result, null );
    }
    abstract public void	SO_ctransform( IqShaderData tospace, IqShaderData c, IqShaderData Result, IqShader pShader );
    final public void	SO_ctransform( IqShaderData tospace, IqShaderData c, IqShaderData Result ){
        SO_ctransform( tospace, c, Result, (IqShaderData)null );
    }
    abstract public void	SO_ptlined( IqShaderData P0, IqShaderData P1, IqShaderData Q, IqShaderData Result, IqShader pShader );
    final public void	SO_ptlined( IqShaderData P0, IqShaderData P1, IqShaderData Q, IqShaderData Result ){
        SO_ptlined( P0, P1, Q, Result, null );
    }
    abstract public void	SO_inversesqrt( IqShaderData x, IqShaderData Result, IqShader pShader );
    final public void	SO_inversesqrt( IqShaderData x, IqShaderData Result ){
        SO_inversesqrt( x, Result, null );
    }
    abstract public void	SO_match( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader );
    final public void	SO_match( IqShaderData a, IqShaderData b, IqShaderData Result ){
        SO_match( a, b, Result, null );
    }
    abstract public void	SO_rotate( IqShaderData Q, IqShaderData angle, IqShaderData P0, IqShaderData P1, IqShaderData Result, IqShader pShader );
    final public void	SO_rotate( IqShaderData Q, IqShaderData angle, IqShaderData P0, IqShaderData P1, IqShaderData Result ){
        SO_rotate( Q, angle, P0, P1, Result, null );
    }
    abstract public void	SO_filterstep( IqShaderData edge, IqShaderData s1, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_filterstep( IqShaderData edge, IqShaderData s1, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_filterstep( edge, s1, Result, null, cParams, apParams );
    }
    abstract public void	SO_filterstep2( IqShaderData edge, IqShaderData s1, IqShaderData s2, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public void	SO_filterstep2( IqShaderData edge, IqShaderData s1, IqShaderData s2, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_filterstep2( edge, s1, s2, Result, null, cParams, apParams );
    }
    abstract public void	SO_specularbrdf( IqShaderData L, IqShaderData N, IqShaderData V, IqShaderData rough, IqShaderData Result, IqShader pShader );
    final public void	SO_specularbrdf( IqShaderData L, IqShaderData N, IqShaderData V, IqShaderData rough, IqShaderData Result ){
        SO_specularbrdf( L, N, V, rough, Result, null );
    }
    abstract public void	SO_setmcomp( IqShaderData M, IqShaderData row, IqShaderData column, IqShaderData val, IqShader pShader );
    final public void	SO_setmcomp( IqShaderData M, IqShaderData row, IqShaderData column, IqShaderData val ){
        SO_setmcomp( M, row, column, val, null );
    }
    abstract public void	SO_determinant( IqShaderData M, IqShaderData Result, IqShader pShader );
    final public void	SO_determinant( IqShaderData M, IqShaderData Result ){
        SO_determinant( M, Result, null );
    }
    abstract public void	SO_mtranslate( IqShaderData M, IqShaderData V, IqShaderData Result, IqShader pShader );
    final public void	SO_mtranslate( IqShaderData M, IqShaderData V, IqShaderData Result ){
        SO_mtranslate( M, V, Result, null );
    }
    abstract public void	SO_mrotate( IqShaderData M, IqShaderData angle, IqShaderData axis, IqShaderData Result, IqShader pShader );
    final public void	SO_mrotate( IqShaderData M, IqShaderData angle, IqShaderData axis, IqShaderData Result ){
        SO_mrotate( M, angle, axis, Result, null );
    }
    abstract public void	SO_mscale( IqShaderData M, IqShaderData s, IqShaderData Result, IqShader pShader );
    final public void	SO_mscale( IqShaderData M, IqShaderData s, IqShaderData Result ){
        SO_mscale( M, s, Result, null );
    }
    abstract public void	SO_fsplinea( IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader );
    final public void	SO_fsplinea( IqShaderData value, IqShaderData a, IqShaderData Result ){
        SO_fsplinea( value, a, Result, null );
    }
    abstract public void	SO_csplinea( IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader );
    final public void	SO_csplinea( IqShaderData value, IqShaderData a, IqShaderData Result ){
        SO_csplinea( value, a, Result, null );
    }
    abstract public void	SO_psplinea( IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader );
    final public void	SO_psplinea( IqShaderData value, IqShaderData a, IqShaderData Result ){
        SO_psplinea( value, a, Result, null );
    }
    abstract public void	SO_sfsplinea( IqShaderData basis, IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader );
    final public void	SO_sfsplinea( IqShaderData basis, IqShaderData value, IqShaderData a, IqShaderData Result ){
        SO_sfsplinea( basis, value, a, Result, null );
    }
    abstract public void	SO_scsplinea( IqShaderData basis, IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader );
    final public void	SO_scsplinea( IqShaderData basis, IqShaderData value, IqShaderData a, IqShaderData Result ){
        SO_scsplinea( basis, value, a, Result, null );
    }
    abstract public void	SO_spsplinea( IqShaderData basis, IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader );
    final public void	SO_spsplinea( IqShaderData basis, IqShaderData value, IqShaderData a, IqShaderData Result ){
        SO_spsplinea( basis, value, a, Result, null );
    }
    abstract public void	SO_shadername( IqShaderData Result, IqShader pShader );
    final public void	SO_shadername( IqShaderData Result ){
        SO_shadername( Result, null );
    }
    abstract public void	SO_shadername2( IqShaderData shader, IqShaderData Result, IqShader pShader );
    final public void	SO_shadername2( IqShaderData shader, IqShaderData Result ){
        SO_shadername2( shader, Result, null );
    }
    abstract public void	SO_textureinfo( IqShaderData shader, IqShaderData dataname, IqShaderData pV, IqShaderData Result, IqShader pShader );
    final public void	SO_textureinfo( IqShaderData shader, IqShaderData dataname, IqShaderData pV, IqShaderData Result ){
        SO_textureinfo( shader, dataname, pV, Result, null );
    }
    abstract public void	SO_bake_f( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader, int cParams, IqShaderData[] apParams);
    final public void	SO_bake_f( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, int cParams, IqShaderData[] apParams){
        SO_bake_f( name, s, t, f, null, cParams, apParams);
    }
    final public void SO_bake_f( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader ){
    	SO_bake_f( name, s, t, f, pShader, 0, null );
    }
    abstract public void	SO_bake_3c( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader, int cParams, IqShaderData[] apParams);
    final public void	SO_bake_3c( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, int cParams, IqShaderData[] apParams){
        SO_bake_3c( name, s, t, f, null, cParams, apParams);
    }
    final public void SO_bake_3c( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader ){
    	SO_bake_3c( name, s, t, f, pShader, 0, null );
    }
    abstract public void	SO_bake_3p( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader, int cParams, IqShaderData[] apParams);
    final public void	SO_bake_3p( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, int cParams, IqShaderData[] apParams){
        SO_bake_3p( name, s, t, f, null, cParams, apParams);
    }
    final public void SO_bake_3p( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader ){
    	SO_bake_3p( name, s, t, f, pShader, 0, null );
    }
    abstract public void	SO_bake_3n( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader, int cParams, IqShaderData[] apParams);
    final public void	SO_bake_3n( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, int cParams, IqShaderData[] apParams){
        SO_bake_3n( name, s, t, f, null, cParams, apParams);
    }
    final public void SO_bake_3n( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader ){
    	SO_bake_3n( name, s, t, f, pShader, 0, null );
    }
    abstract public void	SO_bake_3v( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader, int cParams, IqShaderData[] apParams);
    final public void	SO_bake_3v( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, int cParams, IqShaderData[] apParams){
        SO_bake_3v( name, s, t, f, null, cParams, apParams);
    }
    final public void SO_bake_3v( IqShaderData name, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader ){
    	SO_bake_3v( name, s, t, f, pShader, 0, null );
    }
//    abstract public void	SO_external(DSOMethod method, void *initData, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams);
    abstract public void	SO_external( Method method, Object initData, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams);
    final public void	SO_external( Method method, Object initData, IqShaderData Result, int cParams, IqShaderData[] apParams){
        SO_external( method, initData, Result, null, cParams, apParams);
    }
    abstract public	void	SO_occlusion( IqShaderData occlmap, IqShaderData channel, IqShaderData P, IqShaderData N, IqShaderData samples, IqShaderData Result, IqShader pShader, int cParams, IqShaderData[] apParams );
    final public	void	SO_occlusion( IqShaderData occlmap, IqShaderData channel, IqShaderData P, IqShaderData N, IqShaderData samples, IqShaderData Result, int cParams, IqShaderData[] apParams ){
        SO_occlusion( occlmap, channel, P, N, samples, Result, null, cParams, apParams );
    }


/* IWiۑ
    // ShadeOps
    public void	SO_radians( IqShaderDataIqShaderData degrees, IqShaderData Result, IqShader pShader=0 );
    public void	SO_degrees( IqShaderData radians, IqShaderData Result, IqShader pShader=0 );
    public void	SO_sin( IqShaderData a, IqShaderData Result, IqShader pShader=0 );
    public void	SO_asin( IqShaderData a, IqShaderData Result, IqShader pShader=0 );
    public void	SO_cos( IqShaderData a, IqShaderData Result, IqShader pShader=0 );
    public void	SO_acos( IqShaderData a, IqShaderData Result, IqShader pShader=0 );
    public void	SO_tan( IqShaderData a, IqShaderData Result, IqShader pShader=0 );
    public void	SO_atan( IqShaderData yoverx, IqShaderData Result, IqShader pShader=0 );
    public void	SO_atan( IqShaderData y, IqShaderData x, IqShaderData Result, IqShader pShader=0 );
    public void	SO_pow( IqShaderData x, IqShaderData y, IqShaderData Result, IqShader pShader=0 );
    public void	SO_exp( IqShaderData x, IqShaderData Result, IqShader pShader=0 );
    public void	SO_sqrt( IqShaderData x, IqShaderData Result, IqShader pShader=0 );
    public void	SO_log( IqShaderData x, IqShaderData Result, IqShader pShader=0 );
    public void	SO_log( IqShaderData x, IqShaderData base, IqShaderData Result, IqShader pShader=0 );
    public void	SO_mod( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader=0 );
    public void	SO_abs( IqShaderData x, IqShaderData Result, IqShader pShader=0 );
    public void	SO_sign( IqShaderData x, IqShaderData Result, IqShader pShader=0 );
    public void	SO_min( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_max( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_pmin( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_pmax( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_cmin( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_cmax( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_clamp( IqShaderData a, IqShaderData _min, IqShaderData _max, IqShaderData Result, IqShader pShader=0 );
    public void	SO_pclamp( IqShaderData a, IqShaderData _min, IqShaderData _max, IqShaderData Result, IqShader pShader=0 );
    public void	SO_cclamp( IqShaderData a, IqShaderData _min, IqShaderData _max, IqShaderData Result, IqShader pShader=0 );
    public void	SO_floor( IqShaderData x, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ceil( IqShaderData x, IqShaderData Result, IqShader pShader=0 );
    public void	SO_round( IqShaderData x, IqShaderData Result, IqShader pShader=0 );
    public void	SO_step( IqShaderData _min, IqShaderData value, IqShaderData Result, IqShader pShader=0 );
    public void	SO_smoothstep( IqShaderData _min, IqShaderData _max, IqShaderData value, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fspline( IqShaderData value, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_cspline( IqShaderData value, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_pspline( IqShaderData value, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_sfspline( IqShaderData basis, IqShaderData value, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_scspline( IqShaderData basis, IqShaderData value, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_spspline( IqShaderData basis, IqShaderData value, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_fDu( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fDv( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fDeriv( IqShaderData p, IqShaderData den, IqShaderData Result, IqShader pShader=0 );
    public void	SO_cDu( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_cDv( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_cDeriv( IqShaderData p, IqShaderData den, IqShaderData Result, IqShader pShader=0 );
    public void	SO_pDu( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_pDv( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_pDeriv( IqShaderData p, IqShaderData den, IqShaderData Result, IqShader pShader=0 );
    public void	SO_frandom( IqShaderData Result, IqShader pShader=0 );
    public void	SO_crandom( IqShaderData Result, IqShader pShader=0 );
    public void	SO_prandom( IqShaderData Result, IqShader pShader=0 );
    public void	SO_fnoise1( IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fnoise2( IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fnoise3( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fnoise4( IqShaderData p, IqShaderData t, IqShaderData Result, IqShader pShader=0 );
    public void	SO_cnoise1( IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_cnoise2( IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_cnoise3( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_cnoise4( IqShaderData p, IqShaderData t, IqShaderData Result, IqShader pShader=0 );
    public void	SO_pnoise1( IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_pnoise2( IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_pnoise3( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_pnoise4( IqShaderData p, IqShaderData t, IqShaderData Result, IqShader pShader=0 );
    public void	SO_setcomp( IqShaderData p, IqShaderData i, IqShaderData v, IqShader pShader=0 );
    public void	SO_setxcomp( IqShaderData p, IqShaderData v, IqShader pShader=0 );
    public void	SO_setycomp( IqShaderData p, IqShaderData v, IqShader pShader=0 );
    public void	SO_setzcomp( IqShaderData p, IqShaderData v, IqShader pShader=0 );
    public void	SO_length( IqShaderData V, IqShaderData Result, IqShader pShader=0 );
    public void	SO_distance( IqShaderData P1, IqShaderData P2, IqShaderData Result, IqShader pShader=0 );
    public void	SO_area( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_normalize( IqShaderData V, IqShaderData Result, IqShader pShader=0 );
    public void	SO_faceforward( IqShaderData N, IqShaderData I, IqShaderData Result, IqShader pShader=0 );
    public void	SO_faceforward2( IqShaderData N, IqShaderData I, IqShaderData Nref, IqShaderData Result, IqShader pShader=0 );
    public void	SO_reflect( IqShaderData I, IqShaderData N, IqShaderData Result, IqShader pShader=0 );
    public void	SO_refract( IqShaderData I, IqShaderData N, IqShaderData eta, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fresnel( IqShaderData I, IqShaderData N, IqShaderData eta, IqShaderData Kr, IqShaderData Kt, IqShader pShader=0 );
    public void	SO_fresnel( IqShaderData I, IqShaderData N, IqShaderData eta, IqShaderData Kr, IqShaderData Kt, IqShaderData R, IqShaderData T, IqShader pShader=0 );
    public void	SO_transform( IqShaderData fromspace, IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_transform( IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_transformm( IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_vtransform( IqShaderData fromspace, IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_vtransform( IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_vtransformm( IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ntransform( IqShaderData fromspace, IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ntransform( IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ntransformm( IqShaderData tospace, IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_depth( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_calculatenormal( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_cmix( IqShaderData color0, IqShaderData color1, IqShaderData value, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fmix( IqShaderData f0, IqShaderData f1, IqShaderData value, IqShaderData Result, IqShader pShader=0 );
    public void	SO_pmix( IqShaderData p0, IqShaderData p1, IqShaderData value, IqShaderData Result, IqShader pShader=0 );
    public void	SO_vmix( IqShaderData v0, IqShaderData v1, IqShaderData value, IqShaderData Result, IqShader pShader=0 );
    public void	SO_nmix( IqShaderData n0, IqShaderData n1, IqShaderData value, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ambient( IqShaderData Result, IqShader pShader=0 );
    public void	SO_diffuse( IqShaderData N, IqShaderData Result, IqShader pShader=0 );
    public void	SO_specular( IqShaderData N, IqShaderData V, IqShaderData roughness, IqShaderData Result, IqShader pShader=0 );
    public void	SO_phong( IqShaderData N, IqShaderData V, IqShaderData size, IqShaderData Result, IqShader pShader=0 );
    public void	SO_trace( IqShaderData P, IqShaderData R, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ftexture1( IqShaderData name, IqShaderData channel, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_ftexture2( IqShaderData name, IqShaderData channel, IqShaderData s, IqShaderData t, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_ftexture3( IqShaderData name, IqShaderData channel, IqShaderData s1, IqShaderData t1, IqShaderData s2, IqShaderData t2, IqShaderData s3, IqShaderData t3, IqShaderData s4, IqShaderData t4, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_ctexture1( IqShaderData name, IqShaderData channel, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_ctexture2( IqShaderData name, IqShaderData channel, IqShaderData s, IqShaderData t, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_ctexture3( IqShaderData name, IqShaderData channel, IqShaderData s1, IqShaderData t1, IqShaderData s2, IqShaderData t2, IqShaderData s3, IqShaderData t3, IqShaderData s4, IqShaderData t4, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_fenvironment2( IqShaderData name, IqShaderData channel, IqShaderData R, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_fenvironment3( IqShaderData name, IqShaderData channel, IqShaderData R1, IqShaderData R2, IqShaderData R3, IqShaderData R4, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_cenvironment2( IqShaderData name, IqShaderData channel, IqShaderData R, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_cenvironment3( IqShaderData name, IqShaderData channel, IqShaderData R1, IqShaderData R2, IqShaderData R3, IqShaderData R4, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_bump1( IqShaderData name, IqShaderData channel, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_bump2( IqShaderData name, IqShaderData channel, IqShaderData s, IqShaderData t, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_bump3( IqShaderData name, IqShaderData channel, IqShaderData s1, IqShaderData t1, IqShaderData s2, IqShaderData t2, IqShaderData s3, IqShaderData t3, IqShaderData s4, IqShaderData t4, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_shadow( IqShaderData name, IqShaderData channel, IqShaderData P, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_shadow1( IqShaderData name, IqShaderData channel, IqShaderData P1, IqShaderData P2, IqShaderData P3, IqShaderData P4, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_illuminance( IqShaderData P, IqShaderData nsamples, IqShader pShader=0 );
    public void	SO_illuminance( IqShaderData P, IqShaderData Axis, IqShaderData Angle, IqShaderData nsamples, IqShader pShader=0 );
    public void	SO_illuminate( IqShaderData P, IqShaderData Axis, IqShaderData Angle, IqShader pShader=0 );
    public void	SO_illuminate( IqShaderData P, IqShader pShader=0 );
    public void	SO_solar( IqShaderData Axis, IqShaderData Angle, IqShader pShader=0 );
    public void	SO_solar( IqShader pShader=0 );
    public void	SO_printf( IqShaderData str, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_format( IqShaderData str, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_concat( IqShaderData stra, IqShaderData strb, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_atmosphere( IqShaderData name, IqShaderData* pV, IqShaderData Result, IqShader pShader=0 );
    public void	SO_displacement( IqShaderData name, IqShaderData* pV, IqShaderData Result, IqShader pShader=0 );
    public void	SO_lightsource( IqShaderData name, IqShaderData* pV, IqShaderData Result, IqShader pShader=0 );
    public void	SO_surface( IqShaderData name, IqShaderData* pV, IqShaderData Result, IqShader pShader=0 );
    public void	SO_attribute( IqShaderData name, IqShaderData* pV, IqShaderData Result, IqShader pShader=0 );
    public void	SO_option( IqShaderData name, IqShaderData* pV, IqShaderData Result, IqShader pShader=0 );
    public void	SO_rendererinfo( IqShaderData name, IqShaderData* pV, IqShaderData Result, IqShader pShader=0 );
    public void	SO_incident( IqShaderData name, IqShaderData* pV, IqShaderData Result, IqShader pShader=0 );
    public void	SO_opposite( IqShaderData name, IqShaderData* pV, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fcellnoise1( IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fcellnoise2( IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fcellnoise3( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fcellnoise4( IqShaderData p, IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ccellnoise1( IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ccellnoise2( IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ccellnoise3( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ccellnoise4( IqShaderData p, IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_pcellnoise1( IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_pcellnoise2( IqShaderData u, IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_pcellnoise3( IqShaderData p, IqShaderData Result, IqShader pShader=0 );
    public void	SO_pcellnoise4( IqShaderData p, IqShaderData v, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fpnoise1( IqShaderData v, IqShaderData period, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fpnoise2( IqShaderData u, IqShaderData v, IqShaderData uperiod, IqShaderData vperiod, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fpnoise3( IqShaderData p, IqShaderData pperiod, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fpnoise4( IqShaderData p, IqShaderData t, IqShaderData pperiod, IqShaderData tperiod, IqShaderData Result, IqShader pShader=0 );
    public void	SO_cpnoise1( IqShaderData v, IqShaderData period, IqShaderData Result, IqShader pShader=0 );
    public void	SO_cpnoise2( IqShaderData u, IqShaderData v, IqShaderData uperiod, IqShaderData vperiod, IqShaderData Result, IqShader pShader=0 );
    public void	SO_cpnoise3( IqShaderData p, IqShaderData pperiod, IqShaderData Result, IqShader pShader=0 );
    public void	SO_cpnoise4( IqShaderData p, IqShaderData t, IqShaderData pperiod, IqShaderData tperiod, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ppnoise1( IqShaderData v, IqShaderData period, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ppnoise2( IqShaderData u, IqShaderData v, IqShaderData uperiod, IqShaderData vperiod, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ppnoise3( IqShaderData p, IqShaderData pperiod, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ppnoise4( IqShaderData p, IqShaderData t, IqShaderData pperiod, IqShaderData tperiod, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ctransform( IqShaderData fromspace, IqShaderData tospace, IqShaderData c, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ctransform( IqShaderData tospace, IqShaderData c, IqShaderData Result, IqShader pShader=0 );
    public void	SO_ptlined( IqShaderData P0, IqShaderData P1, IqShaderData Q, IqShaderData Result, IqShader pShader=0 );
    public void	SO_inversesqrt( IqShaderData x, IqShaderData Result, IqShader pShader=0 );
    public void	SO_match( IqShaderData a, IqShaderData b, IqShaderData Result, IqShader pShader=0 );
    public void	SO_rotate( IqShaderData Q, IqShaderData angle, IqShaderData P0, IqShaderData P1, IqShaderData Result, IqShader pShader=0 );
    public void	SO_filterstep( IqShaderData edge, IqShaderData s1, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_filterstep2( IqShaderData edge, IqShaderData s1, IqShaderData s2, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
    public void	SO_specularbrdf( IqShaderData L, IqShaderData N, IqShaderData V, IqShaderData rough, IqShaderData Result, IqShader pShader=0 );
    public void	SO_setmcomp( IqShaderData M, IqShaderData row, IqShaderData column, IqShaderData val, IqShader pShader=0 );
    public void	SO_determinant( IqShaderData M, IqShaderData Result, IqShader pShader=0 );
    public void	SO_mtranslate( IqShaderData M, IqShaderData V, IqShaderData Result, IqShader pShader=0 );
    public void	SO_mrotate( IqShaderData M, IqShaderData angle, IqShaderData axis, IqShaderData Result, IqShader pShader=0 );
    public void	SO_mscale( IqShaderData M, IqShaderData s, IqShaderData Result, IqShader pShader=0 );
    public void	SO_fsplinea( IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader=0 );
    public void	SO_csplinea( IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader=0 );
    public void	SO_psplinea( IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader=0 );
    public void	SO_sfsplinea( IqShaderData basis, IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader=0 );
    public void	SO_scsplinea( IqShaderData basis, IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader=0 );
    public void	SO_spsplinea( IqShaderData basis, IqShaderData value, IqShaderData a, IqShaderData Result, IqShader pShader=0 );
    public void	SO_shadername( IqShaderData Result, IqShader pShader=0 );
    public void	SO_shadername2( IqShaderData shader, IqShaderData Result, IqShader pShader=0 );
    public void	SO_textureinfo( IqShaderData shader, IqShaderData dataname, IqShaderData* pV, IqShaderData Result, IqShader pShader=0 );
    public void	SO_bake_f( IqShaderData shader, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0);
    public void	SO_bake_3c( IqShaderData shader, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0);
    public void	SO_bake_3p( IqShaderData shader, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0);
    public void	SO_bake_3n( IqShaderData shader, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0);
    public void	SO_bake_3v( IqShaderData shader, IqShaderData s, IqShaderData t, IqShaderData f, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0);
    public void	SO_external(DSOMethod method, void *initData, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0);
    public	void	SO_occlusion( IqShaderData occlmap, IqShaderData channel, IqShaderData P, IqShaderData N, IqShaderData samples, IqShaderData Result, IqShader pShader=0, int cParams=0, IqShaderData[] apParams=0 );
*/
    
}
