// Aqsis
// Copyright (c) 1997 - 2001, Paul C. Gregory
//
// Contact: pgregory@aqsis.com
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/**
 * Copyright (C) 2006-2007  NTT DATA CORPORATION
 * 
 * Version: 1.0.0 2007/04/01
 *  
 */
package net.cellcomputing.himawari.library;

import static net.cellcomputing.himawari.library.EqProjection.ProjectionOrthographic;
import static net.cellcomputing.himawari.library.EqProjection.ProjectionPerspective;
import static net.cellcomputing.himawari.library.EqShaderType.Type_Imager;
import static net.cellcomputing.himawari.library.EqVariableType.*;
import static net.cellcomputing.himawari.library.EqDisplayMode.*;
import static net.cellcomputing.himawari.library.RiGlobal.QGetRenderContext;
import static net.cellcomputing.himawari.library.RiGlobal.gColWhite;
import static net.cellcomputing.himawari.library.Float_h.FLT_EPSILON;

import java.lang.reflect.Method;

import net.cellcomputing.himawari.accessory.STLVector;
import net.cellcomputing.himawari.accessory.primitive.p_String;
import net.cellcomputing.himawari.accessory.primitive.p_float;
import net.cellcomputing.himawari.accessory.primitive.p_int;
import net.cellcomputing.himawari.exception.XqException;
import net.cellcomputing.himawari.library.types.CqColor;
import net.cellcomputing.himawari.library.types.CqMatrix;
import net.cellcomputing.himawari.library.types.CqVector3D;
import net.cellcomputing.himawari.util.HimawariLogger;

/**
 * 
 * 
 * 
 * @author NTT DATA Corporation
 */
public strictfp class CqOptions {
	
	private STLVector<CqNamedParameterList> m_aOptions = new STLVector<CqNamedParameterList>(CqNamedParameterList.class);	///< Vector of user specified options.
	
	private Method m_funcFilter;						///< Pointer to the pixel filter function.
	private CqImagersource m_pshadImager;				///< Pointer to the imager shader.
	
	private boolean	m_bFrameAspectRatioCalled,   	///< Indicate RiFrameAspectRatio has been called. Calculation of the screen geometry is reliant on which of these have been called.
	m_bScreenWindowCalled,   							///< Indicate RiScreenWindow has been called. Calculation of the screen geometry is reliant on which of these have been called.
	m_bFormatCalled;									///< Indicate RiFormat has been called. Calculation of the screen geometry is reliant on which of these have been called.
	
	//************************************************************************//
	//*** RXgN^̒`
	//************************************************************************//
	
	public CqOptions()
	{
		try {
			m_funcFilter = RendermanInterface.class.getMethod( "RiGaussianFilter", float.class, float.class, float.class, float.class );
		} catch (SecurityException e) {
			m_funcFilter = null;
//			e.printStackTrace();
		} catch (NoSuchMethodException e) {
			m_funcFilter = null;
//			e.printStackTrace();
		}
		
		m_pshadImager = null;
		m_bFrameAspectRatioCalled = false;
		m_bScreenWindowCalled = false;
		m_bFormatCalled = false;
		
		InitialiseDefaultOptions();
	}
	
	public CqOptions( final CqOptions From )
	{
		this.assignment( From );
	}
	
	public void destruct()
	{
		DeleteImager();
	}
	
	
	//************************************************************************//
	//*** \bh̒`
	//************************************************************************//
	
	/** Add a named user option to the current state.
	 * @param pOption A pointer to a new CqNamedParameterList class containing the initialised option.
	 */
	public void AddOption( CqNamedParameterList pOption )
	{
		m_aOptions.add( pOption );
	}
	
	
	/** Clear all user options from the state.
	 */
	public void ClearOptions()
	{
		m_aOptions.clear();
		InitialiseDefaultOptions();
	}
	
	/** Initialise default system options.
	 */
	//nttdata 
//	public void InitialiseDefaultOptions()
//	{
//		CqNamedParameterList pdefopts = new CqNamedParameterList( "System" );
//		
//		{
//			CqParameterTypedUniform<p_float,p_float> pPixelVariance = new CqParameterTypedUniform<p_float,p_float>("PixelVariance",new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pPixelVariance.pValue()[0]).value = ( 1.0f );
//			pdefopts.AddParameter(pPixelVariance);
//		}
//		{
//			CqParameterTypedUniformArray<p_int,p_float> pPixelSamples = new CqParameterTypedUniformArray<p_int,p_float>("PixelSamples",2,new EqVariableType(type_integer),p_int.class,p_float.class);
//			((p_int)pPixelSamples.pValue()[0]).value = ( 2 );
//			((p_int)pPixelSamples.pValue()[1]).value = ( 2 );
//			pdefopts.AddParameter(pPixelSamples);;
//		}
//		{
//			CqParameterTypedUniformArray<p_float,p_float> pFilterWidth = new CqParameterTypedUniformArray<p_float,p_float>("FilterWidth",2,new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pFilterWidth.pValue()[0]).value = ( 2.0f );
//			((p_float)pFilterWidth.pValue()[1]).value = ( 2.0f );
//			pdefopts.AddParameter(pFilterWidth);;
//		}
//		{
//			CqParameterTypedUniformArray<p_float,p_float> pExposure = new CqParameterTypedUniformArray<p_float,p_float>("Exposure",2,new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pExposure.pValue()[0]).value = ( 1.0f );
//			((p_float)pExposure.pValue()[1]).value = ( 1.0f );
//			pdefopts.AddParameter(pExposure);;
//		}
//		{
//			CqParameterTypedUniform<p_String,p_String> pImager = new CqParameterTypedUniform<p_String,p_String>("Imager",new EqVariableType(type_string),p_String.class,p_String.class);
//			((p_String)pImager.pValue()[0]).value = ( "null" );
//			pdefopts.AddParameter(pImager);;
//		}
//		{
//			CqParameterTypedUniform<p_String,p_String> pDisplayType = new CqParameterTypedUniform<p_String,p_String>("DisplayType",new EqVariableType(type_string),p_String.class,p_String.class);
//			((p_String)pDisplayType.pValue()[0]).value = ( "file" );
//			pdefopts.AddParameter(pDisplayType);;
//		}
//		{
//			CqParameterTypedUniform<p_String,p_String> pDisplayName = new CqParameterTypedUniform<p_String,p_String>("DisplayName",new EqVariableType(type_string),p_String.class,p_String.class);
//			((p_String)pDisplayName.pValue()[0]).value = ( "aqsis.tif" );
//			pdefopts.AddParameter(pDisplayName);;
//		}
//		{
//			CqParameterTypedUniform<p_int,p_float> pDisplayMode = new CqParameterTypedUniform<p_int,p_float>("DisplayMode",new EqVariableType(type_integer),p_int.class,p_float.class);
//			((p_int)pDisplayMode.pValue()[0]).value = ( ModeRGB );
//			pdefopts.AddParameter(pDisplayMode);;
//		}
//		
//		{
//			CqParameterTypedUniform<p_String,p_String> pHider = new CqParameterTypedUniform<p_String,p_String>("Hider",new EqVariableType(type_string),p_String.class,p_String.class);
//			((p_String)pHider.pValue()[0]).value = ( "hidden" );
//			pdefopts.AddParameter(pHider);;
//		}
//		{
//			CqParameterTypedUniform<p_int,p_float> pColorSamples = new CqParameterTypedUniform<p_int,p_float>("ColorSamples",new EqVariableType(type_integer),p_int.class,p_float.class);
//			((p_int)pColorSamples.pValue()[0]).value = ( 3 );
//			pdefopts.AddParameter(pColorSamples);;
//		}
//		{
//			CqParameterTypedUniform<p_float,p_float> pRelativeDetail = new CqParameterTypedUniform<p_float,p_float>("RelativeDetail",new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pRelativeDetail.pValue()[0]).value = ( 1.0f );
//			pdefopts.AddParameter(pRelativeDetail);;
//		}
//		
//		{
//			CqParameterTypedUniformArray<p_int,p_float> pResolution = new CqParameterTypedUniformArray<p_int,p_float>("Resolution",2,new EqVariableType(type_integer),p_int.class,p_float.class);
//			((p_int)pResolution.pValue()[0]).value = ( 640 );
//			((p_int)pResolution.pValue()[1]).value = ( 480 );
//			pdefopts.AddParameter(pResolution);;
//		}
//		{
//			CqParameterTypedUniform<p_float,p_float> pPixelAspectRatio = new CqParameterTypedUniform<p_float,p_float>("PixelAspectRatio",new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pPixelAspectRatio.pValue()[0]).value = ( 1.0f );
//			pdefopts.AddParameter(pPixelAspectRatio);;
//		}
//		{
//			CqParameterTypedUniformArray<p_float,p_float> pCropWindow = new CqParameterTypedUniformArray<p_float,p_float>("CropWindow",4,new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pCropWindow.pValue()[0]).value = ( 0.0f ); 
//			((p_float)pCropWindow.pValue()[1]).value = ( 1.0f );
//			((p_float)pCropWindow.pValue()[2]).value = ( 0.0f );
//			((p_float)pCropWindow.pValue()[3]).value = ( 1.0f );
//			pdefopts.AddParameter(pCropWindow);;
//		}
//		{
//			CqParameterTypedUniform<p_float,p_float> pFrameAspectRatio = new CqParameterTypedUniform<p_float,p_float>("FrameAspectRatio",new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pFrameAspectRatio.pValue()[0]).value = ( 4.0f / 3.0f );
//			pdefopts.AddParameter(pFrameAspectRatio);;
//		}
//		{
//			CqParameterTypedUniformArray<p_float,p_float> pScreenWindow = new CqParameterTypedUniformArray<p_float,p_float>("ScreenWindow",4,new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pScreenWindow.pValue()[0]).value = ( -( 4.0f / 3.0f ) );
//			((p_float)pScreenWindow.pValue()[1]).value = ( ( 4.0f / 3.0f ) );
//			((p_float)pScreenWindow.pValue()[2]).value = ( 1.0f );
//			((p_float)pScreenWindow.pValue()[3]).value = ( -1.0f );
//			pdefopts.AddParameter(pScreenWindow);;
//		}
//		{
//			CqParameterTypedUniform<p_int,p_float> pProjection = new CqParameterTypedUniform<p_int,p_float>("Projection",new EqVariableType(type_integer),p_int.class,p_float.class);
//			((p_int)pProjection.pValue()[0]).value = ( ProjectionOrthographic );
//			pdefopts.AddParameter(pProjection);;
//		}
//		{
//			CqParameterTypedUniformArray<p_float,p_float> pClipping = new CqParameterTypedUniformArray<p_float,p_float>("Clipping",2,new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pClipping.pValue()[0]).value = ( FLT_EPSILON );
//			((p_float)pClipping.pValue()[1]).value = ( Float.MAX_VALUE );
//			pdefopts.AddParameter(pClipping);;
//		}
//		{
//			CqParameterTypedUniformArray<p_float,p_float> pDepthOfField = new CqParameterTypedUniformArray<p_float,p_float>("DepthOfField",3,new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pDepthOfField.pValue()[0]).value = ( Float.MAX_VALUE );
//			((p_float)pDepthOfField.pValue()[1]).value = ( Float.MAX_VALUE );
//			((p_float)pDepthOfField.pValue()[2]).value = ( Float.MAX_VALUE );
//			pdefopts.AddParameter(pDepthOfField);;
//		}
//		{
//			CqParameterTypedUniformArray<p_float,p_float> pShutter = new CqParameterTypedUniformArray<p_float,p_float>("Shutter",2,new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pShutter.pValue()[0]).value = ( 0.0f ); 
//			((p_float)pShutter.pValue()[1]).value = ( 1.0f );
//			pdefopts.AddParameter(pShutter);;
//		}
//		{
//			CqParameterTypedUniform<p_float,p_float> pFOV = new CqParameterTypedUniform<p_float,p_float>("FOV",new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pFOV.pValue()[0]).value = ( 90.0f );
//			pdefopts.AddParameter(pFOV);;
//		}
//		{
//			CqParameterTypedUniform<p_float,p_float> pSqrtGridSize = new CqParameterTypedUniform<p_float,p_float>("SqrtGridSize",new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pSqrtGridSize.pValue()[0]).value = ( 16.0f );
//			pdefopts.AddParameter(pSqrtGridSize);;
//		}
//		
//		AddOption( pdefopts );
//		
//		pdefopts = new CqNamedParameterList( "Quantize" );
//		
//		{
//			CqParameterTypedUniformArray<p_float,p_float> pColor = new CqParameterTypedUniformArray<p_float,p_float>("Color",4,new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pColor.pValue()[0]).value = ( 255.0f );
//			((p_float)pColor.pValue()[1]).value = ( 0.0f );
//			((p_float)pColor.pValue()[2]).value = ( 255.0f );
//			((p_float)pColor.pValue()[3]).value = ( 0.5f );
//			pdefopts.AddParameter(pColor);;
//		}
//		{
//			CqParameterTypedUniformArray<p_float,p_float> pDepth = new CqParameterTypedUniformArray<p_float,p_float>("Depth",4,new EqVariableType(type_float),p_float.class,p_float.class);
//			((p_float)pDepth.pValue()[0]).value = ( 0.0f );
//			((p_float)pDepth.pValue()[1]).value = ( 0.0f );
//			((p_float)pDepth.pValue()[2]).value = ( 0.0f );
//			((p_float)pDepth.pValue()[3]).value = ( 0.0f );
//			pdefopts.AddParameter(pDepth);;
//		}
//		
//		AddOption( pdefopts );
//	}
	public void InitialiseDefaultOptions()
	{
		CqNamedParameterList pdefopts = new CqNamedParameterList( "System" );
		
		{
			CqParameterTypedUniform<p_float,p_float> pPixelVariance = new CqParameterTypedUniform<p_float,p_float>("PixelVariance",new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pPixelVariance.pValue_get(0,0)).value = ( 1.0f );
			pdefopts.AddParameter(pPixelVariance);
		}
		{
			CqParameterTypedUniformArray<p_int,p_float> pPixelSamples = new CqParameterTypedUniformArray<p_int,p_float>("PixelSamples",2,new EqVariableType(type_integer),p_int.class,p_float.class);
			((p_int)pPixelSamples.pValue_get(0,0)).value = ( 2 );
			((p_int)pPixelSamples.pValue_get(0,1)).value = ( 2 );
			pdefopts.AddParameter(pPixelSamples);;
		}
		{
			CqParameterTypedUniformArray<p_float,p_float> pFilterWidth = new CqParameterTypedUniformArray<p_float,p_float>("FilterWidth",2,new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pFilterWidth.pValue_get(0,0)).value = ( 2.0f );
			((p_float)pFilterWidth.pValue_get(0,1)).value = ( 2.0f );
			pdefopts.AddParameter(pFilterWidth);;
		}
		{
			CqParameterTypedUniformArray<p_float,p_float> pExposure = new CqParameterTypedUniformArray<p_float,p_float>("Exposure",2,new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pExposure.pValue_get(0,0)).value = ( 1.0f );
			((p_float)pExposure.pValue_get(0,1)).value = ( 1.0f );
			pdefopts.AddParameter(pExposure);;
		}
		{
			CqParameterTypedUniform<p_String,p_String> pImager = new CqParameterTypedUniform<p_String,p_String>("Imager",new EqVariableType(type_string),p_String.class,p_String.class);
			((p_String)pImager.pValue_get(0,0)).value = ( "null" );
			pdefopts.AddParameter(pImager);;
		}
		{
			CqParameterTypedUniform<p_String,p_String> pDisplayType = new CqParameterTypedUniform<p_String,p_String>("DisplayType",new EqVariableType(type_string),p_String.class,p_String.class);
			((p_String)pDisplayType.pValue_get(0,0)).value = ( "file" );
			pdefopts.AddParameter(pDisplayType);;
		}
		{
			CqParameterTypedUniform<p_String,p_String> pDisplayName = new CqParameterTypedUniform<p_String,p_String>("DisplayName",new EqVariableType(type_string),p_String.class,p_String.class);
			((p_String)pDisplayName.pValue_get(0,0)).value = ( "aqsis.tif" );
			pdefopts.AddParameter(pDisplayName);;
		}
		{
			CqParameterTypedUniform<p_int,p_float> pDisplayMode = new CqParameterTypedUniform<p_int,p_float>("DisplayMode",new EqVariableType(type_integer),p_int.class,p_float.class);
			((p_int)pDisplayMode.pValue_get(0,0)).value = ( ModeRGB );
			pdefopts.AddParameter(pDisplayMode);;
		}
		
		{
			CqParameterTypedUniform<p_String,p_String> pHider = new CqParameterTypedUniform<p_String,p_String>("Hider",new EqVariableType(type_string),p_String.class,p_String.class);
			((p_String)pHider.pValue_get(0,0)).value = ( "hidden" );
			pdefopts.AddParameter(pHider);;
		}
		{
			CqParameterTypedUniform<p_int,p_float> pColorSamples = new CqParameterTypedUniform<p_int,p_float>("ColorSamples",new EqVariableType(type_integer),p_int.class,p_float.class);
			((p_int)pColorSamples.pValue_get(0,0)).value = ( 3 );
			pdefopts.AddParameter(pColorSamples);;
		}
		{
			CqParameterTypedUniform<p_float,p_float> pRelativeDetail = new CqParameterTypedUniform<p_float,p_float>("RelativeDetail",new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pRelativeDetail.pValue_get(0,0)).value = ( 1.0f );
			pdefopts.AddParameter(pRelativeDetail);;
		}
		
		{
			CqParameterTypedUniformArray<p_int,p_float> pResolution = new CqParameterTypedUniformArray<p_int,p_float>("Resolution",2,new EqVariableType(type_integer),p_int.class,p_float.class);
			((p_int)pResolution.pValue_get(0,0)).value = ( 640 );
			((p_int)pResolution.pValue_get(0,1)).value = ( 480 );
			pdefopts.AddParameter(pResolution);;
		}
		{
			CqParameterTypedUniform<p_float,p_float> pPixelAspectRatio = new CqParameterTypedUniform<p_float,p_float>("PixelAspectRatio",new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pPixelAspectRatio.pValue_get(0,0)).value = ( 1.0f );
			pdefopts.AddParameter(pPixelAspectRatio);;
		}
		{
			CqParameterTypedUniformArray<p_float,p_float> pCropWindow = new CqParameterTypedUniformArray<p_float,p_float>("CropWindow",4,new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pCropWindow.pValue_get(0,0)).value = ( 0.0f ); 
			((p_float)pCropWindow.pValue_get(0,1)).value = ( 1.0f );
			((p_float)pCropWindow.pValue_get(0,2)).value = ( 0.0f );
			((p_float)pCropWindow.pValue_get(0,3)).value = ( 1.0f );
			pdefopts.AddParameter(pCropWindow);;
		}
		{
			CqParameterTypedUniform<p_float,p_float> pFrameAspectRatio = new CqParameterTypedUniform<p_float,p_float>("FrameAspectRatio",new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pFrameAspectRatio.pValue_get(0,0)).value = ( 4.0f / 3.0f );
			pdefopts.AddParameter(pFrameAspectRatio);;
		}
		{
			CqParameterTypedUniformArray<p_float,p_float> pScreenWindow = new CqParameterTypedUniformArray<p_float,p_float>("ScreenWindow",4,new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pScreenWindow.pValue_get(0,0)).value = ( -( 4.0f / 3.0f ) );
			((p_float)pScreenWindow.pValue_get(0,1)).value = ( ( 4.0f / 3.0f ) );
			((p_float)pScreenWindow.pValue_get(0,2)).value = ( 1.0f );
			((p_float)pScreenWindow.pValue_get(0,3)).value = ( -1.0f );
			pdefopts.AddParameter(pScreenWindow);;
		}
		{
			CqParameterTypedUniform<p_int,p_float> pProjection = new CqParameterTypedUniform<p_int,p_float>("Projection",new EqVariableType(type_integer),p_int.class,p_float.class);
			((p_int)pProjection.pValue_get(0,0)).value = ( ProjectionOrthographic );
			pdefopts.AddParameter(pProjection);;
		}
		{
			CqParameterTypedUniformArray<p_float,p_float> pClipping = new CqParameterTypedUniformArray<p_float,p_float>("Clipping",2,new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pClipping.pValue_get(0,0)).value = ( FLT_EPSILON );
			((p_float)pClipping.pValue_get(0,1)).value = ( Float.MAX_VALUE );
			pdefopts.AddParameter(pClipping);;
		}
		{
			CqParameterTypedUniformArray<p_float,p_float> pDepthOfField = new CqParameterTypedUniformArray<p_float,p_float>("DepthOfField",3,new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pDepthOfField.pValue_get(0,0)).value = ( Float.MAX_VALUE );
			((p_float)pDepthOfField.pValue_get(0,1)).value = ( Float.MAX_VALUE );
			((p_float)pDepthOfField.pValue_get(0,2)).value = ( Float.MAX_VALUE );
			pdefopts.AddParameter(pDepthOfField);;
		}
		{
			CqParameterTypedUniformArray<p_float,p_float> pShutter = new CqParameterTypedUniformArray<p_float,p_float>("Shutter",2,new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pShutter.pValue_get(0,0)).value = ( 0.0f ); 
			((p_float)pShutter.pValue_get(0,1)).value = ( 1.0f );
			pdefopts.AddParameter(pShutter);;
		}
		{
			CqParameterTypedUniform<p_float,p_float> pFOV = new CqParameterTypedUniform<p_float,p_float>("FOV",new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pFOV.pValue_get(0,0)).value = ( 90.0f );
			pdefopts.AddParameter(pFOV);;
		}
		{
			CqParameterTypedUniform<p_float,p_float> pSqrtGridSize = new CqParameterTypedUniform<p_float,p_float>("SqrtGridSize",new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pSqrtGridSize.pValue_get(0,0)).value = ( 16.0f );
			pdefopts.AddParameter(pSqrtGridSize);;
		}
		
		AddOption( pdefopts );
		
		pdefopts = new CqNamedParameterList( "Quantize" );
		
		{
			CqParameterTypedUniformArray<p_float,p_float> pColor = new CqParameterTypedUniformArray<p_float,p_float>("Color",4,new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pColor.pValue_get(0,0)).value = ( 255.0f );
			((p_float)pColor.pValue_get(0,1)).value = ( 0.0f );
			((p_float)pColor.pValue_get(0,2)).value = ( 255.0f );
			((p_float)pColor.pValue_get(0,3)).value = ( 0.5f );
			pdefopts.AddParameter(pColor);;
		}
		{
			CqParameterTypedUniformArray<p_float,p_float> pDepth = new CqParameterTypedUniformArray<p_float,p_float>("Depth",4,new EqVariableType(type_float),p_float.class,p_float.class);
			((p_float)pDepth.pValue_get(0,0)).value = ( 0.0f );
			((p_float)pDepth.pValue_get(0,1)).value = ( 0.0f );
			((p_float)pDepth.pValue_get(0,2)).value = ( 0.0f );
			((p_float)pDepth.pValue_get(0,3)).value = ( 0.0f );
			pdefopts.AddParameter(pDepth);;
		}
		
		AddOption( pdefopts );
	}
	
	/** Get a read only pointer to a named user option.
	 * @param strName Character pointer to the requested options name.
	 * @return A pointer to the option, or 0 if not found. 
	 */
	public CqNamedParameterList pOption( final String strName )
	{
		long hash = strName.hashCode();
		for( CqNamedParameterList i : m_aOptions )
		{
			if ( i.hash() == hash )
			{
				return ( i );
			}
		}
		CqNamedParameterList retval = null;
		return ( retval );
	}
	
	/** Get a pointer to a named user option.
	 * @param strName Character pointer to the requested options name.
	 * @return A pointer to the option, or 0 if not found. 
	 */
	public CqNamedParameterList pOptionWrite( final String strName )
	{
		long hash = strName.hashCode();
//		std::vector<boost::shared_ptr<CqNamedParameterList> >::iterator
//		i = m_aOptions.begin(), end = m_aOptions.end();
//		for ( CqNamedParameterList i : m_aOptions )
		for( int i=0; i<m_aOptions.size(); i++ )
		{
			CqNamedParameterList plist = m_aOptions.get(i);
			if ( plist.hash() == hash )
			{
				//XXX: unique()ɑāAPreturnB 2005.08.24.nttdata
//				if ( plist.unique() )
//				{
					return plist;
//				}
//				else
//				{
//					CqNamedParameterList pNew = new CqNamedParameterList( plist );
//					m_aOptions.set( i, pNew );
//					
//					return ( pNew );
//				}
			}
		}
		m_aOptions.add( new CqNamedParameterList( strName ) );
		return ( m_aOptions.lastElement() );
	}
	
	/** Get a system option parameter, takes name and parameter name.
	 * \param strName The name of the option.
	 * \param strParam The name of the paramter on the option.
	 * \return CqParameter pointer or 0 if not found.
	 */

	public CqParameter pParameter( final String strName, final String strParam )
	{
	    CqNamedParameterList pList = pOption( strName );
	    if ( pList!=null )
	    {
	        CqParameter pParam;
	        if ( ( pParam = pList.pParameter( strParam ) ) != null )
	            return ( pParam );
	    }
	    return ( null );
	}

	/** Get a system option parameter, takes name and parameter name.
	 * \param strName The name of the option.
	 * \param strParam The name of the paramter on the option.
	 * \return CqParameter pointer or 0 if not found.
	 */
	public CqParameter pParameterWrite( final String strName, final String strParam )
	{
	    CqNamedParameterList pList = pOptionWrite( strName );
	    if ( pList!=null )
	    {
	        CqParameter pParam;
	        if ( ( pParam = pList.pParameter( strParam ) ) != null )
	            return ( pParam );
	    }
	    return ( null );
	}
	
	
	
	
	/** Get a float system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return Float pointer 0 if not found.
	 */
	public final p_float GetFloatOptionIndex( final String strName, final String strParam ,final int index)
	{
	    final CqParameter pParam = pParameter( strName, strParam );
	    if ( pParam != null ){
	    	return ((CqParameterTyped<p_float, p_float>)( pParam ) ).pValue_get(0,index);
	    }
	    else
	        return null;
	}
	
	/** Get an integer system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return Integer pointer 0 if not found.
	 */
	public final p_int GetIntegerOptionIndex( final String strName, final String strParam ,final int index)
	{
	    final CqParameter pParam = pParameter( strName, strParam );
	    if ( pParam != null ){
	    	return ( (CqParameterTyped<p_int, p_float>)( pParam ) ).pValue_get(0,index);
	    }
	    else
	        return ( null );
	}
	
	/** Get a string system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return CqString pointer 0 if not found.
	 */
	public final	p_String GetStringOptionIndex( final String strName, final String strParam ,final int index)
	{
	    final CqParameter pParam = pParameter( strName, strParam );
	    if ( pParam != null ){
	    	return ((CqParameterTyped<p_String, String>)( pParam )).pValue_get(0,index);
	    }
	    else
	        return ( null );
	}
	
	/** Get a point system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return CqVector3D pointer 0 if not found.
	 */
	public final	CqVector3D GetPointOptionIndex( final String strName, final String strParam ,final int index)
	{
	    final CqParameter pParam = pParameter( strName, strParam );
	    if ( pParam != null ){
	    	return ((CqParameterTyped<CqVector3D, CqVector3D>)( pParam )).pValue_get(0,index);
	    }
	    else
	        return ( null );
	}
	
	/** Get a color system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return CqColor pointer 0 if not found.
	 */
	public final	CqColor	GetColorOptionIndex( final String strName, final String strParam ,final int index )
	{
	    final CqParameter pParam = pParameter( strName, strParam );
	    if ( pParam != null ){
	    	return ((CqParameterTyped<CqColor, CqColor>)( pParam )).pValue_get(0,index);
	    }
	    else
	        return ( null );
	}

	/** Get a float system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return Float pointer 0 if not found.
	 */
	public p_float GetFloatOptionWriteIndex( final String strName, final String strParam, final int index )
	{
	    CqParameter pParam = pParameterWrite( strName, strParam );
	    if ( pParam != null ){
//	        return (p_float[])( ((CqParameterTyped<p_float, Float>)pParam).pValue() );
	    	return ((CqParameterTyped<p_float, p_float>)( pParam )).pValue_get(0,index);
	    }
	    else
	        return ( null );
	}
	
	/** Get an integer system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return Integer pointer 0 if not found.
	 */
	public p_int	GetIntegerOptionWriteIndex( final String strName, final String strParam ,final int index)
	{
	    CqParameter pParam = pParameterWrite( strName, strParam );
	    if ( pParam != null ){
//	        return (p_int[])( ((CqParameterTyped<p_int, Float>)pParam).pValue() );
	    	return ((CqParameterTyped<p_int, p_float>)( pParam )).pValue_get(0,index);
	    }
	    else
	        return ( null );
	}
	
	/** Get a string system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return CqString pointer 0 if not found.
	 */
	//nttdata 
	public p_String GetStringOptionWriteIndex( final String strName, final String strParam ,final int index)
	{
	    CqParameter pParam = pParameterWrite( strName, strParam );
	    if ( pParam != null ){
//	        return (String[])( ((CqParameterTyped<String, String>)pParam).pValue() );
	    	return ((CqParameterTyped<p_String, String>)( pParam )).pValue_get(0,index);
	    }
	    else
	        return ( null );
	}
	
	/** Get a point system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return CqVector3D pointer 0 if not found.
	 */
	public CqVector3D GetPointOptionWriteIndex( final String strName, final String strParam ,final int index)
	{
	    CqParameter pParam = pParameterWrite( strName, strParam );
	    if ( pParam != null ){
	    	return ((CqParameterTyped<CqVector3D, CqVector3D>)( pParam )).pValue_get(0,index);
	    }
	    else
	        return ( null );
	}
	
	/** Get a color system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return CqColor pointer 0 if not found.
	 */
	public CqColor GetColorOptionWriteIndex( final String strName, final String strParam ,final int index)
	{
	    CqParameter pParam = pParameterWrite( strName, strParam );
	    if ( pParam != null ){
	    	return  ((CqParameterTyped<CqColor, CqColor>)( pParam )).pValue_get(0,index);
	    }
	    else
	        return ( null );
	}
	
	/** Get a float system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return Float pointer 0 if not found.
	 */
	public final float[] GetFloatOption( final String strName, final String strParam )
	{
	    final CqParameter pParam = pParameter( strName, strParam );
	    if ( pParam != null ){
	    	// float[]Ɍ^ϊĂ݂܂B2005.08.30.nttdata
//	        return (float[])( ((CqParameterTyped<Float, Float>)pParam ).pValue() );
	    	Object[] pTemp = ( (CqParameterTyped<p_float, p_float>)( pParam ) ).pValue();
	    	float[] ret = new float[ pTemp.length ];
	    	for( int i=0; i<pTemp.length; i++ ){
	    		ret[i] = ((p_float)pTemp[i]).value;
	    	}
	    	return ret;
	    }
	    else
	        return ( null );
	}
	
	/** Get an integer system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return Integer pointer 0 if not found.
	 */
	public final int[] GetIntegerOption( final String strName, final String strParam )
	{
	    final CqParameter pParam = pParameter( strName, strParam );
	    if ( pParam != null ){
	    	// int[]Ɍ^ϊĂ݂܂B2005.08.30.nttdata
//	        return (int[])( ((CqParameterTyped<Integer, Float>)pParam ).pValue() );
	    	Object[] pTemp = ( (CqParameterTyped<p_int, p_float>)( pParam ) ).pValue();
	    	int[] ret = new int[ pTemp.length ];
	    	for( int i=0; i<pTemp.length; i++ ){
	    		ret[i] = ((p_int)pTemp[i]).value;
	    	}
	    	return ret;
	    }
	    else
	        return ( null );
	}
	
	/** Get a string system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return CqString pointer 0 if not found.
	 */
	public final	String[] GetStringOption( final String strName, final String strParam )
	{
	    final CqParameter pParam = pParameter( strName, strParam );
	    if ( pParam != null ){
//	        return (String[])( ((CqParameterTyped<String, String>)pParam ).pValue() );
	        
	    	Object[] pvalue = ((CqParameterTyped<p_String, String>)( pParam )).pValue();
	    	String[] ret = new String[pvalue.length];
	    	for( int i=0; i<pvalue.length; i++ ){
	    		ret[i] = ((p_String)pvalue[i]).value;
	    	}
	    	return ret;
	    }
	    else
	        return ( null );
	}
	
	/** Get a point system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return CqVector3D pointer 0 if not found.
	 */
	public final	CqVector3D[]	GetPointOption( final String strName, final String strParam )
	{
	    final CqParameter pParam = pParameter( strName, strParam );
	    if ( pParam != null ){
//	        return (CqVector3D[])( ((CqParameterTyped<CqVector3D, CqVector3D>)pParam ).pValue() );
	        
	       	Object[] pvalue = ((CqParameterTyped<CqVector3D, CqVector3D>)( pParam )).pValue();
	       	CqVector3D[] ret = new CqVector3D[pvalue.length];
	    	for( int i=0; i<pvalue.length; i++ ){
	    		ret[i] = (CqVector3D)pvalue[i];
	    	}
	    	return ret;
	    }
	    else
	        return ( null );
	}
	
	/** Get a color system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return CqColor pointer 0 if not found.
	 */
	public final	CqColor[]	GetColorOption( final String strName, final String strParam )
	{
	    final CqParameter pParam = pParameter( strName, strParam );
	    if ( pParam != null ){
//	        return (CqColor[])( ((CqParameterTyped<CqColor, CqColor>)pParam ).pValue() );
	        
	       	Object[] pvalue = ((CqParameterTyped<CqColor, CqColor>)( pParam )).pValue();
	       	CqColor[] ret = new CqColor[pvalue.length];
	    	for( int i=0; i<pvalue.length; i++ ){
	    		ret[i] = (CqColor)pvalue[i];
	    	}
	    	return ret;
	    }
	    else
	        return ( null );
	}
	
	/** Get a float system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return Float pointer 0 if not found.
	 */
	public p_float[] GetFloatOptionWrite( final String strName, final String strParam )
	{
	    CqParameter pParam = pParameterWrite( strName, strParam );
	    if ( pParam != null ){
//	        return (p_float[])( ((CqParameterTyped<p_float, Float>)pParam).pValue() );
	        
	    	Object[] pvalue = ((CqParameterTyped<p_float, p_float>)( pParam )).pValue();
	    	p_float[] ret = new p_float[pvalue.length];
	    	for( int i=0; i<pvalue.length; i++ ){
	    		ret[i] = (p_float)pvalue[i];
	    	}
	    	return ret;
	    }
	    else
	        return ( null );
	}
		
	/** Get an integer system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return Integer pointer 0 if not found.
	 */
	public p_int[]	GetIntegerOptionWrite( final String strName, final String strParam )
	{
	    CqParameter pParam = pParameterWrite( strName, strParam );
	    if ( pParam != null ){
//	        return (p_int[])( ((CqParameterTyped<p_int, Float>)pParam).pValue() );
	        
	    	Object[] pvalue = ((CqParameterTyped<p_int, p_float>)( pParam )).pValue();
	    	p_int[] ret = new p_int[pvalue.length];
	    	for( int i=0; i<pvalue.length; i++ ){
	    		ret[i] = (p_int)pvalue[i];
	    	}
	    	return ret;
	    }
	    else
	        return ( null );
	}
		
	/** Get a string system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return CqString pointer 0 if not found.
	 */
	public p_String[] GetStringOptionWrite( final String strName, final String strParam )
	{
	    CqParameter pParam = pParameterWrite( strName, strParam );
	    if ( pParam != null ){
//	        return (String[])( ((CqParameterTyped<String, String>)pParam).pValue() );
	        
	    	Object[] pvalue = ((CqParameterTyped<p_String, String>)( pParam )).pValue();
	    	p_String[] ret = new p_String[pvalue.length];
	    	for( int i=0; i<pvalue.length; i++ ){
	    		ret[i] = (p_String)pvalue[i];
	    	}
	    	return ret;
	    }
	    else
	        return ( null );
	}
		
	/** Get a point system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return CqVector3D pointer 0 if not found.
	 */
	public CqVector3D[] GetPointOptionWrite( final String strName, final String strParam )
	{
	    CqParameter pParam = pParameterWrite( strName, strParam );
	    if ( pParam != null ){
//	        return (CqVector3D[])( ((CqParameterTyped<CqVector3D, CqVector3D>)pParam).pValue() );
	        
	        Object[] pvalue = ((CqParameterTyped<CqVector3D, CqVector3D>)( pParam )).pValue();
	        CqVector3D[] ret = new CqVector3D[pvalue.length];
	    	for( int i=0; i<pvalue.length; i++ ){
	    		ret[i] = (CqVector3D)pvalue[i];
	    	}
	    	return ret;
	    }
	    else
	        return ( null );
	}
	
	/** Get a color system option parameter.
	 * \param strName The name of the attribute.
	 * \param strParam The name of the paramter on the attribute.
	 * \return CqColor pointer 0 if not found.
	 */
	public CqColor[] GetColorOptionWrite( final String strName, final String strParam )
	{
	    CqParameter pParam = pParameterWrite( strName, strParam );
	    if ( pParam != null ){
//	        return (CqColor[])( ((CqParameterTyped<CqColor, CqColor>)pParam).pValue() );
	        
	        Object[] pvalue = ((CqParameterTyped<CqColor, CqColor>)( pParam )).pValue();
	        CqColor[] ret = new CqColor[pvalue.length];
	    	for( int i=0; i<pvalue.length; i++ ){
	    		ret[i] = (CqColor)pvalue[i];
	    	}
	    	return ret;
	    }
	    else
	        return ( null );
	}
	
	
	/** Initialise the matrices for this camera according to the status of the camera attributes.
	 */
	public void InitialiseCamera()
	{
		// Setup the Normalising and projection matrices according to the projection
		// type
		CqMatrix	matCameraToScreen	= new CqMatrix();
//		CqMatrix	matScreenToCamera	= new CqMatrix();
		CqMatrix	matScreenToNDC		= new CqMatrix();
		CqMatrix	matNDCToRaster		= new CqMatrix();
		int proj = GetIntegerOptionIndex( "System", "Projection" , 0 ).value;
		switch ( proj )
		{
		case	ProjectionOrthographic:
		{
			// Define a matrix to convert from right top left handed coordinate systems.
			CqMatrix Trl = new CqMatrix( 1, 1, -1 );
			
			float l = GetFloatOptionIndex( "System", "ScreenWindow" , 0 ).value;
			float r = GetFloatOptionIndex( "System", "ScreenWindow" , 1 ).value;
			float t = GetFloatOptionIndex( "System", "ScreenWindow" , 2 ).value;
			float b = GetFloatOptionIndex( "System", "ScreenWindow" , 3 ).value;
			float n = GetFloatOptionIndex( "System", "Clipping" , 0 ).value;
			float f = GetFloatOptionIndex( "System", "Clipping" , 1 ).value;
			
			matCameraToScreen.Identity();
			matCameraToScreen.SetfIdentity( false );

			matCameraToScreen.m_aaElement[0][0] = 2.0f / ( r - l );
			matCameraToScreen.m_aaElement[3][0] = -( r + l ) / ( r - l );
			matCameraToScreen.m_aaElement[1][1] = 2.0f / ( t - b );
			matCameraToScreen.m_aaElement[3][1] = -( t + b ) / ( t - b );
			matCameraToScreen.m_aaElement[2][2] = -2.0f / ( f - n );
			matCameraToScreen.m_aaElement[3][2] = -( f + n ) / ( f - n );
			matCameraToScreen.m_aaElement[2][3] = 0;
			matCameraToScreen.m_aaElement[3][3] = 1;
			
			// Combine with the right to left matrix.
			matCameraToScreen.assignMul( Trl );
			
			// Set up the screen to frame matrix
			float	FrameX = ( GetFloatOptionIndex( "System", "FrameAspectRatio" , 0 ).value >= 1.0 ) ? GetIntegerOptionIndex( "System", "Resolution" , 0 ).value :
				( GetIntegerOptionIndex( "System", "Resolution" , 1 ).value * GetFloatOptionIndex( "System", "FrameAspectRatio" , 0 ).value ) / GetFloatOptionIndex( "System", "PixelAspectRatio" , 0 ).value;
			float	FrameY = ( GetFloatOptionIndex( "System", "FrameAspectRatio" , 0 ).value < 1.0 ) ? GetIntegerOptionIndex( "System", "Resolution" , 1 ).value :
				( GetIntegerOptionIndex( "System", "Resolution" , 0 ).value * GetFloatOptionIndex( "System", "PixelAspectRatio" , 0 ).value ) / GetFloatOptionIndex( "System", "FrameAspectRatio" , 0 ).value;
			matScreenToNDC.Identity();
			matNDCToRaster.Identity();
			// Translate from -1,-1-.1,1 to 0,0-.2,2
			CqMatrix	T = new CqMatrix();
			T.Translate( 1, 1, 0 );
			// Scale by 0.5 (0,0 -. 1,1) NDC
			CqMatrix	S	= new CqMatrix( 0.5f, 0.5f, 0f );
			CqMatrix	S2	= new CqMatrix( FrameX, FrameY, 0 );
			// Invert y to fit top down format
			CqMatrix	S3	= new CqMatrix( 1, -1, 1 );
			matScreenToNDC = S.multiply( T ).multiply( S3 );
			matNDCToRaster = S2;
			
			break;
		}
		
		case	ProjectionPerspective:
		{
			float fov = GetFloatOptionIndex( "System", "Clipping" , 0 ).value * (float)( Math.tan( Math.toRadians( GetFloatOptionIndex( "System", "FOV" , 0 ).value / 2.0f ) ) );
			float l = GetFloatOptionIndex( "System", "ScreenWindow" , 0 ).value * fov;
			float r = GetFloatOptionIndex( "System", "ScreenWindow" , 1 ).value * fov;
			float t = GetFloatOptionIndex( "System", "ScreenWindow" , 2 ).value * fov;
			float b = GetFloatOptionIndex( "System", "ScreenWindow" , 3 ).value * fov;
			float n = GetFloatOptionIndex( "System", "Clipping" , 0 ).value;
			float f = GetFloatOptionIndex( "System", "Clipping" , 1 ).value;
			
			matCameraToScreen.Identity();
			matCameraToScreen.SetfIdentity( false );
			matCameraToScreen.m_aaElement[0][0] = ( 2.0f * n ) / ( r - l );
			matCameraToScreen.m_aaElement[2][0] = ( r + l ) / ( r - l );
			matCameraToScreen.m_aaElement[1][1] = ( 2.0f * n ) / ( t - b );
			matCameraToScreen.m_aaElement[2][1] = ( t + b ) / ( t - b );
			float a = f / ( f - n );
			//			matCameraToScreen.SetElement(2,2,-((f+n)/(f-n)));
			matCameraToScreen.m_aaElement[2][2] = a;
			//			matCameraToScreen.SetElement(3,2,-((2.0f*f*n)/(f-n)));
			matCameraToScreen.m_aaElement[3][2] = -n * a;
			matCameraToScreen.m_aaElement[2][3] = 1;
			matCameraToScreen.m_aaElement[3][3] = 0;
			
			// Set up the screen to frame matrix
			float	FrameX = ( GetFloatOptionIndex( "System", "FrameAspectRatio" , 0 ).value >= 1.0 ) ? GetIntegerOptionIndex( "System", "Resolution" , 0 ).value :
				( GetIntegerOptionIndex( "System", "Resolution" , 1 ).value * GetFloatOptionIndex( "System", "FrameAspectRatio" , 0 ).value ) / GetFloatOptionIndex( "System", "PixelAspectRatio" , 0 ).value;
			float	FrameY = ( GetFloatOptionIndex( "System", "FrameAspectRatio" , 0 ).value < 1.0 ) ? GetIntegerOptionIndex( "System", "Resolution" , 1 ).value :
				( GetIntegerOptionIndex( "System", "Resolution" , 0 ).value * GetFloatOptionIndex( "System", "PixelAspectRatio" , 0 ).value ) / GetFloatOptionIndex( "System", "FrameAspectRatio" , 0 ).value;
			
			matScreenToNDC.Identity();
			matNDCToRaster.Identity();
			// Translate from -1,-1-.1,1 to 0,0-.2,2
			CqMatrix	T	= new CqMatrix();
			T.Translate( 1.0f, 1.0f, 0.0f );
			// Scale by 0.5 (0,0 -. 1,1) NDC
			CqMatrix	S	= new CqMatrix( 0.5f, 0.5f, 1.0f );
			CqMatrix	S2	= new CqMatrix( FrameX, FrameY, 1.0f );
			// Invert y to fit top down format
			CqMatrix	S3	= new CqMatrix( 1.0f, -1.0f, 1.0f );
			matScreenToNDC = S.multiply( T ).multiply( S3 );
			matNDCToRaster = S2;
			
			break;
		}
		}
		CqMatrix matWorldToCamera	= new CqMatrix( QGetRenderContext().matSpaceToSpace( "world", "camera", new CqMatrix(), new CqMatrix(), QGetRenderContext().Time() ) );
		QGetRenderContext().SetmatScreen( matCameraToScreen.multiply( matWorldToCamera ) );
		QGetRenderContext().SetmatNDC( matScreenToNDC.multiply( matCameraToScreen.multiply( matWorldToCamera ) ) );
		QGetRenderContext().SetmatRaster( matNDCToRaster.multiply( matScreenToNDC.multiply( matCameraToScreen.multiply( matWorldToCamera ) ) ) );
		
//		CqMatrix matWorldToScreen = matCameraToScreen.multiply( matWorldToCamera );
		
		CqMatrix dofm = QGetRenderContext().matVSpaceToSpace( "camera", "raster", new CqMatrix(), new CqMatrix(), QGetRenderContext().Time() );
		
		CqVector3D	dofe	= new CqVector3D( 1, 1, -1 );
		CqVector3D	dofc	= new CqVector3D( 0, 0, -1 );
		
		dofe = dofm.multiply( dofe );
		dofc = dofm.multiply( dofc );
		
		QGetRenderContext().SetDepthOfFieldScale( Math.abs(dofe.x-dofc.x), Math.abs(dofe.y-dofc.y) );
		
	}
	
	
	/** Indicate that the RiFormat function has been called.
	 * This and other similar are used to overcome the problem with multiple ways to specify frame dimensions.
	 */
	public void CallFormat()
	{
		m_bFormatCalled = true;
	}
	/** Indicate that the RiScreenWindow function has been called.
	 * This and other similar are used to overcome the problem with multiple ways to specify frame dimensions.
	 */
	public void CallScreenWindow()
	{
		m_bScreenWindowCalled = true;
	}
	/** Indicate that the RiFrameAspectRatio function has been called.
	 * This and other similar are used to overcome the problem with multiple ways to specify frame dimensions.
	 */
	public void CallFrameAspectRatio()
	{
		m_bFrameAspectRatioCalled = true;
	}
	/** Determine if the RiFormat function has been called.
	 * This and other similar are used to overcome the problem with multiple ways to specify frame dimensions.
	 */
	public boolean FormatCalled()
	{
		return ( m_bFormatCalled );
	}
	/** Determine if the RiScreenWindow function has been called.
	 * This and other similar are used to overcome the problem with multiple ways to specify frame dimensions.
	 */
	public boolean ScreenWindowCalled()
	{
		return ( m_bScreenWindowCalled );
	}
	/** Determine if the RiFrameAspectRatio function has been called.
	 * This and other similar are used to overcome the problem with multiple ways to specify frame dimensions.
	 */
	public boolean FrameAspectRatioCalled()
	{
		return ( m_bFrameAspectRatioCalled );
	}
	
	/** Get a pointer to the pixel filter function.
	 */
	public Method funcFilter()
	{
		return ( m_funcFilter );
	}
	/** Set the pixel filter function to use.
	 * @param fValue A pointer to a function which follows the RtFilterFunc convention.
	 */
	public void SetfuncFilter( final Method fValue )
	{
		m_funcFilter = fValue;
	}
	
	
	public void LoadImager( final String strName )
	{
		DeleteImager();
		
		IqShader pShader = QGetRenderContext().CreateShader( strName, new EqShaderType(Type_Imager) );
		
		if ( pShader==null ) return ;
		
		m_pshadImager = new CqImagersource( pShader, true );
		m_pshadImager.pShader().PrepareDefArgs();
		
	}
	public void DeleteImager()
	{
		
		if ( m_pshadImager != null )
		{
//			delete m_pshadImager;
			m_pshadImager = null;
		}
	}
	public void SetValueImager( String token, Object value )
	{
		if ( m_pshadImager != null )
		{
			SqParameterDeclaration Decl;
			try
			{
				Decl = QGetRenderContext().FindParameterDecl( token );
			}
			catch( XqException e )
			{
				HimawariLogger logger = HimawariLogger.getLogger();
				logger.error( e.strReason() + "\n" );
				return;
			}
			m_pshadImager.pShader().SetArgument( Decl.m_strName, Decl.m_Type, Decl.m_strSpace, value );
		}
	}
	
	/** Get a pointer to the imager shader.
	 */
	public CqImagersource	pshadImager()
	{
		return ( m_pshadImager );
	}
	
	public void InitialiseColorImager( IqBucket pBucket )
	{
		// Each time with finished up a bucket; we will execute the imager shader
		// on the gridsize about the same size as the bucket
		if ( m_pshadImager != null )
		{
			m_pshadImager.Initialise( pBucket );
		}
	}
	
	/** Get a color from the imager shader.
	 * @param x The X in raster coordinate system.
	 * @param y The Y in raster coordiante system.
	 * @return Color  Black if not found.
	 * Right now it is returning the current background colour if found
	 */
	public CqColor GetColorImager( float x, float y )
	{
		CqColor result = new CqColor( 0, 0, 0 );
		
		if ( m_pshadImager != null )
		{
			// get the color from the current imager than
			result = m_pshadImager.Color( x, y );
		}
		
		return result;
	}
	
	/** Get a color from the imager shader.
	 * @param x The X in raster coordinate system.
	 * @param y The Y in raster coordiante system.
	 * @return Color  Black if not found.
	 * Right now it is returning the current background colour if found
	 */
	public float GetAlphaImager( float x, float y )
	{
		float result = 1.0f;
		
		if ( m_pshadImager != null )
		{
			// get the color from the current imager than
			result = m_pshadImager.Alpha( x, y );
		}
		
		return result;
	}
	
	/** Get an opacity from the imager shader.
	 * @param x The X in raster coordinate system.
	 * @param y The Y in raster coordiante system.
	 * @return Color  White right now
	 * Right now it is returning the current background colour if found
	 */
	
	public CqColor GetOpacityImager( float x, float y )
	{
		CqColor result = new CqColor( gColWhite );
		
		if ( m_pshadImager != null )
		{
			// get the opacity from the current imager than
			result = m_pshadImager.Opacity( x, y );
		}
		
		
		return result;
	}
	
	//************************************************************************//
	//*** Zq̃I[o[[h
	//************************************************************************//
	
	public CqOptions assignment( CqOptions From )
	{
		m_bFrameAspectRatioCalled = From.m_bFrameAspectRatioCalled;
		m_bScreenWindowCalled = From.m_bScreenWindowCalled;
		m_bFormatCalled = From.m_bFormatCalled;
		
		m_funcFilter = From.m_funcFilter;
		m_pshadImager = From.m_pshadImager;
		
		DeleteImager();
		
		// Copy the system options.
		m_aOptions.setSize( From.m_aOptions.size() );
		int i = From.m_aOptions.size();
		while ( i-- > 0 )
		{
			m_aOptions.set( i, From.m_aOptions.get( i ) );
		}
		
		return ( this );
	}
	
}
