// Aqsis
// Copyright (c) 1997 - 2001, Paul C. Gregory
//
// Contact: pgregory@aqsis.com
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/**
 * Copyright (C) 2006-2007  NTT DATA CORPORATION
 * 
 * Version: 1.0.0 2007/04/01
 *  
 */
package net.cellcomputing.himawari.library;

import net.cellcomputing.himawari.accessory.STLVector;
import net.cellcomputing.himawari.library.types.CqMatrix;
import net.cellcomputing.himawari.library.types.CqVector3D;
import static java.lang.Math.*;

/**
 * Cylinder quadric GPrim.
 * 2Ȗʁ@F@~
 * 
 * @author NTT DATA Corporation
 */
public strictfp class CqCylinder extends CqQuadric {

	private float	m_Radius;		///< Radius
	private float	m_ZMin;			///< Min value on zaxis.
	private float	m_ZMax;			///< Max value on z axis.
	private float	m_ThetaMin;		///< Min angle about z axis.
	private float	m_ThetaMax;		///< Max angle about z axis.


	//*************************************************
	//**  RXgN^`
	//*************************************************
	
	/**
	 * RXgN^B<BR>l^ȂꍇB
	 */
	public CqCylinder()
	{
		this( 1.0f, -1.0f, 1.0f, 0.0f, 360.0f );
	}
	
	/**
	 * RXgN^B<BR>l^ꂽꍇB
	 * 
	 * @param radius	a
	 */
	public CqCylinder( float radius )
	{
		this( radius, -1.0f, 1.0f, 0.0f, 360.0f );
	}
	
	/**
	 * RXgN^B<BR>l^ꂽꍇB
	 * 
	 * @param radius	a
	 * @param zmin		~̍ŏ
	 */
	public CqCylinder( float radius, float zmin )
	{
		this( radius, zmin, 1.0f, 0.0f, 360.0f );
	}
	
	/**
	 * RXgN^B<BR>lO^ꂽꍇB
	 * 
	 * @param radius	a
	 * @param zmin		~̍ő卂
	 * @param zmax		~̍ŏ
	 */
	public CqCylinder( float radius, float zmin, float zmax )
	{
		this( radius, zmin, zmax, 0.0f, 360.0f );
	}
	
	/**
	 * RXgN^B<BR>ll^ꂽꍇB
	 * 
	 * @param radius	a
	 * @param zmin		~̍ő卂
	 * @param zmax		~̍ŏ
	 * @param thetamin	ŏXC[vpx
	 */
	public CqCylinder(  float radius, float zmin, float zmax, float thetamin )
	{
		this( radius, zmin, zmax, thetamin, 360.0f );
	}
	
	/**
	 * RXgN^B<BR>lSė^ꂽꍇB
	 * 
	 * @param radius	a
	 * @param zmin		~̍ő卂
	 * @param zmax		~̍ŏ
	 * @param thetamin	ŏXC[vpx
	 * @param thetamax	őXC[vpx
	 */
	public CqCylinder( float radius, float zmin, float zmax, float thetamin, float thetamax )
	{
		m_Radius = radius;
		m_ZMin = zmin;
		m_ZMax = zmax;
		m_ThetaMin = thetamin;
		m_ThetaMax = thetamax;
	}
	
	/**
	 * Rs[RXgN^
	 * 
	 * @param From@Rs[
	 */
	public CqCylinder( final CqCylinder From )
	{
		this.assignment( From );
	}
	
	/* (non-Javadoc)
	 * @see net.cellcomputing.himawari.library.CqBasicSurface#destruct()
	 */
	public 	void destruct()
	{}
	
	
	//*************************************************
	//**  \bh`
	//*************************************************
	
	/**
	 * Get the geometric bound of this GPrim.
	 * @see net.cellcomputing.himawari.library.IqSurface#Bound()
	 */
	public 	CqBound	Bound()
	{
		STLVector<CqVector3D> curve = new STLVector<CqVector3D>( CqVector3D.class );
		CqVector3D vA = new CqVector3D( m_Radius, 0, m_ZMin );
		CqVector3D vB = new CqVector3D( m_Radius, 0, m_ZMax );
		CqVector3D vC = new CqVector3D( 0, 0, 0 );
		CqVector3D vD = new CqVector3D( 0, 0, 1 );
		
		curve.add( vA );
		curve.add( vB );
		CqMatrix matRot = new CqMatrix( (float)toRadians( m_ThetaMin ), vD );
		
		for ( CqVector3D i : curve )
			i.assignment( matRot.multiply( i ));
		
		CqBound B = new CqBound( RevolveForBound( curve, vC, vD, (float)toRadians( m_ThetaMax - m_ThetaMin )));
		B.Transform( m_matTx );
		
		return ( AdjustBoundForTransformationMotion( B ) );
	}
	
	
	/**  
	 * Get a point on the surface indexed by the surface paramters passed.
	 * @param u Float surface paramter in u.
	 * @param v Float surface paramter in v.
	 * @see net.cellcomputing.himawari.library.CqQuadric#DicePoint(int, int)
	 */
	public 	CqVector3D	DicePoint( int u, int v )
	{
		float theta = (float)toRadians( m_ThetaMin + (( m_ThetaMax - m_ThetaMin )*(float)u ) / (float)m_uDiceSize );
		
		float vz = m_ZMin + ((float)v * ( m_ZMax - m_ZMin )) / (float)m_vDiceSize;
		return ( new CqVector3D( (float)( m_Radius * cos( theta )), (float)( m_Radius * sin( theta )), vz ) );
	}
	
	/** 
	 * Get a point on the surface indexed by the surface paramters passed.
	 * @param u Float surface paramter in u.
	 * @param v Float surface paramter in v.
	 * @param Normal Storage for the surface normal at that point.
	 */
	public 	CqVector3D	DicePoint( int u, int v, CqVector3D Normal )
	{
		CqVector3D p = new CqVector3D( DicePoint( u, v ) );
		Normal.assignment( p ) ; 
		Normal.z( 0 );
		Normal.Unit();
		
		return ( p );
	}
	
	
	public  boolean	CanGenerateNormals()
	{
		return true;
	}
	
	
	/**
	 * Split this GPrim into a NURBS surface. Temp implementation, should split into smalled quadrics.
	 * @see net.cellcomputing.himawari.library.IqSurface#PreSubdivide(net.cellcomputing.himawari.accessory.STLVector, boolean)
	 */
	public 	int PreSubdivide( STLVector<CqBasicSurface> aSplits, boolean u )
	{
	    float zcent = (float)(( m_ZMin + m_ZMax ) * 0.5 );
	    float arccent = (float)(( m_ThetaMin + m_ThetaMax ) * 0.5 );

	    CqCylinder pNew1 = new CqCylinder( this );
	    CqCylinder pNew2 = new CqCylinder( this );

	    if ( u )
	    {
	        pNew1.m_ThetaMax = arccent;
	        pNew2.m_ThetaMin = arccent;
	    }
	    else
	    {
	        pNew1.m_ZMax = zcent;
	        pNew2.m_ZMin = zcent;
	    }

	    aSplits.add( pNew1 );
	    aSplits.add( pNew2 );
	    
	    return ( 2 );
	}
	
	/**
	 * Zq̃I[o[[h
	 * 
	 * @param From	
	 * @return	
	 */
	public  CqCylinder assignment( final CqCylinder From )
	{
		super.assignment( From );
		m_Radius = From.m_Radius;
		m_ZMin = From.m_ZMin;
		m_ZMax = From.m_ZMax;
		m_ThetaMin = From.m_ThetaMin;
		m_ThetaMax = From.m_ThetaMax;
		
		return ( this );
	}
}

