/*
 * segatex SELinux tool.
 *
 * Copyright (C) 2007-2013 Shintaro Fujiwara 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * */

#include <qapplication.h>
#include <qfile.h>
#include <q3filedialog.h>
#include <qmessagebox.h>
#include <qpixmap.h>
//Added by qt3to4:
#include <Q3TextStream>
#include <QCloseEvent>

#include "editor.h"

Editor::Editor(QWidget *parent, const char *name)
	: Q3TextEdit(parent, name)
{
	QWidget::setWindowFlags(Qt::WindowFlags());
	QPixmap myIcon = QPixmap(":/images/document.png"); 
	this->setIcon(myIcon);
	isUntitled = true;
	fileFilters = tr("All files (*)\n"
		"File Context files (*.fc)\n"
		"Interface files (*.if)\n"
		"policy_module files (*.pp)\n"
		"text files (*.txt)\n"
		"raw files (*raw)\n"
		"csv files (*.csv)\n"
		"All files (*)\n"
		"Type Enforcement files (*.te)");
}

void Editor::newFile()
{
	static int documentNumber = 1;

	curFile = tr("document%1.te").arg(documentNumber);
	setCaption(curFile);
	isUntitled = true;
	++documentNumber;
}

bool Editor::open()
{
	QString fileName =
	    Q3FileDialog::getOpenFileName("/usr/share/segatex", fileFilters, this);
	if (fileName.isEmpty())
	return false;

	return openFile(fileName);
}

bool Editor::openFile(const QString &fileName)
{
	bool ok = readFile(fileName);
	if (ok) {
	setCurrentFile(fileName);
	emit message(tr("File loaded"), 2000);
	} else {
	emit message(tr("Loading canceled"), 2000);
	}
	return ok;
}

bool Editor::save()
{
	if (isUntitled) {
		return saveAs();
	} else {
		saveFile(curFile);
		return true;
	}
}

bool Editor::remove()
{
	removeFile(curFile);
	return true;
}
bool Editor::saveAs()
{
	curFile = "/usr/share/segatex/" + curFile; 
	QString fileName =
	    Q3FileDialog::getSaveFileName(curFile, fileFilters, this);
	if (fileName.isEmpty())
	return false;

	if (QFile::exists(fileName)) {
	int ret = QMessageBox::warning(this, tr("Editor"),
		     tr("File %1 already exists.\n"
			"Do you want to overwrite it?")
		     .arg(QDir::convertSeparators(fileName)),
		     QMessageBox::Yes | QMessageBox::Default,
		     QMessageBox::No | QMessageBox::Escape);
	if (ret == QMessageBox::No)
	    return true;
	}
	if (!fileName.isEmpty())
	saveFile(fileName);
	return true;
}

void Editor::closeEvent(QCloseEvent *event)
{
	if (maybeSave())
	event->accept();
	else
	event->ignore();
}

bool Editor::maybeSave()
{
	if (isModified()) {
	int ret = QMessageBox::warning(this, tr("Editor"),
		     tr("File %1 has been modified.\n"
			"Do you want to save your changes?")
		     .arg(strippedName(curFile)),
		     QMessageBox::Yes | QMessageBox::Default,
		     QMessageBox::No,
		     QMessageBox::Cancel | QMessageBox::Escape);
	if (ret == QMessageBox::Yes)
	    return save();
	else if (ret == QMessageBox::Cancel)
	    return false;
	}
	return true;
}

void Editor::saveFile(const QString &fileName)
{
	if (writeFile(fileName)) {
		setCurrentFile(fileName);
		emit message(tr("File saved"), 2000);
	} else {
		emit message(tr("Saving canceled"), 2000);
	}
}

void Editor::setCurrentFile(const QString &fileName)
{
	curFile = fileName;
	setCaption(strippedName(curFile));
	isUntitled = false;
	setModified(false);
}

QSize Editor::sizeHint() const
{
	return QSize(72 * fontMetrics().width('x'),
		 25 * fontMetrics().lineSpacing());
}

bool Editor::readFile(const QString &fileName)
{
	QFile file(fileName);
	if (!file.open(QIODevice::ReadOnly)) {
	QMessageBox::warning(this, tr("Editor"),
			     tr("Cannot read file %1:\n%2.")
			     .arg(file.name())
			     .arg(file.errorString()));
	return false;
	}

	Q3TextStream in(&file);
	QApplication::setOverrideCursor(Qt::waitCursor);
	setText(in.read());
	QApplication::restoreOverrideCursor();
	return true;
}

bool Editor::writeFile(const QString &fileName)
{
	QFile file(fileName);
	if (!file.open(QIODevice::WriteOnly)) {
		QMessageBox::warning(this, tr("Editor"),
		tr("Cannot write file %1:\n%2.")
			.arg(file.name())
			.arg(file.errorString()));
		return false;
	}

	Q3TextStream out(&file);
	QApplication::setOverrideCursor(Qt::waitCursor);
	out << text();
	QApplication::restoreOverrideCursor();
	return true;
}

bool Editor::removeFile(const QString &fileName)
{
	QFile file(fileName);
	setCurrentFile(fileName);
	file.setFileName("filename");
	if (!file.open(QIODevice::WriteOnly)) {
		QMessageBox::warning(this, tr("Editor"),
		tr("file does not exist%1:\n%2.")
			.arg(file.name())
			.arg(file.errorString()));
		return false;
	}

	file.remove();
	QApplication::restoreOverrideCursor();
	return true;
}

QString Editor::strippedName(const QString &fullFileName)
{
	return QFileInfo(fullFileName).fileName();
}
