package org.seasar.framework.util;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import org.seasar.framework.exception.ClassNotFoundRuntimeException;
import org.seasar.framework.exception.IllegalAccessRuntimeException;
import org.seasar.framework.exception.InstantiationRuntimeException;
import org.seasar.framework.exception.NoSuchConstructorRuntimeException;
import org.seasar.framework.exception.NoSuchMethodRuntimeException;

/**
 * @author higa
 *
 */
public final class ClassUtil {

	private static Map wrapperToPrimitiveMap_ = new HashMap();

	static {
		wrapperToPrimitiveMap_.put(Character.class, Character.TYPE);
		wrapperToPrimitiveMap_.put(Short.class, Short.TYPE);
		wrapperToPrimitiveMap_.put(Integer.class, Integer.TYPE);
		wrapperToPrimitiveMap_.put(Long.class, Long.TYPE);
		wrapperToPrimitiveMap_.put(Double.class, Double.TYPE);
		wrapperToPrimitiveMap_.put(Float.class, Float.TYPE);
		wrapperToPrimitiveMap_.put(Boolean.class, Boolean.TYPE);
	}

	/**
	 * 
	 */
	private ClassUtil() {
	}

	public static Class forName(String className)
		throws ClassNotFoundRuntimeException {

		try {
			return Class.forName(className);
		} catch (ClassNotFoundException ex) {
			throw new ClassNotFoundRuntimeException(ex);
		}
	}

	public static Object newInstance(Class clazz)
		throws InstantiationRuntimeException, IllegalAccessRuntimeException {

		try {
			return clazz.newInstance();
		} catch (InstantiationException ex) {
			throw new InstantiationRuntimeException(clazz, ex);
		} catch (IllegalAccessException ex) {
			throw new IllegalAccessRuntimeException(clazz, ex);
		}
	}

	public static Object newInstance(String className)
		throws
			ClassNotFoundRuntimeException,
			InstantiationRuntimeException,
			IllegalAccessRuntimeException {

		return newInstance(forName(className));
	}

	public static boolean isAssignableFrom(Class toClass, Class fromClass) {
		if (toClass == Object.class && !fromClass.isPrimitive()) {
			return true;
		}
		if (toClass.isPrimitive()) {
			fromClass = getPrimitiveClassIfWrapper(fromClass);
		}
		return toClass.isAssignableFrom(fromClass);
	}

	public static Class getPrimitiveClass(Class clazz) {
		return (Class) wrapperToPrimitiveMap_.get(clazz);
	}

	public static Class getPrimitiveClassIfWrapper(Class clazz) {
		Class ret = getPrimitiveClass(clazz);
		if (ret != null) {
			return ret;
		} else {
			return clazz;
		}
	}

	public static Constructor getConstructor(Class clazz, Class[] argTypes) {
		try {
			return clazz.getConstructor(argTypes);
		} catch (NoSuchMethodException ex) {
			throw new NoSuchConstructorRuntimeException(clazz, argTypes, ex);
		}
	}

	public static Method getMethod(
		Class clazz,
		String methodName,
		Class[] argTypes) {

		try {
			return clazz.getMethod(methodName, argTypes);
		} catch (NoSuchMethodException ex) {
			throw new NoSuchMethodRuntimeException(
				clazz,
				methodName,
				argTypes,
				ex);
		}
	}

	public static String getShortClassName(Class clazz) {
		String s = clazz.getName();
		int i = s.lastIndexOf('.');
		if (i > 0) {
			return s.substring(i + 1);
		} else {
			return s;
		}
	}
}
