package org.seasar.framework.sel.parser;

import java.util.ArrayList;
import java.util.List;

import org.seasar.framework.exception.lang.ClassNotFoundRuntimeException;
import org.seasar.framework.reflect.ClassUtil;
import org.seasar.framework.sel.BoolExpression;
import org.seasar.framework.sel.Expression;
import org.seasar.framework.sel.UnexpectedTokenRuntimeException;
import org.seasar.framework.sel.boolexps.AndExp;
import org.seasar.framework.sel.boolexps.BetweenExp;
import org.seasar.framework.sel.boolexps.BoolExp;
import org.seasar.framework.sel.boolexps.EqualExp;
import org.seasar.framework.sel.boolexps.GreaterEqualExp;
import org.seasar.framework.sel.boolexps.GreaterThanExp;
import org.seasar.framework.sel.boolexps.InExp;
import org.seasar.framework.sel.boolexps.IsFalseExp;
import org.seasar.framework.sel.boolexps.IsNullExp;
import org.seasar.framework.sel.boolexps.IsTrueExp;
import org.seasar.framework.sel.boolexps.LessEqualExp;
import org.seasar.framework.sel.boolexps.LessThanExp;
import org.seasar.framework.sel.boolexps.LikeExp;
import org.seasar.framework.sel.boolexps.NotBetweenExp;
import org.seasar.framework.sel.boolexps.NotEqualExp;
import org.seasar.framework.sel.boolexps.NotExp;
import org.seasar.framework.sel.boolexps.OrExp;
import org.seasar.framework.sel.exps.AddExp;
import org.seasar.framework.sel.exps.BooleanExp;
import org.seasar.framework.sel.exps.ConcatenateExp;
import org.seasar.framework.sel.exps.DivideExp;
import org.seasar.framework.sel.exps.GetArrayExp;
import org.seasar.framework.sel.exps.GetPropertyExp;
import org.seasar.framework.sel.exps.GetVariableExp;
import org.seasar.framework.sel.exps.MethodExp;
import org.seasar.framework.sel.exps.ModExp;
import org.seasar.framework.sel.exps.MultiplyExp;
import org.seasar.framework.sel.exps.NewArrayExp;
import org.seasar.framework.sel.exps.NewExp;
import org.seasar.framework.sel.exps.NowExp;
import org.seasar.framework.sel.exps.NullExp;
import org.seasar.framework.sel.exps.ObjectExp;
import org.seasar.framework.sel.exps.StaticMethodExp;
import org.seasar.framework.sel.exps.SubtractExp;
import org.seasar.framework.sel.exps.ToStringExp;
import org.seasar.framework.sel.exps.ToTimestampExp;
import org.seasar.framework.sel.tokenizer.SelTokenizer;
import org.seasar.framework.sel.util.ExpressionUtil;
import org.seasar.framework.string.StringUtil;

public final class SelParser {

	private SelTokenizer tokenizer_;
	private int token_;

	public SelParser(String s) {
		tokenizer_ = new SelTokenizer(s);
		nextToken();
	}

	public final BoolExpression parseBoolExpression() {
		BoolExpression boolExp = getOrExp();
		expected(SelTokenizer.EOF);
		return boolExp;
	}

	public final Expression parseExpression() {
		Expression exp = getAddSubExp();
		expected(SelTokenizer.EOF);
		return exp;
	}

	private BoolExpression getOrExp() {
		BoolExpression left = getAndExp();
		while (token_ == SelTokenizer.OR) {
			nextToken();
			left = new OrExp(left, getAndExp());
		}
		return left;
	}

	private BoolExpression getAndExp() {
		BoolExpression left = getNotExp();
		while (token_ == SelTokenizer.AND) {
			nextToken();
			left = new AndExp(left, getNotExp());
		}
		return left;
	}

	private BoolExpression getNotExp() {
		switch (token_) {
			case SelTokenizer.NOT :
				nextToken();
				BoolExpression e = getSimpleBoolExp();
				return new NotExp(e);
			default :
				return getSimpleBoolExp();
		}
	}

	private BoolExpression getSimpleBoolExp() {
		switch (token_) {
			case SelTokenizer.LEFT_PAREN :
				nextToken();
				BoolExpression e = getOrExp();
				expected(SelTokenizer.RIGHT_PAREN);
				return e;
			case SelTokenizer.TRUE :
				nextToken();
				if (token_ == SelTokenizer.IS) {
					nextToken();
					switch (token_) {
						case SelTokenizer.TRUE :
							nextToken();
							return BoolExp.TRUE;
						case SelTokenizer.FALSE :
							nextToken();
							return BoolExp.FALSE;
					}
				} else {
					return BoolExp.TRUE;
				}
			case SelTokenizer.FALSE :
				nextToken();
				if (token_ == SelTokenizer.IS) {
					nextToken();
					switch (token_) {
						case SelTokenizer.TRUE :
							nextToken();
							return BoolExp.FALSE;
						case SelTokenizer.FALSE :
							nextToken();
							return BoolExp.TRUE;
					}
				} else {
					return BoolExp.FALSE;
				}
			default :
				Expression targetExp = getAddSubExp();
				switch (token_) {
					case SelTokenizer.EQUAL :
						nextToken();
						return new EqualExp(targetExp, getAddSubExp());
					case SelTokenizer.NOT_EQUAL :
						nextToken();
						return new NotEqualExp(targetExp, getAddSubExp());
					case SelTokenizer.GREATER_EQUAL :
						nextToken();
						return new GreaterEqualExp(targetExp, getAddSubExp());
					case SelTokenizer.GREATER_THAN :
						nextToken();
						return new GreaterThanExp(targetExp, getAddSubExp());
					case SelTokenizer.LESS_EQUAL :
						nextToken();
						return new LessEqualExp(targetExp, getAddSubExp());
					case SelTokenizer.LESS_THAN :
						nextToken();
						return new LessThanExp(targetExp, getAddSubExp());
					case SelTokenizer.IN :
						return getInExp(targetExp);
					case SelTokenizer.BETWEEN :
						return getBetweenExp(targetExp);
					case SelTokenizer.IS :
						nextToken();
						switch (token_) {
							case SelTokenizer.NULL :
								nextToken();
								return new IsNullExp(targetExp);
							case SelTokenizer.NOT :
								nextToken();
								expected(SelTokenizer.NULL);
								return new NotExp(new IsNullExp(targetExp));
							case SelTokenizer.TRUE :
								nextToken();
								return new IsTrueExp(targetExp);
							case SelTokenizer.FALSE :
								nextToken();
								return new IsFalseExp(targetExp);
						}
					case SelTokenizer.NOT :
						nextToken();
						switch (token_) {
							case SelTokenizer.LIKE :
								nextToken();
								return new NotExp(
									new LikeExp(targetExp, getAddSubExp()));
							case SelTokenizer.BETWEEN :
								return getNotBetweenExp(targetExp);
							case SelTokenizer.IN :
								return new NotExp(getInExp(targetExp));
						}
					case SelTokenizer.LIKE :
						nextToken();
						return new LikeExp(targetExp, getAddSubExp());
				}
		}
		throw new UnexpectedTokenRuntimeException(
			tokenizer_.getReadString(),
			SelTokenizer.getTokenName(token_));
	}

	private BoolExpression getInExp(Expression targetExp) {
		nextToken();
		expected(SelTokenizer.LEFT_PAREN);
		List inExpList = new ArrayList();
		inExpList.add(getAddSubExp());
		while (token_ == SelTokenizer.COMMA) {
			nextToken();
			inExpList.add(getAddSubExp());
		}
		expected(SelTokenizer.RIGHT_PAREN);
		return new InExp(
			targetExp,
			ExpressionUtil.toExpressionArray(inExpList));
	}

	private BoolExpression getBetweenExp(Expression targetExp) {
		nextToken();
		Expression fromExp = getAddSubExp();
		expected(SelTokenizer.AND);
		Expression toExp = getAddSubExp();
		return new BetweenExp(targetExp, fromExp, toExp);
	}

	private BoolExpression getNotBetweenExp(Expression targetExp) {
		nextToken();
		Expression fromExp = getAddSubExp();
		expected(SelTokenizer.AND);
		Expression toExp = getAddSubExp();
		return new NotBetweenExp(targetExp, fromExp, toExp);
	}

	private Expression getAddSubExp() {
		Expression arg1Exp = getMultDivModExp();
		while (true) {
			switch (token_) {
				case SelTokenizer.ADD :
					nextToken();
					arg1Exp = new AddExp(arg1Exp, getMultDivModExp());
					break;
				case SelTokenizer.SUBTRACT :
					nextToken();
					arg1Exp = new SubtractExp(arg1Exp, getMultDivModExp());
					break;
				default :
					return arg1Exp;
			}
		}
	}

	private Expression getMultDivModExp() {
		Expression arg1Exp = getConcatenateExp();
		while (true) {
			switch (token_) {
				case SelTokenizer.MULTIPLY :
					nextToken();
					arg1Exp = new MultiplyExp(arg1Exp, getConcatenateExp());
					break;
				case SelTokenizer.DIVIDE :
					nextToken();
					arg1Exp = new DivideExp(arg1Exp, getConcatenateExp());
					break;
				case SelTokenizer.MOD :
					nextToken();
					arg1Exp = new ModExp(arg1Exp, getConcatenateExp());
					break;
				default :
					return arg1Exp;
			}
		}
	}

	private Expression getConcatenateExp() {
		Expression exp = getSimpleExp();
		if (token_ != SelTokenizer.CONCATENATE) {
			return exp;
		}
		List exps = new ArrayList();
		exps.add(exp);
		while (token_ == SelTokenizer.CONCATENATE) {
			nextToken();
			exps.add(getSimpleExp());
		}
		return new ConcatenateExp(ExpressionUtil.toExpressionArray(exps));
	}

	private Expression getSimpleExp() {
		switch (token_) {
			case SelTokenizer.LEFT_PAREN :
				nextToken();
				Expression e = getAddSubExp();
				expected(SelTokenizer.RIGHT_PAREN);
				return e;
			case SelTokenizer.LONG :
				return getLongExp();
			case SelTokenizer.INTEGER :
				return getIntegerExp();
			case SelTokenizer.BIGDECIMAL :
				return getBigDecimalExp();
			case SelTokenizer.QUOTED_STRING :
				return getStringExp();
			case SelTokenizer.WORD :
				return getWordExp();
			case SelTokenizer.TO_TIMESTAMP :
				return getToTimestampExp();
			case SelTokenizer.TO_STRING :
				return getToStringExp();
			case SelTokenizer.NOW :
				return getNowExp();
			case SelTokenizer.NULL :
				return getNullExp();
			case SelTokenizer.NEW :
				return getNewExp();
			case SelTokenizer.TRUE :
				nextToken();
				return BooleanExp.TRUE;
			case SelTokenizer.FALSE :
				nextToken();
				return BooleanExp.FALSE;
			default :
				throw new UnexpectedTokenRuntimeException(
					tokenizer_.getReadString(),
					SelTokenizer.getTokenName(token_));
		}
	}

	private Expression getIntegerExp() {
		Expression e = new ObjectExp(tokenizer_.getIntegerValue());
		nextToken();
		return e;
	}

	private Expression getLongExp() {
		Expression e = new ObjectExp(tokenizer_.getLongValue());
		nextToken();
		return e;
	}

	private Expression getBigDecimalExp() {
		Expression e = new ObjectExp(tokenizer_.getBigDecimalValue());
		nextToken();
		return e;
	}

	private Expression getStringExp() {
		Expression e = new ObjectExp(tokenizer_.getStringValue());
		nextToken();
		return e;
	}

	private Expression getToTimestampExp() {
		nextToken();
		expected(SelTokenizer.LEFT_PAREN);
		Expression p1 = getSimpleExp();
		Expression p2 = null;
		if (token_ == SelTokenizer.COMMA) {
			nextToken();
			p2 = getStringExp();
		}
		String pattern = null;
		if (p2 != null) {
			pattern = (String) p2.evaluateValue(null);
		}
		expected(SelTokenizer.RIGHT_PAREN);
		return new ToTimestampExp(p1, pattern);
	}

	private Expression getToStringExp() {
		nextToken();
		expected(SelTokenizer.LEFT_PAREN);
		Expression p1 = getAddSubExp();
		Expression p2 = null;
		if (token_ == SelTokenizer.COMMA) {
			nextToken();
			p2 = getStringExp();
		}
		String pattern = null;
		if (p2 != null) {
			pattern = (String) p2.evaluateValue(null);
		}
		expected(SelTokenizer.RIGHT_PAREN);
		return new ToStringExp(p1, pattern);
	}

	private Expression getNowExp() {
		nextToken();
		expected(SelTokenizer.LEFT_PAREN);
		expected(SelTokenizer.RIGHT_PAREN);
		return NowExp.NOW;
	}

	private Expression getNullExp() {
		nextToken();
		return NullExp.NULL;
	}

	private Expression getWordExp() {
		Expression left = null;
		String s = tokenizer_.getStringValue();
		nextToken();
		if (token_ == SelTokenizer.LEFT_PAREN) {
			left = getMethodExp(s);
		} else {
			left = getVariableExp(s);
		}
		while (true) {
			if (token_ == SelTokenizer.DOT) {
				nextToken();
				s = tokenizer_.getStringValue();
				nextToken();
				if (token_ == SelTokenizer.LEFT_PAREN) {
					left = getMethodExp(left, s);
				} else {
					left = getVariableExp(left, s);
				}
			} else if (token_ == SelTokenizer.LEFT_BRACKET) {
				nextToken();
				Expression indexExp = getAddSubExp();
				expected(SelTokenizer.RIGHT_BRACKET);
				left = new GetArrayExp(left, indexExp);
			} else {
				break;
			}

		}
		return left;
	}

	private Expression getVariableExp(String name) {
		Expression targetExp = getVariableAndPropertyExp(name);
		if (token_ == SelTokenizer.LEFT_BRACKET) {
			nextToken();
			Expression indexExp = getAddSubExp();
			expected(SelTokenizer.RIGHT_BRACKET);
			return new GetArrayExp(targetExp, indexExp);
		}
		return targetExp;
	}

	private Expression getVariableExp(Expression targetExp, String name) {
		targetExp = getPropertyExp(targetExp, name);
		if (token_ == SelTokenizer.LEFT_BRACKET) {
			nextToken();
			Expression indexExp = getAddSubExp();
			expected(SelTokenizer.RIGHT_BRACKET);
			return new GetArrayExp(targetExp, indexExp);
		}
		return targetExp;
	}

	private Expression getMethodExp(String s) {
		nextToken();
		int pos = s.lastIndexOf('.');
		if (pos < 0) {
			throw new UnexpectedTokenRuntimeException(
				tokenizer_.getReadString(),
				s);
		}
		String targetName = s.substring(0, pos);
		String methodName = s.substring(pos + 1);
		Expression[] exps = getArgExpsForParen();
		try {
			Class targetClass = ClassUtil.forName(targetName);
			return new StaticMethodExp(targetClass, methodName, exps);
		} catch (ClassNotFoundRuntimeException ignore) {
			return new MethodExp(
				getVariableAndPropertyExp(targetName),
				methodName,
				exps);
		}

	}

	private Expression[] getArgExpsForParen() {
		List argExps = new ArrayList();
		if (token_ != SelTokenizer.RIGHT_PAREN) {
			argExps.add(getAddSubExp());
			while (token_ == SelTokenizer.COMMA) {
				nextToken();
				argExps.add(getAddSubExp());
			}
		}
		expected(SelTokenizer.RIGHT_PAREN);
		return ExpressionUtil.toExpressionArray(argExps);
	}
	
	private Expression[] getArgExpsForBrace() {
		List argExps = new ArrayList();
		if (token_ != SelTokenizer.RIGHT_BRACE) {
			argExps.add(getAddSubExp());
			while (token_ == SelTokenizer.COMMA) {
				nextToken();
				argExps.add(getAddSubExp());
			}
		}
		expected(SelTokenizer.RIGHT_BRACE);
		return ExpressionUtil.toExpressionArray(argExps);
	}

	private Expression getVariableAndPropertyExp(String s) {
		String[] names = StringUtil.split(s, ".");
		Expression targetExp = new GetVariableExp(names[0]);
		for (int i = 1; i < names.length; ++i) {
			targetExp = new GetPropertyExp(targetExp, names[i]);
		}
		return targetExp;
	}

	private Expression getPropertyExp(Expression targetExp, String s) {
		String[] names = StringUtil.split(s, ".");
		for (int i = 0; i < names.length; ++i) {
			targetExp = new GetPropertyExp(targetExp, names[i]);
		}
		return targetExp;
	}

	private Expression getMethodExp(Expression targetExp, String s) {
		nextToken();
		String methodName = null;
		int pos = s.lastIndexOf('.');
		if (pos < 0) {
			methodName = s;
		} else {
			targetExp = getPropertyExp(targetExp, s.substring(0, pos));
			methodName = s.substring(pos + 1);
		}
		Expression[] exps = getArgExpsForParen();
		return new MethodExp(targetExp, methodName, exps);
	}

	private Expression getNewExp() {
		nextToken();
		Class clazz = ClassUtil.forName(tokenizer_.getStringValue());
		nextToken();
		if (token_ == SelTokenizer.LEFT_PAREN) {
			nextToken();
			return new NewExp(clazz, getArgExpsForParen());
		} else {
			expected(SelTokenizer.LEFT_BRACKET);
			Expression indexExp = null;
			if (token_ != SelTokenizer.RIGHT_BRACKET) {
				indexExp = getAddSubExp();
			}
			expected(SelTokenizer.RIGHT_BRACKET);
			Expression[] argExps = ExpressionUtil.EMPTY_EXPRESSIONS;
			if (token_ == SelTokenizer.LEFT_BRACE) {
				nextToken();
				argExps = getArgExpsForBrace();
				if (argExps.length > 0) {
					indexExp = new ObjectExp(new Integer(argExps.length));
				}
			}
			return new NewArrayExp(clazz, indexExp, argExps);
		}
	}

	private void nextToken() {
		token_ = tokenizer_.nextToken();
	}

	private void expected(int t) {
		tokenizer_.expect(t, token_);
		nextToken();
	}
}
