package org.seasar.system;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import javax.management.ListenerNotFoundException;
import javax.management.NotificationBroadcaster;
import javax.management.NotificationListener;

import org.seasar.log.Logger;
import org.seasar.util.Reflector;
import org.seasar.util.SeasarException;

public final class JMXService implements Lifecycle {

	private static Logger _logger = Logger.getLogger(JMXService.class);
	private static Map _mbeans = new HashMap();
	private static Map _listenersMap = new HashMap();
	
	public JMXService() {
	}

	public static Object invoke(final String name, final String methodName,
            final Object[] parameters, final String[] signatures) throws Exception {

		
		Object mbean = getMBean(name);
		MBeanMetaData mbeanMetaData = MBeanMetaData.getInstance(mbean.getClass());
        Method method = mbeanMetaData.getMethod(methodName, signatures);
        return Reflector.invoke(method, mbean, parameters);
    }
    
    public static void registerMBean(final Object mbean, final String name) throws SeasarException {
    	MBeanMetaData.checkCompliance(mbean.getClass());
    	if (_mbeans.put(name, mbean) != null) {
    		throw new SeasarException("ESSR0335", new Object[]{name});
    	}
    }

    public static void unregisterMBean(final String name) {
        _mbeans.remove(name);
    }
    
    public static Object getMBean(final String name) throws SeasarException {
        Object mbean = _mbeans.get(name);
        if (mbean == null) {
        	throw new SeasarException("ESSR0001", new Object[]{name});
        }
        return mbean;
    }

	public static void addNotificationListener(String targetName, String listenerName)
        	throws SeasarException {

        Object listener = getMBean(listenerName);
        if (!(listener instanceof NotificationListener )) {
	    	throw new SeasarException("ESSR0336", new Object[]{listener.getClass().getName()});
        }
        addNotificationListener(targetName,(NotificationListener) listener);
    }
    
	public static void addNotificationListener(String targetName, NotificationListener listener)
        	throws SeasarException{
       
        Object target = getMBean(targetName);
		if (!(target instanceof NotificationBroadcaster)) {
            throw new SeasarException("ESSR0337", new Object[]{target.getClass().getName()});
        }       
		NotificationBroadcaster broadcaster =  (NotificationBroadcaster) target;      
        if (listener == null ) {
	    	throw new SeasarException("ESSR0007", new Object[]{"listener"});
		}
		NotificationListener listenerProxy = new NotificationListenerProxy(
			JMXUtil.getObjectName(targetName), listener);
	    broadcaster.addNotificationListener(listenerProxy, null, null);	
	    synchronized (_listenersMap) {
			Map listeners = (Map) _listenersMap.get(targetName);
			if (listeners == null)  {
				listeners = new HashMap();
				_listenersMap.put(targetName, listeners);
			}
			listeners.put(listener, listenerProxy);
		}
    }

	public static void removeNotificationListener(String targetName, String listenerName)
        	throws SeasarException {

        Object listener = getMBean(listenerName);
        if (!(listener instanceof NotificationListener )) {
	    	throw new SeasarException("ESSR0336", new Object[]{listener.getClass().getName()});
        }
        removeNotificationListener(targetName,(NotificationListener) listener);
    }
    
	public static void removeNotificationListener(String targetName, NotificationListener listener)
        	throws SeasarException {
       
        Object target = getMBean(targetName);
		if (!(target instanceof NotificationBroadcaster)) {
            throw new SeasarException("ESSR0337", new Object[]{target.getClass().getName()});
        }       
		NotificationBroadcaster broadcaster =  (NotificationBroadcaster) target;      
        if (listener == null ) {
	    	throw new SeasarException("ESSR0007", new Object[]{"listener"});
		}
		NotificationListener listenerProxy = null;
        synchronized (_listenersMap) {
			Map listeners = (Map) _listenersMap.get(targetName);
			listenerProxy = (NotificationListener) listeners.remove(listener);
		}
		if (listenerProxy == null) {
			throw new SeasarException("ESSR0001", new Object[]{"listenerProxy"});
		}
		try {
	    	broadcaster.removeNotificationListener(listenerProxy);
		} catch (ListenerNotFoundException ex) {
			throw SeasarException.convertSeasarException(ex);
		}
    }
    
    public static boolean isRegistered(String name) {
    	return _mbeans.get(name) != null;
    }
    	
	public void start() throws SeasarException {
	}

	public void stop() throws SeasarException {
		_mbeans.clear();
		_listenersMap.clear();
	}
}
