package org.seasar.nazuna.amf;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Map;
import java.util.TimeZone;

import org.seasar.log.Logger;
import org.seasar.util.EMap;
import org.seasar.util.PropertyDesc;
import org.seasar.util.Reflector;
import org.seasar.util.XMLUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public class AMFWriter {

	private static Logger _logger = Logger.getLogger(AMFWriter.class);
	private DataOutputStream _outputStream;
	private AMFMessage _message;
	private ArrayList _sharedObjects;

	public AMFWriter(DataOutputStream outputStream, AMFMessage message) {
		_outputStream = outputStream;
		_message = message;
	}

	public final void write() throws IOException {
		_outputStream.writeShort(0);
		_outputStream.writeShort(0);
		writeBodies();
	}
	
	public final void writeBodies() throws IOException {
		_outputStream.writeShort(_message.getBodyCount());
		for (int i = 0; i < _message.getBodyCount(); ++i) {
			_sharedObjects = new ArrayList();
			writeBody(_message.getBody(i));
		}
	}

	public final void writeBody(AMFBody body) throws IOException {
		_outputStream.writeUTF(body.getTarget());
		_outputStream.writeUTF(body.getResponse());
		_outputStream.writeInt(-1);
		writeData(body.getValue());
	}

	protected void writeData(Object value) throws IOException {
		if (value == null) {
			_outputStream.writeByte(AMFDataType.NULL);
		} else if (value instanceof String) {
			writeString((String) value);
		} else if (value instanceof BigDecimal) {
			writeString(value.toString());
		} else if (value instanceof Number) {
			writeNumber((Number) value);
		} else if (value instanceof Boolean) {
			writeBoolean((Boolean) value);
		} else if (value instanceof Date) {
			writeDate((Date) value);
		} else if (value instanceof Object[]) {
			writeArray((Object[]) value);
		} else if (value instanceof Iterator) {
			writeIterator((Iterator) value);
		} else if (value instanceof Collection) {
			writeCollection((Collection) value);
		} else if (value instanceof Map) {
			writeMap((Map) value);
		} else if (value instanceof Document) {
			writeXML((Document) value);
		} else {
			writeCustomClass(value);
		}
	}
	
	public final void writeNumber(Number value) throws IOException {
		_outputStream.writeByte(AMFDataType.NUMBER);
		_outputStream.writeDouble(value.doubleValue());
	}
	
	public final void writeString(String value) throws IOException {
		_outputStream.writeByte(AMFDataType.STRING);
		_outputStream.writeUTF(value);
	}
	
	public final void writeBoolean(Boolean value) throws IOException {
		_outputStream.writeByte(AMFDataType.BOOLEAN);
		_outputStream.writeBoolean(value.booleanValue());
	}
	
	public final void writeFlashedSharedObject(int index) throws IOException {
		_logger.debug("writeFlashedSharedObject:index=" + index);
		_outputStream.writeByte(AMFDataType.FLASHED_SHARED_OBJECT);
		_outputStream.writeShort(index);
	}
	
	public final void writeDate(Date value) throws IOException {
		_outputStream.writeByte(AMFDataType.DATE);
		_outputStream.writeDouble(((Date) value).getTime());
		int offset = TimeZone.getDefault().getRawOffset();
		_outputStream.writeShort(offset / AMFConstants.MILLS_PER_HOUR);
	}

	public final void writeCustomClass(Object object) throws IOException {
		int index = getSharedIndex(object);
		if (index >= 0) {
			writeFlashedSharedObject(index);
			return;
		}
		addSharedObject(object);
		int id = System.identityHashCode(object);
		String type = object.getClass().getName();
		_logger.debug("writeCustomClass:" + type + ",id=" + id);
		_outputStream.writeByte(AMFDataType.CUSTOM_CLASS);
		_outputStream.writeUTF(type);
		EMap map = Reflector.getPropertyDescMap(object.getClass());
		for (int i = 0; i < map.size(); ++i) {
			PropertyDesc propertyDesc = (PropertyDesc) map.get(i);
			if (propertyDesc.getReadMethod() != null) {
				Object value = propertyDesc.getValue(object);
				_logger.debug("id=" + id + ",propetyName=" + propertyDesc.getPropertyName() + ",value=" + value);
				_outputStream.writeUTF(propertyDesc.getPropertyName());
				writeData(value);
			} else {
				_logger.debug("id=" + id + "propetyName=" + propertyDesc.getPropertyName() + " readMethod not found");
			}
		}
		_outputStream.writeShort(0);
		_outputStream.writeByte(9);
	}

	public final void writeArray(Object[] value) throws IOException {
		int index = getSharedIndex(value);
		if (index >= 0) {
			writeFlashedSharedObject(index);
			return;
		}
		addSharedObject(value);
		_logger.debug("writeArray:size=" + value.length);
		_outputStream.writeByte(AMFDataType.ARRAY);
		_outputStream.writeInt(value.length);
		for (int i = 0; i < value.length; i++) {
			writeData(value[i]);
		}
	}

	public final void writeIterator(Iterator value) throws IOException {
		ArrayList list = new ArrayList();
		while (value.hasNext()) {
			list.add(value.next());
		}
		writeArray(list.toArray(new Object[list.size()]));
	}

	public final void writeCollection(Collection value) throws IOException {
		ArrayList list = new ArrayList();
		list.addAll(value);
		writeArray(list.toArray(new Object[list.size()]));
	}

	public final void writeMap(Map value) throws IOException {
		int index = getSharedIndex(value);
		if (index >= 0) {
			writeFlashedSharedObject(index);
			return;
		}
		int id = System.identityHashCode(value);
		addSharedObject(value);
		_logger.debug("writeMap:id=" + id);
		_outputStream.writeByte(AMFDataType.OBJECT);
		for (Iterator i = value.entrySet().iterator(); i.hasNext();) {
			Map.Entry e = (Map.Entry) i.next();
			String propertyName = String.valueOf(e.getKey());
			_logger.debug("id=" + id + ",propetyName=" + propertyName + ",value=" + e.getValue());
			_outputStream.writeUTF(propertyName);
			writeData(e.getValue());
		}
		_outputStream.writeShort(0);
		_outputStream.writeByte(9);
	}

	public final void writeXML(Document document) throws IOException {
		_outputStream.writeByte(AMFDataType.XML);
		Element element = document.getDocumentElement();
		String xmlData = XMLUtil.toString(element);
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		baos.write(xmlData.getBytes());
		_outputStream.writeInt(baos.size());
		baos.writeTo(_outputStream);
	}
	
	private void addSharedObject(Object o) {
		_logger.debug("addSharedObject:index=" + _sharedObjects.size() + ",value=" + o);
		_sharedObjects.add(o);
	}

	private final int getSharedIndex(Object o) {
		for (int i = 0; i < _sharedObjects.size(); ++i) {
			if (o == _sharedObjects.get(i)) {
				return i;
			}
		}
		return -1;
	}
}