package org.seasar.nazuna;

import java.math.BigDecimal;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.Map;

import org.seasar.util.ArrayUtil;
import org.seasar.util.Assertion;
import org.seasar.util.SMap;
import org.seasar.util.SeasarException;

public abstract class ValueType {

	public final static String STRING_NAME = "String";
	public final static String INTEGER_NAME = "Integer";
	public final static String LONG_NAME = "Long";
	public final static String DOUBLE_NAME = "Double";
	public final static String BIGDECIMAL_NAME = "BigDecimal";
	public final static String TIMESTAMP_NAME = "Timestamp";
	public final static String BINARY_NAME = "Binary";
	public final static ValueType STRING = new StringType(STRING_NAME);
	public final static ValueType INTEGER = new IntegerType(INTEGER_NAME);
	public final static ValueType LONG = new LongType(LONG_NAME);
	public final static ValueType DOUBLE = new DoubleType(DOUBLE_NAME);
	public final static ValueType BIGDECIMAL =
		new BigDecimalType(BIGDECIMAL_NAME);
	public final static ValueType TIMESTAMP = new TimestampType(TIMESTAMP_NAME);
	public final static ValueType BINARY = new BinaryType(BINARY_NAME);

	private static Map _typesByName = new SMap();
	private static Map _typesByClass = new SMap();
	private String _name;

	static {
		_typesByName.put(STRING_NAME, STRING);
		_typesByName.put(INTEGER_NAME, INTEGER);
		_typesByName.put(LONG_NAME, LONG);
		_typesByName.put(DOUBLE_NAME, DOUBLE);
		_typesByName.put(BIGDECIMAL_NAME, BIGDECIMAL);
		_typesByName.put(TIMESTAMP_NAME, TIMESTAMP);
		_typesByName.put(BINARY_NAME, BINARY);

		_typesByClass.put(String.class, STRING);
		_typesByClass.put(int.class, INTEGER);
		_typesByClass.put(long.class, LONG);
		_typesByClass.put(double.class, DOUBLE);
		_typesByClass.put(BigDecimal.class, BIGDECIMAL);
		_typesByClass.put(Timestamp.class, TIMESTAMP);
		_typesByClass.put(ArrayUtil.EMPTY_BYTES.getClass(), BINARY);
	}

	protected ValueType(final String name) {
		_name = name;
	}

	public static ValueType getType(final String name) {
		ValueType valueType = (ValueType) _typesByName.get(name);
		Assertion.assertFound("name", valueType);

		return valueType;
	}

	public static ValueType getType(final Class clazz) {
		ValueType valueType = (ValueType) _typesByClass.get(clazz);
		Assertion.assertFound(clazz.getName(), valueType);

		return valueType;
	}

	public static ValueType getType(int type) {
		switch (type) {
			case Types.TINYINT :
			case Types.SMALLINT :
			case Types.INTEGER :
				return INTEGER;
			case Types.BIGINT :
				return LONG;
			case Types.REAL :
			case Types.FLOAT :
			case Types.DOUBLE :
			case Types.DECIMAL :
			case Types.NUMERIC :
				return DOUBLE;
			case Types.DATE :
			case Types.TIME :
			case Types.TIMESTAMP :
				return TIMESTAMP;
			case Types.BINARY :
			case Types.VARBINARY :
				return BINARY;
			default :
				return STRING;
		}
	}

	public final String getName() {
		return _name;
	}

	public abstract Object getValue(ResultSet rs, String columnName)
		throws SeasarException;

	private static class StringType extends ValueType {

		StringType(final String name) {
			super(name);
		}

		public Object getValue(final ResultSet rs, final String columnName)
			throws SeasarException {

			try {
				return rs.getString(columnName);
			} catch (SQLException ex) {
				throw new SeasarException(
					"ESSR0347",
					new Object[] { columnName, ex },
					ex);
			}
		}
	}

	private static class IntegerType extends ValueType {

		IntegerType(final String name) {
			super(name);
		}

		public Object getValue(final ResultSet rs, final String columnName)
			throws SeasarException {

			try {
				return new Integer(rs.getInt(columnName));
			} catch (SQLException ex) {
				throw new SeasarException(
					"ESSR0347",
					new Object[] { columnName, ex },
					ex);
			}
		}
	}

	private static class LongType extends ValueType {

		LongType(final String name) {
			super(name);
		}

		public Object getValue(final ResultSet rs, final String columnName)
			throws SeasarException {

			try {
				return new Long(rs.getLong(columnName));
			} catch (SQLException ex) {
				throw new SeasarException(
					"ESSR0347",
					new Object[] { columnName, ex },
					ex);
			}
		}
	}

	private static class DoubleType extends ValueType {

		DoubleType(final String name) {
			super(name);
		}

		public Object getValue(final ResultSet rs, final String columnName)
			throws SeasarException {

			try {
				return new Double(rs.getDouble(columnName));
			} catch (SQLException ex) {
				throw new SeasarException(
					"ESSR0347",
					new Object[] { columnName, ex },
					ex);
			}
		}
	}

	private static class BigDecimalType extends ValueType {

		BigDecimalType(final String name) {
			super(name);
		}

		public Object getValue(final ResultSet rs, final String columnName)
			throws SeasarException {

			try {
				return rs.getBigDecimal(columnName);
			} catch (SQLException ex) {
				throw new SeasarException(
					"ESSR0347",
					new Object[] { columnName, ex },
					ex);
			}
		}
	}

	private static class TimestampType extends ValueType {

		TimestampType(final String name) {
			super(name);
		}

		public Object getValue(final ResultSet rs, final String columnName)
			throws SeasarException {

			try {
				return rs.getTimestamp(columnName);
			} catch (SQLException ex) {
				throw new SeasarException(
					"ESSR0347",
					new Object[] { columnName, ex },
					ex);
			}
		}
	}

	private static class BinaryType extends ValueType {

		BinaryType(final String name) {
			super(name);
		}

		public Object getValue(final ResultSet rs, final String columnName)
			throws SeasarException {

			try {
				return rs.getBytes(columnName);
			} catch (SQLException ex) {
				throw new SeasarException(
					"ESSR0347",
					new Object[] { columnName, ex },
					ex);
			}
		}
	}
}
