package org.seasar.nazuna;

import java.io.File;

import org.seasar.util.Assertion;
import org.seasar.util.SeasarException;

public final class RuletDeployer {

	private String _className;
	private File _srcFile;
	private File _classFile;
	private RuletFacade _ruletFacade;
	private long _lastModified = 0;
	
	public RuletDeployer(String className) {
		Assertion.assertNotNull("className", className);
		
		_className = className;
		_srcFile = RuletCompiler.getSrcFile(className);
		_classFile = RuletCompiler.getClassFile(className);
	}
	
	public static final RuletFacade createRuletFacade(Class clazz) throws SeasarException {
		RuletFacade facade = RuletFacade.create(clazz);
		facade.init();
		return facade;
	}
	
	public final RuletFacade getRuletFacade() throws SeasarException {
		return _ruletFacade;
	}
	
	public final File getSrcFile() {
		return _srcFile;
	}
	
	public final File getClassFile() {
		return _classFile;
	}
	
	public final String getClassName() {
		return _className;
	}
	
	public final long getLastModified() {
		return _lastModified;
	}
	
	public final void setLastModified(long lastModified) {
		_lastModified = lastModified;
	}
	
	public synchronized final void deploy() throws SeasarException {
		if (_srcFile.exists() && _classFile.exists()) {
			if (_classFile.lastModified() >= _srcFile.lastModified()) {
				if (_classFile.lastModified() > _lastModified) {
					deployFromClass();
				}
			} else {
				if (_srcFile.lastModified() > _lastModified) {
					deployFromSrc();
				}
			}
		} else if (_srcFile.exists()) {
			if (_srcFile.lastModified() > _lastModified) {
				deployFromSrc();
			}
		} else if (_classFile.exists()) {
			if (_classFile.lastModified() > _lastModified) {
				deployFromClass();
			}
		} else {
			throw new SeasarException("ESSR0001", new Object[]{_className});
		}
	}
	
	public final Class defineClass() throws SeasarException {
		RuletClassLoader loader = new RuletClassLoader();
		return loader.defineClass(_className, _classFile);
	}
	
	public final void deployFromSrc() throws SeasarException {
		destroy();
		RuletCompiler.compile(_srcFile, _classFile);
		Class ruletClass = defineClass();
		_lastModified = _srcFile.lastModified();
		_ruletFacade = createRuletFacade(ruletClass);
	}
	
	public final void deployFromClass() throws SeasarException {
		destroy();
		Class ruletClass = defineClass();
		_lastModified = _classFile.lastModified();
		_ruletFacade = createRuletFacade(ruletClass);
	}

	public final void destroy() throws SeasarException {
		if (_ruletFacade != null) {
			_ruletFacade.destroy();
			_ruletFacade = null;
		}
	}
}

