package org.seasar.naming;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.naming.Binding;
import javax.naming.Context;
import javax.naming.InvalidNameException;
import javax.naming.LinkRef;
import javax.naming.Name;
import javax.naming.NameClassPair;
import javax.naming.NameParser;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.Reference;
import javax.naming.Referenceable;
import javax.naming.spi.ObjectFactory;

import org.seasar.util.Reflector;

public final class NamingContext implements Context, Externalizable {

    final static long serialVersionUID = -4711790340772057976L;

    private transient Hashtable _env;
    private transient Name _prefix;
    private transient NamingServerMBean _naming;

    public NamingContext() {
    }

	public NamingContext(final Hashtable env, final Name prefix) throws NamingException {
		this(env, prefix, null);
	}
	
    public NamingContext(final Hashtable env, final Name prefix, final NamingServer naming)
    		throws NamingException {		
    			
        if (env != null) {
            _env = (Hashtable) env.clone();
        } else {
            _env = new Hashtable();
        }
        if (prefix != null) {
            _prefix = prefix;
        } else {
            _prefix = NamingParser.getInstance().parse("");
        }
        if (naming != null) {
        	_naming = naming;
        } else {
        	_naming = NamingServerManager.getNamingServer(env);
        }
    }

    public static ObjectFactory getObjectFactoryFromReference(final Reference ref)
             throws NamingException {

        try {
            String factoryName = ref.getFactoryClassName();
            if (factoryName != null) {
                try {
                    return (ObjectFactory) Thread.currentThread().getContextClassLoader().loadClass(
                            factoryName).newInstance();
                } catch (ClassNotFoundException ex) {
                }
            }
        } catch (Exception e) {
            handleException(e);
        }
        return null;
    }

    public void rebind(final String name, final Object obj) throws NamingException {
        rebind(getNameParser(name).parse(name), obj);
    }

    public void rebind(final Name name, final Object obj) throws NamingException {
        _naming.rebind(getAbsoluteName(name), obj);
    }

    public void bind(final String name, final Object obj) throws NamingException {
        bind(getNameParser(name).parse(name), obj);
    }

    public void bind(final Name name, final Object obj) throws NamingException {
        _naming.bind(getAbsoluteName(name), obj);
    }

    public Object lookup(final String name) throws NamingException {
        return lookup(getNameParser(name).parse(name));
    }

    public Object lookup(final Name name) throws NamingException {
        if (name.isEmpty()) {
            return this;
        }
        return resolveObject(name, _naming.lookup(getAbsoluteName(name)));
    }

    public void unbind(final String name) throws NamingException {
        unbind(getNameParser(name).parse(name));
    }

    public void unbind(final Name name) throws NamingException {
        _naming.unbind(getAbsoluteName(name));
    }

    public void rename(final String oldname, final String newname) throws NamingException {
        rename(getNameParser(oldname).parse(oldname), getNameParser(newname).parse(newname));
    }

    public void rename(final Name oldName, final Name newName) throws NamingException {
        bind(newName, lookup(oldName));
        unbind(oldName);
    }

    public NamingEnumeration list(final String name) throws NamingException {
        return list(getNameParser(name).parse(name));
    }

    public NamingEnumeration list(final Name name) throws NamingException {
        Map nm = _naming.getNameMap(getAbsoluteName(name));
        List list = new ArrayList(nm.size());
        for (Iterator i = nm.entrySet().iterator(); i.hasNext(); ) {
            Map.Entry e = (Map.Entry) i.next();
            list.add(getNameClassPair((String) e.getKey(), e.getValue()));
        }
        return new NamingEnumerationImpl(list);
    }

    public NamingEnumeration listBindings(final String name) throws NamingException {
        return listBindings(getNameParser(name).parse(name));
    }

    public NamingEnumeration listBindings(final Name name) throws NamingException {
        Map nm = _naming.getNameMap(getAbsoluteName(name));
        List list = new ArrayList(nm.size());
        for (Iterator i = nm.entrySet().iterator(); i.hasNext(); ) {
            Map.Entry e = (Map.Entry) i.next();
            list.add(getBinding((String) e.getKey(), e.getValue()));
        }
        return new NamingEnumerationImpl(list);
    }

    public String composeName(final String name, final String prefix)
             throws NamingException {

        return composeName(NamingParser.getInstance().parse(name),
                NamingParser.getInstance().parse(prefix)).toString();
    }

    public Name composeName(final Name name, final Name prefix)
             throws NamingException {

        Name result = (Name) (prefix.clone());
        result.addAll(name);
        return result;
    }

    public NameParser getNameParser(final Name name) throws NamingException {
        return NamingParser.getInstance();
    }

    public NameParser getNameParser(final String name) throws NamingException {
        return NamingParser.getInstance();
    }

    public Context createSubcontext(final String name) throws NamingException {
        return createSubcontext(getNameParser(name).parse(name));
    }

    public Context createSubcontext(final Name name) throws NamingException {
        return resolveNamingContext(_naming.createSubcontext(getAbsoluteName(name)));
    }

    public Object addToEnvironment(final String key, final Object value)
             throws NamingException {

        return _env.put(key, value);
    }

    public Object removeFromEnvironment(final String key) throws NamingException {
        return _env.remove(key);
    }

    public Hashtable getEnvironment() throws NamingException {
        return _env;
    }

    public void close() throws NamingException {
        _env = null;
        _prefix = null;
        _naming = null;
    }

    public String getNameInNamespace() throws NamingException {
        return _prefix.toString();
    }

    public void destroySubcontext(final String name) throws NamingException {
        destroySubcontext(getNameParser(name).parse(name));
    }

    public void destroySubcontext(final Name name) throws NamingException {
        _naming.destroySubcontext(getAbsoluteName(name));
    }

    public Object lookupLink(final String name) throws NamingException {
        return lookupLink(getNameParser(name).parse(name));
    }

    public Object lookupLink(final Name name) throws NamingException {
        return lookup(name);
    }

    public void writeExternal(ObjectOutput s) throws IOException {
        s.writeObject(_env);
        s.writeObject(_prefix);
        s.writeObject(_naming);
    }

    public void readExternal(ObjectInput s)
             throws IOException, ClassNotFoundException {

        _env = (Hashtable) s.readObject();
        _prefix = (Name) s.readObject();
        _naming = (NamingServerMBean) s.readObject();
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (o == null || !(o instanceof NamingContext)) {
            return false;
        }
        NamingContext other = (NamingContext) o;
        return _prefix.equals(other._prefix);
    }

    public String toString() {
        return _prefix.toString();
    }

    Name getAbsoluteName(final Name n) throws NamingException {
        if (n.isEmpty()) {
            throw new InvalidNameException();
        } else if (n.get(0).toString().equals("")) {
            return n.getSuffix(1);
        } else {
            return composeName(n, _prefix);
        }
    }

    private static void handleException(final Exception e) throws NamingException {
        throw NamingUtil.convertNamingException(e);
    }

    private NameClassPair getNameClassPair(final String name, final Object obj) throws NamingException {
        return new NameClassPair(name, getClassName(obj), true);
    }

    private Binding getBinding(final String name, final Object obj) throws NamingException {
        return new Binding(name, getClassName(obj), resolveObject(
                getNameParser(name).parse(name), obj), true);
    }

    private String getClassName(final Object obj) throws NamingException {
        String className = obj.getClass().getName();
        Reference ref = null;
        if (obj instanceof Referenceable) {
            ref = ((Referenceable) obj).getReference();
        } else if (obj instanceof Reference) {
            ref = (Reference) obj;
        }
        if (ref != null) {
            className = ref.getClassName();
        }
        return className;
    }

    private Object resolveObject(final Name name, final Object obj) throws NamingException {
        if (obj instanceof NamingContext) {
            return resolveNamingContext((NamingContext) obj);
        } else if (obj instanceof LinkRef) {
            return resolveLinkRef((LinkRef) obj);
        } else if (obj instanceof Reference || obj instanceof Referenceable) {
            return getObjectInstance(name, obj);
        } else {
            return obj;
        }
    }

    private NamingContext resolveNamingContext(final NamingContext ctx) throws NamingException {
        for (Iterator i = _env.entrySet().iterator(); i.hasNext(); ) {
            Map.Entry e = (Map.Entry) i.next();
            ctx.addToEnvironment((String) e.getKey(), e.getValue());
        }
        ctx._naming = _naming;
        return ctx;
    }

    private Object resolveLinkRef(final LinkRef linkRef) throws NamingException {
        String linkName = linkRef.getLinkName();
        if (linkName.startsWith("./")) {
            return lookup(linkName.substring(2));
        } else {
            return lookup(linkName);
        }
    }

    private Object getObjectInstance(final Name name, final Object obj)
             throws NamingException {

        Reference ref = null;
        if (obj instanceof Reference) {
            ref = (Reference) obj;
        } else if (obj instanceof Referenceable) {
            ref = ((Referenceable) (obj)).getReference();
        }
        if (ref != null) {
            ObjectFactory factory = getObjectFactoryFromReference(ref);
            if (factory != null) {
                try {
                    return factory.getObjectInstance(ref, getAbsoluteName(name), this, _env);
                } catch (Exception e) {
                    handleException(e);
                }
            }
            String className = ref.getClassName();
            if (className != null) {
                return Reflector.newInstance(className);
            }
        }
        return obj;
    }
}