package test.org.seasar.nazuna.amf;

import java.util.*;
import java.io.*;
import java.math.BigDecimal;

import org.seasar.nazuna.amf.*;

import junit.framework.TestCase;

public class AMFReaderWriterTest extends TestCase {
	
	public AMFReaderWriterTest(String arg0) {
		super(arg0);
	}
	
	public void testNumber() throws Exception {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		DataOutputStream dos = new DataOutputStream(baos);
		AMFMessage message = new AMFMessage();
		AMFBody body = new AMFBody("aaa.Hoge.foo", "response", new Double(1));
		message.addBody(body);
		AMFWriter writer = new AMFWriter(dos, message);
		writer.write();
		
		ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
		DataInputStream dis = new DataInputStream(bais);
		AMFReader reader = new AMFReader(dis);
		AMFMessage message2 = reader.read();
		assertEquals("1", 1, message2.getBodyCount());
		AMFBody body2 = message2.getBody(0);
		assertEquals("2", "aaa.Hoge.foo", body2.getTarget());
		assertEquals("3", "response", body2.getResponse());
		assertEquals("4", new Double(1), body2.getValue());
	}
	
	public void testBoolean() throws Exception {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		DataOutputStream dos = new DataOutputStream(baos);
		AMFMessage message = new AMFMessage();
		AMFBody body = new AMFBody("aaa.Hoge.foo", "response", Boolean.TRUE);
		message.addBody(body);
		AMFWriter writer = new AMFWriter(dos, message);
		writer.write();
		
		ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
		DataInputStream dis = new DataInputStream(bais);
		AMFReader reader = new AMFReader(dis);
		AMFMessage message2 = reader.read();
		AMFBody body2 = message2.getBody(0);
		assertEquals("1", Boolean.TRUE, body2.getValue());
	}
	
	public void testString() throws Exception {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		DataOutputStream dos = new DataOutputStream(baos);
		AMFMessage message = new AMFMessage();
		AMFBody body = new AMFBody("aaa.Hoge.foo", "response", "abc");
		message.addBody(body);
		AMFWriter writer = new AMFWriter(dos, message);
		writer.write();
		
		ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
		DataInputStream dis = new DataInputStream(bais);
		AMFReader reader = new AMFReader(dis);
		AMFMessage message2 = reader.read();
		AMFBody body2 = message2.getBody(0);
		assertEquals("1", "abc", body2.getValue());
	}
	
	public void testObject() throws Exception {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		DataOutputStream dos = new DataOutputStream(baos);
		AMFMessage message = new AMFMessage();
		Map value = new HashMap();
		value.put("aaa", "111");
		value.put("bbb", "222");
		AMFBody body = new AMFBody("aaa.Hoge.foo", "response", value);
		message.addBody(body);
		AMFWriter writer = new AMFWriter(dos, message);
		writer.write();
		
		ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
		DataInputStream dis = new DataInputStream(bais);
		AMFReader reader = new AMFReader(dis);
		AMFMessage message2 = reader.read();
		AMFBody body2 = message2.getBody(0);
		Map value2 = (Map) body2.getValue();
		assertEquals("1", 2, value2.size());
		assertEquals("2", "111", value2.get("aaa"));
		assertEquals("3", "222", value2.get("bbb"));
	}
	
	public void testNull() throws Exception {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		DataOutputStream dos = new DataOutputStream(baos);
		AMFMessage message = new AMFMessage();
		AMFBody body = new AMFBody("aaa.Hoge.foo", "response", null);
		message.addBody(body);
		AMFWriter writer = new AMFWriter(dos, message);
		writer.write();
		
		ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
		DataInputStream dis = new DataInputStream(bais);
		AMFReader reader = new AMFReader(dis);
		AMFMessage message2 = reader.read();
		AMFBody body2 = message2.getBody(0);
		assertEquals("1", null, body2.getValue());
	}
	
	public void testArray() throws Exception {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		DataOutputStream dos = new DataOutputStream(baos);
		AMFMessage message = new AMFMessage();
		Object[] value = new Object[]{"111", "222"};
		AMFBody body = new AMFBody("aaa.Hoge.foo", "response", value);
		message.addBody(body);
		AMFWriter writer = new AMFWriter(dos, message);
		writer.write();
		
		ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
		DataInputStream dis = new DataInputStream(bais);
		AMFReader reader = new AMFReader(dis);
		AMFMessage message2 = reader.read();
		AMFBody body2 = message2.getBody(0);
		List value2 = (List) body2.getValue();
		assertEquals("1", 2, value2.size());
		assertEquals("2", "111", value2.get(0));
		assertEquals("3", "222", value2.get(1));
	}
	
	public void testArray2() throws Exception {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		DataOutputStream dos = new DataOutputStream(baos);
		AMFMessage message = new AMFMessage();
		List value = new ArrayList();
		value.add("111");
		value.add("222");
		AMFBody body = new AMFBody("aaa.Hoge.foo", "response", value);
		message.addBody(body);
		AMFWriter writer = new AMFWriter(dos, message);
		writer.write();
		
		ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
		DataInputStream dis = new DataInputStream(bais);
		AMFReader reader = new AMFReader(dis);
		AMFMessage message2 = reader.read();
		AMFBody body2 = message2.getBody(0);
		List value2 = (List) body2.getValue();
		assertEquals("1", 2, value2.size());
		assertEquals("2", "111", value2.get(0));
		assertEquals("3", "222", value2.get(1));
	}
	
	public void testArray3() throws Exception {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		DataOutputStream dos = new DataOutputStream(baos);
		AMFMessage message = new AMFMessage();
		List value = new ArrayList();
		value.add("111");
		value.add("222");
		AMFBody body = new AMFBody("aaa.Hoge.foo", "response", value.iterator());
		message.addBody(body);
		AMFWriter writer = new AMFWriter(dos, message);
		writer.write();
		
		ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
		DataInputStream dis = new DataInputStream(bais);
		AMFReader reader = new AMFReader(dis);
		AMFMessage message2 = reader.read();
		AMFBody body2 = message2.getBody(0);
		List value2 = (List) body2.getValue();
		assertEquals("1", 2, value2.size());
		assertEquals("2", "111", value2.get(0));
		assertEquals("3", "222", value2.get(1));
	}

	public void testCustomClass() throws Exception {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		DataOutputStream dos = new DataOutputStream(baos);
		AMFMessage message = new AMFMessage();
		MyBean value = new MyBean();
		value.setAaa(1);
		value.setBbb(2);
		value.setCcc(3);
		value.setDdd("4");
		value.setEee(true);
		value.setFff(new Date(5));
		List ggg = new ArrayList();
		MyBean b1 = new MyBean();
		b1.setAaa(2);
		ggg.add(b1);
		MyBean b2 = new MyBean();
		b2.setAaa(3);
		ggg.add(b2);
		value.setGgg(ggg);
		MyBean hhh = new MyBean();
		hhh.setAaa(4);
		value.setHhh(hhh);
		BigDecimal iii = new BigDecimal("1234567890123456789");
		value.setIii(iii);
		
		AMFBody body = new AMFBody("aaa.Hoge.foo", "response", value);
		message.addBody(body);
		AMFWriter writer = new AMFWriter(dos, message);
		writer.write();
		
		ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
		DataInputStream dis = new DataInputStream(bais);
		AMFReader reader = new AMFReader(dis);
		AMFMessage message2 = reader.read();
		AMFBody body2 = message2.getBody(0);
		MyBean value2 = (MyBean) body2.getValue();
		assertEquals("1", 1, value2.getAaa());
		assertEquals("2", 2, value2.getBbb());
		assertEquals("3", 3, value2.getCcc(), 0);
		assertEquals("4", "4", value2.getDdd());
		assertEquals("5", true, value2.isEee());
		assertEquals("6", new Date(5), value2.getFff());
		List ggg2 = value2.getGgg();
		assertEquals("7", 2, ggg2.size());
		MyBean b4 = (MyBean) ggg2.get(0);
		MyBean b5 = (MyBean) ggg2.get(1);
		assertEquals("8", 2, b4.getAaa());
		assertEquals("9", 3, b5.getAaa());
		MyBean hhh2 = value2.getHhh();
		assertEquals("10", 4, hhh2.getAaa());
		BigDecimal iii2 = value2.getIii();
		assertEquals("11", iii, iii2);
	}
	
	public static void main(String[] args) {
		junit.textui.TestRunner.run(AMFReaderWriterTest.class);
	}

	protected void setUp() throws Exception {
		super.setUp();
	}

	protected void tearDown() throws Exception {
		super.tearDown();
	}
	
	public static class MyBean {
		
		private int _aaa;
		private long _bbb;
		private double _ccc;
		private String _ddd;
		private boolean _eee;
		private Date _fff;
		private List _ggg = new ArrayList();
		private MyBean _hhh;
		private BigDecimal _iii;
		
		public int getAaa() {
			return _aaa;
		}
		
		public void setAaa(int aaa) {
			_aaa = aaa;
		}
		
		public long getBbb() {
			return _bbb;
		}
		
		public void setBbb(long bbb) {
			_bbb = bbb;
		}
		
		public double getCcc() {
			return _ccc;
		}
		
		public void setCcc(double ccc) {
			_ccc = ccc;
		}
		
		public String getDdd() {
			return _ddd;
		}
		
		public void setDdd(String ddd) {
			_ddd = ddd;
		}
		
		public boolean isEee() {
			return _eee;
		}
		
		public void setEee(boolean eee) {
			_eee = eee;
		}
		
		public Date getFff() {
			return _fff;
		}
		
		public void setFff(Date fff) {
			_fff = fff;
		}
		
		public List getGgg() {
			return _ggg;
		}
		
		public void setGgg(List ggg) {
			_ggg = ggg;
		}
		
		public MyBean getHhh() {
			return _hhh;
		}
		
		public void setHhh(MyBean hhh) {
			_hhh = hhh;
		}
		
		public BigDecimal getIii() {
			return _iii;
		}
		
		public void setIii(BigDecimal iii) {
			_iii = iii;
		}
	}

}
