package org.seasar.util;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;
import java.lang.reflect.Array;
import java.util.AbstractList;
import java.util.AbstractMap;
import java.util.AbstractSet;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

public final class EMap extends AbstractMap implements Map, Cloneable, Externalizable {

	static final long serialVersionUID = -6319986824963561001L;
    private static final int INITIAL_CAPACITY = 17;
    private static final float LOAD_FACTOR = 0.75f;
    private transient int _threshold;
    private transient Entry[] _mapTable;
    private transient Entry[] _listTable;
    private transient int _size = 0;
    private transient Set _entrySet = null;

    public EMap() {
        this(INITIAL_CAPACITY);
    }

    public EMap(int initialCapacity) {
        if (initialCapacity <= 0) {
            initialCapacity = INITIAL_CAPACITY;
        }
        _mapTable = new Entry[initialCapacity];
        _listTable = new Entry[initialCapacity];
        _threshold = (int) (initialCapacity * LOAD_FACTOR);
    }
    
    public EMap(Map map) {
    	this((int) (map.size() / LOAD_FACTOR) + 1);
        putAll(map);
    }

    public final int size() {
        return _size;
    }

    public final boolean isEmpty() {
        return _size == 0;
    }

    public final boolean containsValue(Object value) {
        return indexOf(value) >= 0;
    }

    public final int indexOf(Object value) {
        if (value != null) {
            for (int i = 0; i < _size; i++) {
                if (value.equals(_listTable[i]._value)) {
                    return i;
                }
            }
        } else {
            for (int i = 0; i < _size; i++) {
                if (_listTable[i]._value == null) {
                    return i;
                }
            }
        }
        return -1;
    }
    
    public final boolean containsKey(final Object key) {
        Entry[] mapTable = _mapTable;
        if (key != null) {
            int hashCode = key.hashCode();
            int index = (hashCode & 0x7FFFFFFF) % mapTable.length;
            for (Entry e = mapTable[index]; e != null; e = e._next) {
                if (e._hashCode == hashCode && key.equals(e._key)) {
                    return true;
                }
            }
        } else {
            for (Entry e = mapTable[0]; e != null; e = e._next) {
                if (e._key == null) {
                    return true;
                }
            }
        }
        return false;
    }

    public final Object get(final Object key) {
        Entry[] mapTable = _mapTable;
        if (key != null) {
            int hashCode = key.hashCode();
            int index = (hashCode & 0x7FFFFFFF) % mapTable.length;
            for (Entry e = mapTable[index]; e != null; e = e._next) {
                if (e._hashCode == hashCode && key.equals(e._key)) {
                    return e._value;
                }
            }
        } else {
            for (Entry e = mapTable[0]; e != null; e = e._next) {
                if (e._key == null) {
                    return e._value;
                }
            }
        }
        return null;
    }

    public final Object get(final int index) {
        return getEntry(index)._value;
    }

    public final Object getKey(final int index) {
        return getEntry(index)._key;
    }

    public final Entry getEntry(final int index) {
        if (index >= _size) {
            throw new SeasarRuntimeException("ESSR0012",
                    new Object[]{Integer.toString(index), Integer.toString(_size)});
        }
        return _listTable[index];
    }

    public final List getList() {
        return new EMapList();
    }

    public final Object put(final Object key, final Object value) {
        int hashCode = 0;
        int index = 0;

        if (key != null) {
            hashCode = key.hashCode();
            index = (hashCode & 0x7FFFFFFF) % _mapTable.length;
            for (Entry e = _mapTable[index]; e != null; e = e._next) {
                if ((e._hashCode == hashCode) && key.equals(e._key)) {
                    return swapValue(e, value);
                }
            }
        } else {
            for (Entry e = _mapTable[0]; e != null; e = e._next) {
                if (e._key == null) {
                    return swapValue(e, value);
                }
            }
        }
        ensureCapacity();
        index = (hashCode & 0x7FFFFFFF) % _mapTable.length;
        Entry e = new Entry(hashCode, key, value, _mapTable[index]);
        _mapTable[index] = e;
        _listTable[_size++] = e;
        return null;
    }

    public final Object put(final int index, final Object key, final Object value) {
        int hashCode = 0;
        int mapIndex = 0;

        if (key != null) {
            hashCode = key.hashCode();
            mapIndex = (hashCode & 0x7FFFFFFF) % _mapTable.length;
            for (Entry e = _mapTable[mapIndex]; e != null; e = e._next) {
                if ((e._hashCode == hashCode) && key.equals(e._key)) {
                    return swapValue(index, e, value);
                }
            }
        } else {
            for (Entry e = _mapTable[0]; e != null; e = e._next) {
                if (e._key == null) {
                    return swapValue(index, e, value);
                }
            }
        }
        ensureCapacity();
        mapIndex = (hashCode & 0x7FFFFFFF) % _mapTable.length;
        Entry e = new Entry(hashCode, key, value, _mapTable[mapIndex]);
        _mapTable[mapIndex] = e;
        addList(index, e);
        return null;
    }

    public final Object remove(final Object key) {
        Entry e = removeMap(key);
        if (e != null) {
            Object value = e._value;
            removeList(indexOf(e));
            e.clear();
            return value;
        } else {
            return null;
        }
    }

    public final Object remove(int index) {
        Entry e = removeList(index);
        Object value = e._value;
        removeMap(e._key);
        e._value = null;
        return value;
    }

    public final void putAll(Map map) {
        for (Iterator i = map.entrySet().iterator(); i.hasNext(); ) {
            Map.Entry e = (Map.Entry) i.next();
            put(e.getKey(), e.getValue());
        }
    }

    public final void clear() {
        for (int i = 0; i < _mapTable.length; i++) {
            _mapTable[i] = null;
        }
        for (int i = 0; i < _listTable.length; i++) {
            _listTable[i] = null;
        }
        _size = 0;
    }

    public final Object[] toArray() {
        Object[] array = new Object[_size];
        for (int i = 0; i < array.length; i++) {
            array[i] = get(i);
        }
        return array;
    }

    public final Object[] toArray(final Object proto[]) {
        Object[] array = proto;
        if (proto.length < _size) {
            array = (Object[]) Array.newInstance(
                    proto.getClass().getComponentType(), _size);
        }
        for (int i = 0; i < array.length; i++) {
            array[i] = get(i);
        }
        if (array.length > _size) {
            array[_size] = null;
        }
        return array;
    }

    public final void sort() {
        Sorter sorter = new EMapSorter();
        sorter.sort(_listTable, 0, _size);
    }

    public final void sort(final Comparator comparator) {
        Sorter sorter = new EMapSorterForComparator(comparator);
        sorter.sort(_listTable, 0, _size);
    }

    public final Object clone() {
        EMap emap = new EMap();
        emap._mapTable = new Entry[_mapTable.length];
        emap._listTable = new Entry[_listTable.length];
        for (int i = 0; i < _size; i++) {
            Entry e = (Entry) _listTable[i].clone();
            emap.put(e._key, e._value);
        }
        emap._entrySet = null;
        return emap;
    }

    public final boolean equals(Object o) {
        if (!getClass().isInstance(o)) {
            return false;
        }
        EMap e = (EMap) o;
        if (_size != e._size) {
            return false;
        }
        for (int i = 0; i < _size; i++) {
            if (!_listTable[i].equals(e._listTable[i])) {
                return false;
            }
        }
        return true;
    }

    public final Set entrySet() {
        if (_entrySet == null) {
            _entrySet =
                new AbstractSet() {
                    public Iterator iterator() {
                        return new EMapIterator();
                    }

                    public boolean contains(Object o) {
                        if (!(o instanceof Entry)) {
                            return false;
                        }
                        Entry entry = (Entry) o;
                        int index = (entry._hashCode & 0x7FFFFFFF) % _mapTable.length;
                        for (Entry e = _mapTable[index]; e != null; e = e._next) {
                            if (e.equals(entry)) {
                                return true;
                            }
                        }
                        return false;
                    }

                    public boolean remove(Object o) {
                        if (!(o instanceof Entry)) {
                            return false;
                        }
                        Entry entry = (Entry) o;
                        return EMap.this.remove(entry._key) != null;
                    }

                    public int size() {
                        return _size;
                    }

                    public void clear() {
                        EMap.this.clear();
                    }
                };
        }
        return _entrySet;
    }
    
    public final void writeExternal(final ObjectOutput out) throws IOException {
        out.writeInt(_listTable.length);
        out.writeInt(_size);
        for (int i = 0; i < _size; i++) {
            out.writeObject(_listTable[i]._key);
            out.writeObject(_listTable[i]._value);
        }
    }

    public final void readExternal(final ObjectInput in)
             throws IOException, ClassNotFoundException {

        int num = in.readInt();
        _mapTable = new Entry[num];
        _listTable = new Entry[num];
        _threshold = (int) (num * LOAD_FACTOR);
        int size = in.readInt();
        for (int i = 0; i < size; i++) {
            Object key = in.readObject();
            Object value = in.readObject();
            put(key, value);
        }
    }

    private final void addList(final int index, final Entry entry) {
        checkHighRange(index);
        System.arraycopy(_listTable, index, _listTable, index + 1, _size - index);
        _listTable[index] = entry;
        _size++;
    }

    private final Entry removeMap(Object key) {
        int hashCode = 0;
        int index = 0;

        if (key != null) {
            hashCode = key.hashCode();
            index = (hashCode & 0x7FFFFFFF) % _mapTable.length;
            for (Entry e = _mapTable[index], prev = null; e != null; prev = e, e = e._next) {
            	if ((e._hashCode == hashCode) && key.equals(e._key)) {
                	if (prev != null) {
                    	prev._next = e._next;
                	} else {
                    	_mapTable[index] = e._next;
                	}
                	return e;
            	}
        	}
        } else {
        	for (Entry e = _mapTable[index], prev = null; e != null; prev = e, e = e._next) {
         	   if ((e._hashCode == hashCode) && e._key == null) {
            	    if (prev != null) {
                	    prev._next = e._next;
                	} else {
                    	_mapTable[index] = e._next;
                	}
                	return e;
            	}
        	}
        }
        return null;
    }

    private final Entry removeList(int index) {
        Entry e = _listTable[index];
        int numMoved = _size - index - 1;
        if (numMoved > 0) {
            System.arraycopy(_listTable, index + 1, _listTable, index, numMoved);
        }
        _listTable[--_size] = null;
        return e;
    }

    private final int indexOf(final Entry entry) {
        for (int i = 0; i < _size; i++) {
            if (_listTable[i] == entry) {
                return i;
            }
        }
        throw new SeasarRuntimeException("ESSR0001", new Object[]{entry});
    }

    private final void ensureCapacity() {
        if (_size >= _threshold) {
            Entry[] oldTable = _listTable;
            int newCapacity = oldTable.length * 2 + 1;
            Entry[] newMapTable = new Entry[newCapacity];
            Entry[] newListTable = new Entry[newCapacity];
            _threshold = (int) (newCapacity * LOAD_FACTOR);
            System.arraycopy(oldTable, 0, newListTable, 0, _size);
            for (int i = 0; i < _size; i++) {
                Entry old = oldTable[i];
                int index = (old._hashCode & 0x7FFFFFFF) % newCapacity;
                Entry e = old;
                old = old._next;
                e._next = newMapTable[index];
                newMapTable[index] = e;
            }
            _mapTable = newMapTable;
            _listTable = newListTable;
        }
    }

    private final Object swapValue(final Entry entry, final Object value) {
        Object old = entry._value;
        entry._value = value;
        return old;
    }

    private final Object swapValue(final int index, final Entry entry, final Object value) {
        Object old = swapValue(entry, value);
        removeList(indexOf(entry));
        addList(index, entry);
        return old;
    }

    private final void checkHighRange(final int index) {
        if (index > _size) {
            throw new ArrayIndexOutOfBoundsException("index:" + index);
        }
    }

    private class EMapIterator implements Iterator {

        private int _current = 0;
        private int _last = -1;

        public boolean hasNext() {
            return _current != _size;
        }

        public Object next() {
            try {
                Object n = _listTable[_current];
                _last = _current++;
                return n;
            } catch (IndexOutOfBoundsException e) {
                throw new NoSuchElementException();
            }
        }

        public void remove() {
            if (_last == -1) {
                throw new IllegalStateException();
            }
            EMap.this.remove(_last);
            if (_last < _current) {
                _current--;
            }
            _last = -1;
        }
    }

    private static class Entry implements Map.Entry, Externalizable {

        transient int _hashCode;
        transient Object _key;
        transient Object _value;
        transient Entry _next;

        public Entry(final int hashCode, final Object key, final Object value,
                final Entry next) {

            _hashCode = hashCode;
            _key = key;
            _value = value;
            _next = next;
        }

        public Object clone() {
            Object value = _value;
            if (value instanceof Cloneable) {
                value = Reflector.clone(value);
            }
            return new Entry(_hashCode, _key, value,
                    (_next == null ? null : (Entry) _next.clone()));
        }

        public Object getKey() {
            return _key;
        }

        public Object getValue() {
            return _value;
        }

        public Object setValue(final Object value) {
            Object oldValue = _value;
            _value = value;
            return oldValue;
        }
        
        public void clear() {
        	_key = null;
        	_value = null;
        	_next = null;
        }

        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            Entry e = (Entry) o;
            return (_key != null ? _key.equals(e._key) : e._key == null) &&
                    (_value != null ? _value.equals(e._value) : e._value == null);
        }

        public int hashCode() {
            return _hashCode;
        }

        public String toString() {
            return _key + "=" + _value;
        }

        public void writeExternal(final ObjectOutput s) throws IOException {
            s.writeInt(_hashCode);
            s.writeObject(_key);
            s.writeObject(_value);
            s.writeObject(_next);
        }

        public void readExternal(final ObjectInput s)
                 throws IOException, ClassNotFoundException {

            _hashCode = s.readInt();
            _key = s.readObject();
            _value = s.readObject();
            _next = (Entry) s.readObject();
        }
    }

    private class EMapList extends AbstractList implements Serializable {

        final static long serialVersionUID = -351835908133693895L;
        private Entry[] _array;

        EMapList() {
            _array = new Entry[_size];
            System.arraycopy(_listTable, 0, _array, 0, _size);
        }

        public int size() {
            return _array.length;
        }

        public Object get(int index) {
            return _array[index]._value;
        }

        public Object set(int index, Object value) {
            Object oldValue = get(index);
            int originalIndex = indexOf(value);
            _listTable[index] = _array[originalIndex];
            return oldValue;
        }

        public int indexOf(Object o) {
            for (int i = 0; i < size(); i++) {
                if (get(i).equals(o)) {
                    return i;
                }
            }
            return -1;
        }
    }

    private class EMapSorter extends Sorter {

        protected int doCompare(final Object o1, final Object o2) {
            Object v = ((Entry) o1)._value;
            Object v2 = ((Entry) o2)._value;
            return super.doCompare(v, v2);
        }
    }

    private class EMapSorterForComparator extends Sorter {

        private Comparator _comparator;

        EMapSorterForComparator(final Comparator comparator) {
            if (comparator == null) {
                throw new SeasarRuntimeException("ESSR0007", new Object[]{"comparator"});
            }
            _comparator = comparator;
        }

        protected int doCompare(final Object o1, final Object o2) {
            Object v = ((Entry) o1)._value;
            Object v2 = ((Entry) o2)._value;
            return _comparator.compare(v, v2);
        }
    }
}