package org.seasar.nazuna;

import java.util.Arrays;
import java.util.List;

import org.seasar.util.Assertion;
import org.seasar.util.EArrayList;
import org.seasar.util.EMap;
import org.seasar.util.SeasarException;

public final class GroupDesc {

    private String[] _keys;
    private GroupElement[] _groupElements;
    
    public GroupDesc(String clause) throws SeasarException {
    	Assertion.assertNotNull("clause", clause);
    	
    	List keyList = new EArrayList();
    	List groupElementList = new EArrayList();
    	RuleTokenizer tokenizer = new RuleTokenizer(clause);
    	int token = tokenizer.nextToken();
    	while (token != RuleTokenizer.EOF) {
    		switch (token) {
    			case RuleTokenizer.WORD:
    				String key = tokenizer.getString();
    				keyList.add(key);
    				token = tokenizer.nextToken();
    				break;
    			default:
    				String propertyName = "_" + groupElementList.size();
    				GroupType groupType = null;
    				String groupedPropertyName = null;
    				switch (token) {
		    			case RuleTokenizer.COUNT:
		    				groupType = GroupType.COUNT;
		    				break;
		    			case RuleTokenizer.SUM:
		    				groupType = GroupType.SUM;
		    				break;
		    			case RuleTokenizer.MAX:
		    				groupType = GroupType.MAX;
		    				break;
		    			case RuleTokenizer.MIN:
		    				groupType = GroupType.MIN;
		    				break;
		    			case RuleTokenizer.AVG:
		    				groupType = GroupType.AVG;
		    				break;
		    			default:
		    				throw new SeasarException("ESSR0022",
		    					new Object[]{RuleTokenizer.getTokenName(token)});
    				}
    				tokenizer.nextToken(RuleTokenizer.LEFT_PAREN);
    				token = tokenizer.nextToken();
    				if (token != RuleTokenizer.MULTIPLY || !groupType.equals(GroupType.COUNT)) {
    					groupedPropertyName = tokenizer.getString();
    				}
    				tokenizer.nextToken(RuleTokenizer.RIGHT_PAREN);
    				token = tokenizer.nextToken();
    				switch (token) {
		    			case RuleTokenizer.AS:
		    				tokenizer.nextToken();
		    				propertyName = tokenizer.getString();
		    				token = tokenizer.nextToken();
		    				break;
		    			case RuleTokenizer.COMMA:
		    			case RuleTokenizer.EOF:
		    				break;
		    			default:
		    				propertyName = tokenizer.getString();
		    				token = tokenizer.nextToken();
		    				break;
    				}
    				GroupElement groupElement = new GroupElement(propertyName, groupType, groupedPropertyName);
    				groupElementList.add(groupElement);
    			
    		}
    		if (token == RuleTokenizer.COMMA) {
    			token = tokenizer.nextToken();
    		}
    	}
    	_keys = (String[]) keyList.toArray(new String[keyList.size()]);
    	_groupElements = (GroupElement[]) groupElementList.toArray(
    		new GroupElement[groupElementList.size()]);
    }

    public GroupDesc(String[] keys, GroupElement[] groupElements) {
    	Assertion.assertNotNull("keys", keys);
    	Assertion.assertNotNull("groupElements", groupElements);
    	
    	_keys = keys;
    	_groupElements = groupElements;
    }

    public String[] getKeys() {
        return _keys;
    }

    public GroupElement[] getGroupElements() {
        return _groupElements;
    }
    
    public List group(List items) throws SeasarException {
    	Assertion.assertNotNull("items", items);
    	
    	Calculator calculator = new Calculator(items);
    	return calculator.calculate();
    }

    private SortDesc getSortDesc() {
        SortElement[] sortElements = new SortElement[_keys.length];
        for (int i = 0; i < _keys.length; ++i) {
        	sortElements[i] = new SortElement(_keys[i]);
        }
        return new SortDesc(sortElements);
    }
    
    private GroupValue[] getGroupValues() {
        GroupValue[] groupValues = new GroupValue[_groupElements.length];
        for (int i = 0; i < _groupElements.length; ++i) {
        	groupValues[i] = _groupElements[i].newGroupValue();
        }
        return groupValues;
    }
    
    private final class Calculator {

	    private List _items;
	
	    public Calculator(final List items) {
	    	SortDesc sortDesc = getSortDesc();
	        _items = sortDesc.sort(items);
	    }
	
	    public List calculate() throws SeasarException {
	        List ret = new EArrayList();
	        if (_items.size() == 0) {
	        	return ret;
	        }
	        Object item = _items.get(0);
	        Object[] prevKeyValues = getKeyValues(item);
	        ValueCalculator valueCalculator = new ValueCalculator(prevKeyValues);
	    	valueCalculator.calculate(item);
	        for (int i = 1; i < _items.size(); ++i) {
	        	item = _items.get(i);
	            Object[] keyValues = getKeyValues(item);
	            if (!Arrays.equals(keyValues, prevKeyValues)) {
	                ret.add(valueCalculator.getGroupedItem());
	                prevKeyValues = keyValues;
	                valueCalculator = new ValueCalculator(prevKeyValues);
	            }
	            valueCalculator.calculate(item);
	        }
	        ret.add(valueCalculator.getGroupedItem());
	        return ret;
	    }

	    private Object[] getKeyValues(final Object item) throws SeasarException {
	        Object[] keyValues = new Object[_keys.length];
	        for (int i = 0; i < _keys.length; i++) {
	            keyValues[i] = NazunaUtil.getProperty(item, _keys[i]);
	        }
	        return keyValues;
	    }
	}
	
	private class ValueCalculator {
	
		private Object[] _keyValues;
        private GroupValue[] _groupValues;

        private ValueCalculator(Object[] keyValues) {
        	_keyValues = keyValues;
            _groupValues = getGroupValues();
        }

        void calculate(final Object item) throws SeasarException {
            for (int i = 0; i < _groupValues.length; i++) {
                _groupValues[i].calculate(item);
            }
        }

        EMap getGroupedItem() throws SeasarException {
        	EMap groupedItem = new EMap();
        	for (int i = 0; i < _keys.length; ++i) {
                groupedItem.put(_keys[i], _keyValues[i]);
            }
            for (int i = 0; i < _groupValues.length; ++i) {
            	groupedItem.put(_groupValues[i].getGroupElement().getPropertyName(),
            		_groupValues[i].getValue());
            }
            return groupedItem;
        }
    }
}