package org.seasar.hsqldb;

import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Statement;

import org.seasar.log.Logger;
import org.seasar.system.Lifecycle;
import org.seasar.timer.TimeoutManager;
import org.seasar.timer.TimeoutTarget;
import org.seasar.timer.TimeoutTask;
import org.seasar.util.SeasarException;
import org.seasar.util.ThreadUtil;


public final class HsqldbService implements Lifecycle, TimeoutTarget {

	private String _database = "test";
	private int _port = 9001;
	private boolean _trace = false;
	private boolean _silent = true;
	private boolean _compact = false;
	private int _retryCount = 10;
	private int _checkpointInterval = 0;
	private TimeoutTask _timeoutTask;
	
	static {
		try {
			Class.forName("org.hsqldb.jdbcDriver");
		} catch (Throwable t) {
			Logger.getLogger(HsqldbService.class).log(t);
		}
		
	}
	public final String getDatabase() {
		return _database;
	}
	
	public final void setDatabase(String database) {
		_database = database;
	}
	
	public final int getPort() {
		return _port;
	}
	
	public final void setPort(int port) {
		_port = port;
	}
	
	public final boolean isTrace() {
		return _trace;
	}
	
	public final void setTrace(boolean trace) {
		_trace = trace;
	}

	public final boolean isSilent() {
		return _silent;
	}
	
	public final void setSilent(boolean silent) {
		_silent = silent;
	}
	
	public final boolean isCompact() {
		return _compact;
	}

	public final void setCompact(boolean compact) {
		_compact = compact;
	}
	
	public final int getRetryCount() {
		return _retryCount;
	}

	public final void setRetryCount(int retryCount) {
		_retryCount = retryCount;
	}
	
	public final int getCheckpointInterval() {
		return _checkpointInterval;
	}

	public final void setCheckpointInterval(int checkpointInterval) {
		_checkpointInterval = checkpointInterval;
	}
	
	public final String[] createArgs() {
		String[] args = new String[10];
		args[0] = "-database";
		args[1] = System.getProperty("seasar.home") + File.separator +
			"data" + File.separator + _database;
		args[2] = "-port";
		args[3] = String.valueOf(_port);
		args[4] = "-trace";
		args[5] = String.valueOf(_trace);
		args[6] = "-silent";
		args[7] = String.valueOf(_silent);
		args[8] = "-no_system_exit";
		args[9] = String.valueOf(true);
		return args;
	}
	
	public HsqldbService() {
	}

	public void start() throws SeasarException {
		new HsqldbStarter(createArgs()).start();
		for (int i = 0; i < _retryCount; ++i) {
			ThreadUtil.sleep(500);
			try {
				execute("COMMIT");
				break;
			} catch (SQLException ignore) {
			}
		}
		if (_checkpointInterval > 0) {
			_timeoutTask = TimeoutManager.getInstance().addTimeoutTarget(
				this, _checkpointInterval, true);
		}
	}

	public void stop() {
		try { 
			Connection connection = getConnection();
			if (_compact) {
				execute(connection, "SHUTDOWN COMPACT");
			} else {
				execute(connection, "SHUTDOWN");
			}
		} catch (SQLException ex) {
			Logger.getLogger(getClass()).log(ex);
		}
		stopCheckpoint();
	}
	
	public void expired() {
		try {
			execute("CHECKPOINT");
			Logger.getLogger(getClass()).debug("HSQLDB CHECKPOINT");
		} catch (SQLException e) {
			Logger.getLogger(getClass()).log(e);
			stopCheckpoint();
		}
	}

	private void execute(String sql) throws SQLException { 
		Connection connection = getConnection();
		try {
			execute(connection, sql);
		} finally {
			connection.close();
		}
	}
	
	private void execute(Connection connection, String sql) throws SQLException {
		Statement statement = connection.createStatement();
		try {
			statement.execute(sql);
		} finally {
			statement.close();
		}
	}
	
	private Connection getConnection() throws SQLException {
		String url = "jdbc:hsqldb:hsql://localhost:" + _port; 
		return DriverManager.getConnection(url, "sa", "");
	}
	
	private void stopCheckpoint() {
		if (_timeoutTask != null) {
			_timeoutTask.stop();
			_timeoutTask = null;
		}
	}
}
