/*
 *
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:
 *    "This product includes software developed by the
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission of
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.seasar.axis.server.deployment;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.axis.Constants;
import org.apache.axis.deployment.wsdd.WSDDConstants;
import org.apache.axis.deployment.wsdd.WSDDException;
import org.apache.axis.deployment.wsdd.WSDDService;
import org.apache.axis.deployment.wsdd.WSDDTypeMapping;
import org.apache.axis.description.JavaServiceDesc;
import org.apache.axis.encoding.TypeMappingRegistry;
import org.apache.axis.providers.java.JavaProvider;
import org.apache.axis.wsdl.fromJava.Namespaces;
import org.apache.axis.wsdl.fromJava.Types;
import org.seasar.axis.S2AxisConstants;
import org.seasar.axis.encoding.AutoRegisterTypeMappingImpl;
import org.seasar.axis.server.ServiceDef;
import org.seasar.axis.server.TypeMappingDef;
import org.seasar.framework.container.ComponentDef;
import org.seasar.framework.message.MessageFormatter;
import org.seasar.framework.util.StringUtil;
import org.w3c.dom.Element;

/**
 * S2ɂCX^XǗT[rXWSDDɂ\łB
 * 
 * @author koichik
 */
public class WSDDS2Service extends WSDDService {
    //static fields
    protected static final Map providerMapping = new HashMap();
    static {
        providerMapping.put(PROVIDER_RPC, S2AxisConstants.PROVIDER_S2RPC);
        providerMapping.put(PROVIDER_RPC, S2AxisConstants.PROVIDER_S2RPC);
    }

    //instance fields
    protected ComponentDef componentDef;

    /**
     * R|[lg`CX^X\z܂B
     * 
     * @param componentDef
     *            R|[lg`
     * @throws WSDDException
     *             CX^X̍\zɎsꍇɃX[܂B
     */
    public WSDDS2Service(final ComponentDef componentDef) throws WSDDException {
        this(componentDef, new ServiceDef());
    }

    /**
     * R|[lg`ƃT[rX`CX^X\z܂B
     * 
     * @param componentDef
     *            R|[lg`
     * @param serviceDef
     *            T[rX`
     * @throws WSDDException
     *             CX^X̍\zɎsꍇɃX[܂B
     */
    public WSDDS2Service(final ComponentDef componentDef, final ServiceDef serviceDef)
            throws WSDDException {
        this.componentDef = componentDef;

        final String name = getName();
        setQName(new QName(name));

        final JavaServiceDesc serviceDesc = (JavaServiceDesc) getServiceDesc();
        serviceDesc.setName(name);

        Class serviceType = serviceDef.getServiceType();
        if (serviceType == null) {
            serviceType = componentDef.getComponentClass();
        }
        if (serviceType == null) {
            throw new WSDDException(MessageFormatter.getSimpleMessage("EAXS0006",
                    new Object[] { name }));
        }
        setParameter(JavaProvider.OPTION_CLASSNAME, serviceType.getName());

        setProviderQName(new QName(WSDDConstants.URI_WSDD_JAVA, getS2Provider(serviceDef
                .getProvider())));

        final String allowedMethods = serviceDef.getAllowedMethods();
        if (!StringUtil.isEmpty(allowedMethods)) {
            setParameter(JavaProvider.OPTION_ALLOWEDMETHODS, allowedMethods);
        }

        final Iterator typeMappingDefs = serviceDef.getTypeMappings();
        while (typeMappingDefs.hasNext()) {
            final TypeMappingDef typeMappingDef = (TypeMappingDef) typeMappingDefs.next();
            addTypeMapping(createWSDDTypeMapping(typeMappingDef));
        }

        initTMR();
        validateDescriptors();

        final TypeMappingRegistry tmr = serviceDesc.getTypeMappingRegistry();
        final AutoRegisterTypeMappingImpl autoTM = new AutoRegisterTypeMappingImpl(null);
        tmr.register(getUse().getEncoding(), autoTM);
        serviceDesc.setTypeMapping(autoTM);
    }

    /**
     * R|[lg`WSDDt@C(XML)DOMc[CX^X\z܂B <br>
     * WSDD <code>&lt;service&gt;</code> vf <code>provider</code>
     * Ŏw肳ꂽvoC_ΉS2Axis̃voC_ɕύX܂B
     * 
     * @param componentDef
     *            R|[lg`
     * @param serviceElement
     *            WSDDt@C(XML)DOMc[
     * @throws WSDDException
     *             CX^X̍\zɎsꍇɃX[܂B
     */
    public WSDDS2Service(final ComponentDef componentDef, final Element serviceElement)
            throws WSDDException {
        super(serviceElement);
        this.componentDef = componentDef;
        final String provider = getProviderQName().getLocalPart();
        setProviderQName(new QName(WSDDConstants.URI_WSDD_JAVA, getS2Provider(provider)));
    }

    /**
     * R|[lg`Ԃ܂B
     * 
     * @return R|[lg`
     */
    public ComponentDef getComponentDef() {
        return componentDef;
    }

    /**
     * T[rXԂ܂B <br>
     * T[rX́AR|[lg`ɖOԂw肳Ă"O/R|[lg"AłȂ"R|[lg"ɂȂ܂B
     * 
     * @return T[rX
     */
    protected String getName() {
        final String namespace = componentDef.getContainer().getNamespace();
        final String componentName = componentDef.getComponentName();
        if (StringUtil.isEmpty(namespace)) {
            return componentName;
        }
        return namespace + "/" + componentName;
    }

    /**
     * T[rX̌^Ԃ܂B <br>
     * T[rX̌^́AT[rX`Ɍ^w肳Ă΂̌^AłȂ΃R|[lǧ^ɂȂ܂B
     * 
     * @param serviceDef
     *            T[rX`
     * @return T[rX̌^
     */
    protected Class getServiceClass(final ServiceDef serviceDef) {
        Class serviceType = null;
        serviceType = serviceDef.getServiceType();
        if (serviceType == null) {
            serviceType = componentDef.getComponentClass();
        }
        return serviceType;
    }

    /**
     * Axis̃voC_ɑΉS2Axisp̃voC_Ԃ܂B
     * 
     * @param provider AxisW̃voC_
     * @return S2Axisp̃voC_
     * @throws WSDDException Axis̃voC_ΉS2Axis̃voC_ȂꍇɃX[܂
     */
    protected String getS2Provider(final String provider) throws WSDDException {
        final String s2Provider = (String) providerMapping.get(provider);
        if (s2Provider == null) {
            throw new WSDDException(provider);
        }
        return s2Provider;
    }

    /**
     * WSDD^Cv}bsO쐬ĕԂ܂B
     * 
     * @param typeMappingDef ^Cv}bsO`
     * @return WSDD^Cv}bsO
     */
    protected WSDDTypeMapping createWSDDTypeMapping(final TypeMappingDef typeMappingDef) {
        final WSDDTypeMapping wsddTypeMapping = new WSDDTypeMapping();

        final Class type = typeMappingDef.getType();
        wsddTypeMapping.setLanguageSpecificType(typeMappingDef.getType());

        wsddTypeMapping.setQName(createQNameOfType(type, typeMappingDef.getNamespaceURI(),
                typeMappingDef.getLocalPart(), typeMappingDef.getNamespacePrefix()));

        if (typeMappingDef.getSerializer() != null) {
            wsddTypeMapping.setSerializer(typeMappingDef.getSerializer());
        }
        if (typeMappingDef.getDeserializer() != null) {
            wsddTypeMapping.setDeserializer(typeMappingDef.getDeserializer());
        }

        if (typeMappingDef.getEncodingStyle() == null) {
            wsddTypeMapping.setEncodingStyle(Constants.URI_DEFAULT_SOAP_ENC);
        }
        else {
            wsddTypeMapping.setEncodingStyle(typeMappingDef.getEncodingStyle());
        }

        return wsddTypeMapping;
    }

    /**
     * XML^QName쐬ĕԂ܂B
     * 
     * @param type Java^
     * @param namespaceURI XML^̖OURIBȗJava^̃pbP[W瓱o܂
     * @param localPart XML^̃[JBȗJava^̃NXg܂
     * @param namespacePrefix XML^̖OԐړBȗƃftHgOԂɂȂ܂
     * @return XML^QName
     */
    protected QName createQNameOfType(final Class type, String namespaceURI, String localPart,
            final String namespacePrefix) {
        if (StringUtil.isEmpty(namespaceURI)) {
            namespaceURI = Namespaces.makeNamespace(type.getName());
        }

        if (StringUtil.isEmpty(localPart)) {
            localPart = Types.getLocalNameFromFullName(type.getName());
        }

        return new QName(namespaceURI, localPart, namespacePrefix);
    }
}