package org.seasar.dao.impl;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import org.seasar.dao.BeanMetaData;
import org.seasar.dao.DaoMetaData;
import org.seasar.dao.Dbms;
import org.seasar.dao.IllegalSignatureRuntimeException;
import org.seasar.dao.SqlCommand;
import org.seasar.dao.dbms.DbmsManager;
import org.seasar.extension.jdbc.ResultSetHandler;
import org.seasar.extension.jdbc.impl.ObjectResultSetHandler;
import org.seasar.framework.beans.BeanDesc;
import org.seasar.framework.beans.MethodNotFoundRuntimeException;
import org.seasar.framework.beans.factory.BeanDescFactory;
import org.seasar.framework.util.ClassUtil;
import org.seasar.framework.util.ConnectionUtil;
import org.seasar.framework.util.DataSourceUtil;
import org.seasar.framework.util.DatabaseMetaDataUtil;
import org.seasar.framework.util.FieldUtil;
import org.seasar.framework.util.MethodUtil;
import org.seasar.framework.util.ResourceUtil;
import org.seasar.framework.util.StringUtil;
import org.seasar.framework.util.TextUtil;

/**
 * @author higa
 *  
 */
public class DaoMetaDataImpl implements DaoMetaData {

	private static final String[] INSERT_NAMES = new String[] { "insert",
			"create", "add" };

	private static final String[] UPDATE_NAMES = new String[] { "update",
			"modify", "store" };

	private static final String[] DELETE_NAMES = new String[] { "delete",
			"remove" };

	private Class daoClass_;

	private BeanDesc daoBeanDesc_;

	private DataSource dataSource_;

	private Dbms dbms_;

	private Class beanClass_;

	private BeanMetaData beanMetaData_;

	private Map sqlCommands_ = new HashMap();

	public DaoMetaDataImpl(Class daoClass, DataSource dataSource) {
		daoClass_ = daoClass;
		daoBeanDesc_ = BeanDescFactory.getBeanDesc(daoClass);
		Field beanField = daoBeanDesc_.getField(BEAN_KEY);
		beanClass_ = (Class) FieldUtil.get(beanField, null);
		dataSource_ = dataSource;
		Connection con = DataSourceUtil.getConnection(dataSource_);
		try {
			DatabaseMetaData dbMetaData = ConnectionUtil.getMetaData(con);
			beanMetaData_ = new BeanMetaDataImpl(beanClass_, dbMetaData);
			String productName = DatabaseMetaDataUtil
					.getDatabaseProductName(dbMetaData);
			dbms_ = DbmsManager.getDbms(productName);
		} finally {
			ConnectionUtil.close(con);
		}
		setupSqlCommand();
	}

	private void setupSqlCommand() {
		String[] names = daoBeanDesc_.getMethodNames();
		for (int i = 0; i < names.length; ++i) {
			Method[] methods = daoBeanDesc_.getMethods(names[i]);
			if (methods.length == 1 && MethodUtil.isAbstract(methods[0])) {
				setupMethod(methods[0]);
			}
		}
	}

	private void setupMethod(Method method) {
		String base = daoClass_.getName().replace('.', '/') + "_"
				+ method.getName();
		String dbmsPath = base + dbms_.getSuffix() + ".sql";
		String standardPath = base + ".sql";
		if (ResourceUtil.isExist(dbmsPath)) {
			String sql = TextUtil.readText(dbmsPath);
			setupMethodByManual(method, sql);
		} else if (ResourceUtil.isExist(standardPath)) {
			String sql = TextUtil.readText(standardPath);
			setupMethodByManual(method, sql);
		} else {
			setupMethodByAuto(method);
		}
	}

	private void setupMethodByManual(Method method, String sql) {
		if (isSelect(method)) {
			setupSelectMethodByManual(method, sql);
		} else {
			setupUpdateMethodByManual(method, sql);
		}
	}

	private void setupMethodByAuto(Method method) {
		if (isInsert(method.getName())) {
			setupInsertMethodByAuto(method);
		} else if (isUpdate(method.getName())) {
			setupUpdateMethodByAuto(method);
		} else if (isDelete(method.getName())) {
			setupDeleteMethodByAuto(method);
		} else {
			setupSelectMethodByAuto(method);
		}
	}

	private void setupSelectMethodByManual(Method method, String sql) {
		SelectDynamicCommand cmd = new SelectDynamicCommand(dataSource_,
				createResultSetHandler(method));
		cmd.setSql(sql);
		cmd.setArgNames(getArgNames(method.getName()));
		sqlCommands_.put(method.getName(), cmd);
	}

	private ResultSetHandler createResultSetHandler(Method method) {
		if (method.getReturnType().isAssignableFrom(List.class)) {
			return new BeanListMetaDataResultSetHandler(beanMetaData_);
		} else if (method.getReturnType().isAssignableFrom(beanClass_)) {
			return new BeanMetaDataResultSetHandler(beanMetaData_);
		} else {
			return new ObjectResultSetHandler();
		}
	}

	private void setupUpdateMethodByManual(Method method, String sql) {
		UpdateDynamicCommand cmd = new UpdateDynamicCommand(dataSource_);
		cmd.setSql(sql);
		String[] argNames = getArgNames(method.getName());
		if (argNames.length == 0 && method.getParameterTypes().length == 1
				&& method.getParameterTypes()[0].isAssignableFrom(beanClass_)) {
			argNames = new String[] { StringUtil.decapitalize(ClassUtil
					.getShortClassName(beanClass_)) };
		}
		cmd.setArgNames(argNames);
		sqlCommands_.put(method.getName(), cmd);
	}

	private void setupInsertMethodByAuto(Method method) {
		checkAutoUpdateMethod(method);
		SqlCommand cmd = null;
		if (method.getParameterTypes()[0].isAssignableFrom(beanClass_)) {
			cmd = new InsertAutoStaticCommand(dataSource_, beanMetaData_);
		} else {
			cmd = new InsertBatchAutoStaticCommand(dataSource_, beanMetaData_);
		}
		sqlCommands_.put(method.getName(), cmd);
	}

	private void setupUpdateMethodByAuto(Method method) {
		checkAutoUpdateMethod(method);
		SqlCommand cmd = null;
		if (method.getParameterTypes()[0].isAssignableFrom(beanClass_)) {
			cmd = new UpdateAutoStaticCommand(dataSource_, beanMetaData_);
		} else {
			cmd = new UpdateBatchAutoStaticCommand(dataSource_, beanMetaData_);
		}
		sqlCommands_.put(method.getName(), cmd);
	}

	private void setupDeleteMethodByAuto(Method method) {
		checkAutoUpdateMethod(method);
		SqlCommand cmd = null;
		if (method.getParameterTypes()[0].isAssignableFrom(beanClass_)) {
			cmd = new DeleteAutoStaticCommand(dataSource_, beanMetaData_);
		} else {
			cmd = new DeleteBatchAutoStaticCommand(dataSource_, beanMetaData_);
		}
		sqlCommands_.put(method.getName(), cmd);
	}

	private void setupSelectMethodByAuto(Method method) {
		String query = getQuery(method.getName());
		ResultSetHandler handler = createResultSetHandler(method);
		SelectDynamicCommand cmd = null;
		if (query != null) {
			cmd = createSelectCommand(query, handler);
		} else {
			cmd = new SelectDynamicCommand(dataSource_, handler);
			String[] argNames = getArgNames(method.getName());
			cmd.setArgNames(argNames);
			String sql = dbms_.getAutoSelectSql(getBeanMetaData());
			StringBuffer buf = new StringBuffer(sql);
			if (argNames.length != 0) {
				boolean began = false;
				if (!(sql.lastIndexOf("WHERE") > 0)) {
					buf.append("/*BEGIN*/ WHERE ");
					began = true;
				}
				for (int i = 0; i < argNames.length; ++i) {
					String columnName = beanMetaData_
							.convertFullColumnName(argNames[i]);
					buf.append("/*IF ");
					buf.append(argNames[i]);
					buf.append(" != null*/");
					buf.append(" ");
					if (!began || i != 0) {
						buf.append("AND ");
					}
					buf.append(columnName);
					buf.append(" = /*");
					buf.append(argNames[i]);
					buf.append("*/null");
					buf.append("/*END*/");
				}
				if (began) {
					buf.append("/*END*/");
				}
			}
			cmd.setSql(buf.toString());
		}
		sqlCommands_.put(method.getName(), cmd);
	}

	private void checkAutoUpdateMethod(Method method) {
		if (method.getParameterTypes().length != 1
				|| !method.getParameterTypes()[0].isAssignableFrom(beanClass_)
				&& !method.getParameterTypes()[0].isAssignableFrom(List.class)
				&& !method.getParameterTypes()[0].isArray()) {
			throw new IllegalSignatureRuntimeException("EDAO0006", method
					.toString());
		}
	}

	private boolean isSelect(Method method) {
		if (isInsert(method.getName())) {
			return false;
		}
		if (isUpdate(method.getName())) {
			return false;
		}
		if (isDelete(method.getName())) {
			return false;
		}
		return true;
	}

	private boolean isInsert(String methodName) {
		for (int i = 0; i < INSERT_NAMES.length; ++i) {
			if (methodName.startsWith(INSERT_NAMES[i])) {
				return true;
			}
		}
		return false;
	}

	private boolean isUpdate(String methodName) {
		for (int i = 0; i < UPDATE_NAMES.length; ++i) {
			if (methodName.startsWith(UPDATE_NAMES[i])) {
				return true;
			}
		}
		return false;
	}

	private boolean isDelete(String methodName) {
		for (int i = 0; i < DELETE_NAMES.length; ++i) {
			if (methodName.startsWith(DELETE_NAMES[i])) {
				return true;
			}
		}
		return false;
	}

	private String[] getArgNames(String methodName) {
		String argsKey = methodName + ARGS_KEY_SUFFIX;
		if (daoBeanDesc_.hasField(argsKey)) {
			Field argNamesField = daoBeanDesc_.getField(argsKey);
			String argNames = (String) FieldUtil.get(argNamesField, null);
			return StringUtil.split(argNames, " ,");
		} else {
			return new String[0];
		}
	}

	private String getQuery(String methodName) {
		String key = methodName + QUERY_KEY_SUFFIX;
		if (daoBeanDesc_.hasField(key)) {
			Field queryField = daoBeanDesc_.getField(key);
			return (String) FieldUtil.get(queryField, null);
		} else {
			return null;
		}
	}

	/**
	 * @see org.seasar.dao.DaoMetaData#getBeanClass()
	 */
	public Class getBeanClass() {
		return daoClass_;
	}

	/**
	 * @see org.seasar.dao.DaoMetaData#getBeanMetaData()
	 */
	public BeanMetaData getBeanMetaData() {
		return beanMetaData_;
	}

	/**
	 * @see org.seasar.dao.DaoMetaData#getSqlCommand(java.lang.String)
	 */
	public SqlCommand getSqlCommand(String methodName)
			throws MethodNotFoundRuntimeException {

		SqlCommand cmd = (SqlCommand) sqlCommands_.get(methodName);
		if (cmd == null) {
			throw new MethodNotFoundRuntimeException(daoClass_, methodName,
					null);
		}
		return cmd;
	}

	/**
	 * @see org.seasar.dao.DaoMetaData#hasSqlCommand(java.lang.String)
	 */
	public boolean hasSqlCommand(String methodName) {
		return sqlCommands_.containsKey(methodName);
	}

	/**
	 * @see org.seasar.dao.DaoMetaData#createFindCommand(java.lang.String)
	 */
	public SqlCommand createFindCommand(String query) {
		return createSelectCommand(query, new BeanListMetaDataResultSetHandler(
				beanMetaData_));
	}

	/**
	 * @see org.seasar.dao.DaoMetaData#createFindBeanCommand(java.lang.String)
	 */
	public SqlCommand createFindBeanCommand(String query) {
		return createSelectCommand(query, new BeanMetaDataResultSetHandler(
				beanMetaData_));
	}

	/**
	 * @see org.seasar.dao.DaoMetaData#createFindObjectCommand(java.lang.String)
	 */
	public SqlCommand createFindObjectCommand(String query) {
		return createSelectCommand(query, new ObjectResultSetHandler());
	}

	private SelectDynamicCommand createSelectCommand(String query,
			ResultSetHandler resultSetHandler) {

		SelectDynamicCommand cmd = new SelectDynamicCommand(dataSource_,
				resultSetHandler);
		StringBuffer buf = new StringBuffer(255);
		if (query != null && query.length() > 6
				&& query.substring(0, 6).toLowerCase().equals("select")) {
			buf.append(query);
		} else {
			String sql = dbms_.getAutoSelectSql(getBeanMetaData());
			buf.append(sql);
			if (query != null) {
				if (query.length() > 8
						&& query.substring(0, 8).toLowerCase().equals(
								"order by")) {
					buf.append(" ");
				} else if (sql.lastIndexOf("WHERE") < 0) {
					buf.append(" WHERE ");
				}
				buf.append(query);
			}
		}
		cmd.setSql(buf.toString());
		return cmd;
	}
}