/************************************************************
Copyright (C) 2005 Masahiko SAWAI All Rights Reserved. 
************************************************************/
#include "sdl4gcj/video/Screen.h"
#include "sdl4gcj/video/PixelFormat.h"
#include "sdl4gcj/video/VideoInfo.h"
#include "sdl4gcj/video/Rect.h"
#include "sdl4gcj/video/Rectangle.h"
#include "sdl4gcj/SDLException.h"

#include <java/lang/String.h>

#include <stdlib.h>
#include <gcj/cni.h>
#include <SDL.h>
#include <SDL_opengl.h>


using namespace sdl4gcj::video;
using sdl4gcj::SDLException;

// functions

static
inline
SDLException*
createIllegalStateScreenException()
{
	return new SDLException(JvNewStringLatin1("This Screen has Illegal state."));
}

// class methods
JArray<Rect *> *
Screen::listModes (::sdl4gcj::video::PixelFormat *format, jint flags)
{
	fprintf(stderr, "hello\n");
	JvInitClass(&Screen::class$);
	JArray<Rect *> *modeArray = NULL;

	SDL_PixelFormat* nativeFormat = NULL;
	if (format != NULL)
		nativeFormat = (SDL_PixelFormat*)format->implementation;

	SDL_Rect** nativeModes = SDL_ListModes(nativeFormat, flags);
	if (nativeModes != NULL)
	{
		int length = 0;
		if (nativeModes != (SDL_Rect**)-1)
		{
			while (nativeModes[length]) length++; 
		}

		modeArray = (JArray<Rect *> *)JvNewObjectArray(length, &Rect::class$, NULL);
		Rect** modes = elements(modeArray);
		for (int i = 0;i < length;i++)
		{
			modes[i] = (Rect*)new Rectangle(0, 0,
				nativeModes[i]->w, nativeModes[i]->h);
		}
	}

	return modeArray;
}

jint
Screen::videoModeOK(jint width, jint height, jint bpp, jint flags)
{
	JvInitClass(&Screen::class$);
	return ::SDL_VideoModeOK(width, height, bpp, flags);
}

Screen* 
Screen::setVideoMode (jint width, jint height, jint bpp, jint flags)
{
	JvInitClass(&Screen::class$);
	Screen* screen = NULL;

	SDL_Surface* nativeScreen = ::SDL_SetVideoMode(width, height, bpp, flags);
	if (nativeScreen == NULL)
		throw new SDLException();
	screen = Screen::getVideoSurface();

	return screen;
}

Screen* 
Screen::getVideoSurface ()
{
	JvInitClass(&Screen::class$);
	SDL_Surface* nativeScreen = SDL_GetVideoSurface();

	if (nativeScreen != NULL)
	{
		if (singleton == NULL || ((SDL_Surface*)singleton->implementation) != nativeScreen)
		{
			singleton = new Screen();
			singleton->implementation = (::gnu::gcj::RawData*)nativeScreen;
		}
	}
	return singleton;
}

VideoInfo* 
Screen::getVideoInfo ()
{
	JvInitClass(&Screen::class$);
	VideoInfo* videoInfo = NULL;
	const SDL_VideoInfo* native = NULL;

	native = ::SDL_GetVideoInfo();
	videoInfo = new VideoInfo();
	videoInfo->implementation = (::gnu::gcj::RawData*)native;

	return videoInfo;
}

void 
Screen::setGamma (jfloat redgamma, jfloat greengamma, jfloat bluegamma)
{
	JvInitClass(&Screen::class$);

	int result = ::SDL_SetGamma(redgamma, greengamma, bluegamma);
	if (result == -1)
		throw new SDLException();

	return ;
}


void 
Screen::setGammaRamp (jintArray redArray, jintArray greenArray, jintArray blueArray)
{
	JvInitClass(&Screen::class$);

	Uint16* arguments[3] = {NULL, NULL, NULL};
	Uint16 nativeRamp[3][256];
	jintArray ramp[3];
	ramp[0] = redArray;
	ramp[1] = greenArray;
	ramp[2] = blueArray;

	for (int i = 0;i < 3;i++)
	{
		if (ramp[i] != NULL)
		{
			jint* a = elements(ramp[i]);
			int length = JvGetArrayLength(ramp[i]);
			if (length > 256) length = 256;
			for (int j = 0;j < 256;j++) nativeRamp[i][j] = 0;
			for (int j = 0;j < length;j++) nativeRamp[i][j] = a[j];
			arguments[i] = nativeRamp[i];
		}
	}

	int result = ::SDL_SetGammaRamp(arguments[0], 
		arguments[1], arguments[2]);
	if (result == -1)
		throw new SDLException();

	return ;
}


jintArray 
Screen::getRedGammaRamp ()
{
	JvInitClass(&Screen::class$);

	Uint16 redTable[256];
	::SDL_GetGammaRamp(redTable, NULL, NULL);

	jintArray gammaRampArray = JvNewIntArray(256);
	jint* gammaRamp = elements(gammaRampArray);
	for (int i = 0;i < 256;i++)
		gammaRamp[i] = (jint)redTable[i];

	return gammaRampArray;
}

jintArray
Screen::getGreenGammaRamp ()
{
	JvInitClass(&Screen::class$);

	Uint16 greenTable[256];
	::SDL_GetGammaRamp(NULL, greenTable, NULL);

	jintArray gammaRampArray = JvNewIntArray(256);
	jint* gammaRamp = elements(gammaRampArray);
	for (int i = 0;i < 256;i++)
		gammaRamp[i] = (jint)greenTable[i];

	return gammaRampArray;
}

jintArray
Screen::getBlueGammaRamp ()
{
	JvInitClass(&Screen::class$);

	Uint16 blueTable[256];
	::SDL_GetGammaRamp(NULL, NULL, blueTable);

	jintArray gammaRampArray = JvNewIntArray(256);
	jint* gammaRamp = elements(gammaRampArray);
	for (int i = 0;i < 256;i++)
		gammaRamp[i] = (jint)blueTable[i];

	return gammaRampArray;
}


void
Screen::setCaption (jstring windowCaption, jstring iconCaption)
{
	if (windowCaption != NULL)
	{
		jbyteArray byteArray = windowCaption->getBytes();
		jbyte* bytes = elements(byteArray);
		int length = JvGetArrayLength(byteArray);
		char buffer[length+1];

		memcpy(buffer, bytes, length);
		buffer[length] = '\0';
		SDL_WM_SetCaption(buffer, NULL);
	}

	if (iconCaption != NULL)
	{
		jbyteArray byteArray = iconCaption->getBytes();
		jbyte* bytes = elements(byteArray);
		int length = JvGetArrayLength(byteArray);
		char buffer[length+1];

		memcpy(buffer, bytes, length);
		buffer[length] = '\0';
		SDL_WM_SetCaption(NULL, buffer);
	}
}


jstring
Screen::getWindowCaption()
{
	jstring windowCaption = NULL;
	char* nativeWindowCaption;

	SDL_WM_GetCaption(&nativeWindowCaption, NULL);
	if (nativeWindowCaption != NULL)
		windowCaption = JvNewStringLatin1(nativeWindowCaption);
	
	return windowCaption;
}

jstring
Screen::getIconCaption()
{
	jstring iconCaption = NULL;
	char* nativeIconCaption;

	SDL_WM_GetCaption(NULL, &nativeIconCaption);
	if (nativeIconCaption != NULL)
		iconCaption = JvNewStringLatin1(nativeIconCaption);

	return iconCaption;
}

void
Screen::setIcon(Surface* icon, jbyteArray mask)
{
	SDL_Surface* nativeIcon = NULL;
	if (icon != NULL)
	{
		if (icon->implementation == NULL)
			throw new SDLException(JvNewStringLatin1("This Surface was already freed."));
		nativeIcon = (SDL_Surface*)icon->implementation;
	}

	Uint8* nativeMask = NULL;
	if (mask != NULL)
	{
		int length = (nativeIcon->w + 7) / 8 * nativeIcon->h;
		if(JvGetArrayLength(mask) < length)
			throw new SDLException(JvNewStringLatin1("The mask array size is too small."));
		nativeMask = (Uint8*)elements(mask);
	}

	SDL_WM_SetIcon(nativeIcon, nativeMask);
}

void
Screen::glSetAttribute (jint attr, jint value)
{
	::SDL_GL_SetAttribute((SDL_GLattr)attr, value);
}


jint
Screen::glGetAttribute (jint attr)
{
	int value;
	::SDL_GL_GetAttribute((SDL_GLattr)attr, &value);

	return (jint)value;
}


// instance methods

void 
Screen::updateRect (jint x, jint y, jint w, jint h)
{
	if (this->implementation == NULL) throw createIllegalStateScreenException();
	SDL_Surface* nativeScreen = (SDL_Surface*)this->implementation;

	::SDL_UpdateRect(nativeScreen, x, y, w, h);
}


void 
Screen::updateRects (JArray< ::sdl4gcj::video::Rect *> *rectArray)
{
	if (this->implementation == NULL) throw createIllegalStateScreenException();
	SDL_Surface* nativeScreen = (SDL_Surface*)this->implementation;

	int length = JvGetArrayLength(rectArray);
	Rect** rects = elements(rectArray);
	SDL_Rect nativeRects[length];

	for (int i = length-1;i >= 0;i--)
	{
		if (rects[i] != NULL)
		{
			nativeRects[i].x = rects[i]->getX();
			nativeRects[i].y = rects[i]->getY();
			nativeRects[i].w = rects[i]->getW();
			nativeRects[i].h = rects[i]->getH();
		}
		else
		{
			nativeRects[i].x = 0;
			nativeRects[i].y = 0;
			nativeRects[i].w = 0;
			nativeRects[i].h = 0;
		}
	}

	::SDL_UpdateRects(nativeScreen, length, nativeRects);
}

void 
Screen::flip()
{
	if (this->implementation == NULL) throw createIllegalStateScreenException();
	SDL_Surface* nativeScreen = (SDL_Surface*)this->implementation;

	int result = ::SDL_Flip(nativeScreen);
	if (result == -1) new SDLException();
}

void
Screen::glSwapBuffers ()
{
	::SDL_GL_SwapBuffers();
}

void
Screen::iconifyWindow ()
{
	::SDL_WM_IconifyWindow();
}

jint
Screen::toggleFullScreen ()
{
	if (this->implementation == NULL) throw createIllegalStateScreenException();
	SDL_Surface* nativeScreen = (SDL_Surface*)this->implementation;
	return ::SDL_WM_ToggleFullScreen(nativeScreen);
}

jboolean
Screen::grabInput (jboolean grabMode)
{
	return (::SDL_WM_GrabInput(grabMode ? SDL_GRAB_ON : SDL_GRAB_OFF) == SDL_GRAB_ON);
}

void
Screen::setGrabInputMode (jboolean grabMode)
{
	::SDL_WM_GrabInput(grabMode ? SDL_GRAB_ON : SDL_GRAB_OFF);
}

jboolean 
Screen::getGrabInputMode ()
{
	return (::SDL_WM_GrabInput(SDL_GRAB_QUERY) == SDL_GRAB_ON);
}

