#include "java/lang/String.h"
#include "sdl4gcj/video/Screen.h"
#include "sdl4gcj/video/VideoInfo.h"
#include "sdl4gcj/video/Rect.h"
#include "sdl4gcj/SDLException.h"

#include <stdlib.h>
#include <gcj/cni.h>
#include <SDL.h>
#include <SDL_opengl.h>


using namespace sdl4gcj::video;

// functions
static
inline
sdl4gcj::SDLException*
createFreedSurfaceException()
{
	return new sdl4gcj::SDLException(JvNewStringLatin1("This Surface was already freed."));
}

static
inline
sdl4gcj::SDLException*
createIllegalStateScreenException()
{
	return new sdl4gcj::SDLException(JvNewStringLatin1("This Screen has Illegal state."));
}

// class methods
Screen* 
Screen::setVideoMode (jint width, jint height, jint bpp, jint flags)
{
	JvInitClass(&Screen::class$);
	Screen* result = NULL;

	SDL_Surface* screen = ::SDL_SetVideoMode(width, height, bpp, flags);
	if (screen != NULL)
		result = Screen::getVideoSurface();
	return result;
}

Screen* 
Screen::getVideoSurface ()
{
	JvInitClass(&Screen::class$);
	SDL_Surface* screen = SDL_GetVideoSurface();

	if (screen != NULL)
	{
		if (singleton == NULL || ((SDL_Surface*)singleton->implementation) != screen)
		{
			singleton = new Screen();
			singleton->implementation = (::gnu::gcj::RawData*)screen;
		}
	}
	return singleton;
}

VideoInfo* 
Screen::getVideoInfo ()
{
	JvInitClass(&Screen::class$);
	VideoInfo* videoInfo = NULL;
	const SDL_VideoInfo* native = NULL;

	native = ::SDL_GetVideoInfo();
	videoInfo = new VideoInfo();
	videoInfo->implementation = (::gnu::gcj::RawData*)native;

	return videoInfo;
}

void 
Screen::setGamma (jfloat redgamma, jfloat greengamma, jfloat bluegamma)
{
	JvInitClass(&Screen::class$);

	int result = ::SDL_SetGamma(redgamma, greengamma, bluegamma);
	if (result == -1) throw new SDLException();

	return ;
}


void 
Screen::setGammaRamp (jintArray redArray, jintArray greenArray, jintArray blueArray)
{
	JvInitClass(&Screen::class$);

	Uint16* arguments[3] = {NULL, NULL, NULL};
	Uint16 nativeRamp[3][256];
	jintArray ramp[3];
	ramp[0] = redArray;
	ramp[1] = greenArray;
	ramp[2] = blueArray;

	for (int i = 0;i < 3;i++)
	{
		if (ramp[i] != NULL)
		{
			jint* a = elements(ramp[i]);
			int length = JvGetArrayLength(ramp[i]);
			if (length > 256) length = 256;
			for (int j = 0;j < 256;j++) nativeRamp[i][j] = 0;
			for (int j = 0;j < length;j++) nativeRamp[i][j] = a[j];
			arguments[i] = nativeRamp[i];
		}
	}

	int result = ::SDL_SetGammaRamp(arguments[0], 
		arguments[1], arguments[2]);
	if (result == -1) throw new SDLException();

	return ;
}


jintArray 
Screen::getRedGammaRamp ()
{
	JvInitClass(&Screen::class$);

	Uint16 redTable[256];
	::SDL_GetGammaRamp(redTable, NULL, NULL);

	jintArray gammaRampArray = JvNewIntArray(256);
	jint* gammaRamp = elements(gammaRampArray);
	for (int i = 0;i < 256;i++)
		gammaRamp[i] = (jint)redTable[i];

	return gammaRampArray;
}

jintArray
Screen::getGreenGammaRamp ()
{
	JvInitClass(&Screen::class$);

	Uint16 greenTable[256];
	::SDL_GetGammaRamp(NULL, greenTable, NULL);

	jintArray gammaRampArray = JvNewIntArray(256);
	jint* gammaRamp = elements(gammaRampArray);
	for (int i = 0;i < 256;i++)
		gammaRamp[i] = (jint)greenTable[i];

	return gammaRampArray;
}

jintArray
Screen::getBlueGammaRamp ()
{
	JvInitClass(&Screen::class$);

	Uint16 blueTable[256];
	::SDL_GetGammaRamp(NULL, NULL, blueTable);

	jintArray gammaRampArray = JvNewIntArray(256);
	jint* gammaRamp = elements(gammaRampArray);
	for (int i = 0;i < 256;i++)
		gammaRamp[i] = (jint)blueTable[i];

	return gammaRampArray;
}


void
Screen::setCaption (jstring windowCaption, jstring iconCaption)
{
	if (windowCaption != NULL)
	{
		jbyteArray byteArray = windowCaption->getBytes();
		jbyte* bytes = elements(byteArray);
		int length = byteArray->length;
		char buffer[length+1];

		memcpy(buffer, bytes, length);
		buffer[length] = '\0';
		SDL_WM_SetCaption(buffer, NULL);
	}

	if (iconCaption != NULL)
	{
		jbyteArray byteArray = iconCaption->getBytes();
		jbyte* bytes = elements(byteArray);
		int length = byteArray->length;
		char buffer[length+1];

		memcpy(buffer, bytes, length);
		buffer[length] = '\0';
		SDL_WM_SetCaption(NULL, buffer);
	}
}


jstring
Screen::getWindowCaption()
{
	jstring windowCaption = NULL;
	char* nativeWindowCaption;

	SDL_WM_GetCaption(&nativeWindowCaption, NULL);
	if (nativeWindowCaption != NULL)
		windowCaption = JvNewStringLatin1(nativeWindowCaption);
	
	return windowCaption;
}

jstring
Screen::getIconCaption()
{
	jstring iconCaption = NULL;
	char* nativeIconCaption;

	SDL_WM_GetCaption(NULL, &nativeIconCaption);
	if (nativeIconCaption != NULL)
		iconCaption = JvNewStringLatin1(nativeIconCaption);

	return iconCaption;
}

void
Screen::setIcon(Surface* icon, jbyteArray mask)
{
	SDL_Surface* nativeIcon = NULL;
	if (icon != NULL)
	{
		if (icon->implementation == NULL) throw createFreedSurfaceException();
		nativeIcon = (SDL_Surface*)icon->implementation;
	}

	Uint8* nativeMask = NULL;
	if (mask != NULL)
	{
		int length = (nativeIcon->w + 7) / 8 * nativeIcon->h;
		//printf("nativeIcon->w : %d, nativeIcon->h : %d\n", nativeIcon->w, nativeIcon->h);
		//printf("length %d, mask->length %d\n", length, mask->length);
		if(mask->length < length)
			throw new SDLException(JvNewStringLatin1("The mask array size is too small."));
		nativeMask = (Uint8*)elements(mask);
	}

	SDL_WM_SetIcon(nativeIcon, nativeMask);
}

void
Screen::glSetAttribute (jint attr, jint value)
{
	::SDL_GL_SetAttribute((SDL_GLattr)attr, value);
}


jint
Screen::glGetAttribute (jint attr)
{
	int value;
	::SDL_GL_GetAttribute((SDL_GLattr)attr, &value);

	return (jint)value;
}


// instance methods

void 
Screen::updateRect (jint x, jint y, jint w, jint h)
{
	if (this->implementation == NULL) throw createIllegalStateScreenException();
	SDL_Surface* nativeScreen = (SDL_Surface*)this->implementation;

	::SDL_UpdateRect(nativeScreen, x, y, w, h);
}


void 
Screen::updateRects (JArray< ::sdl4gcj::video::Rect *> *rectArray)
{
	if (this->implementation == NULL) throw createIllegalStateScreenException();
	SDL_Surface* nativeScreen = (SDL_Surface*)this->implementation;

	int length = JvGetArrayLength(rectArray);
	Rect** rects = elements(rectArray);
	SDL_Rect nativeRects[length];

	for (int i = length-1;i >= 0;i--)
	{
		if (rects[i] != NULL)
		{
			nativeRects[i].x = rects[i]->getX();
			nativeRects[i].y = rects[i]->getY();
			nativeRects[i].w = rects[i]->getW();
			nativeRects[i].h = rects[i]->getH();
		}
		else
		{
			nativeRects[i].x = 0;
			nativeRects[i].y = 0;
			nativeRects[i].w = 0;
			nativeRects[i].h = 0;
		}
	}

	::SDL_UpdateRects(nativeScreen, length, nativeRects);
}

void 
Screen::flip()
{
	if (this->implementation == NULL) throw createIllegalStateScreenException();
	SDL_Surface* nativeScreen = (SDL_Surface*)this->implementation;

	int result = ::SDL_Flip(nativeScreen);
	if (result == -1) new SDLException();
}

void
Screen::glSwapBuffers ()
{
	::SDL_GL_SwapBuffers();
}

void
Screen::iconifyWindow ()
{
	::SDL_WM_IconifyWindow();
}

jint
Screen::toggleFullScreen ()
{
	if (this->implementation == NULL) throw createIllegalStateScreenException();
	SDL_Surface* nativeScreen = (SDL_Surface*)this->implementation;
	::SDL_WM_ToggleFullScreen(nativeScreen);
}

jboolean
Screen::grabInput (jboolean grabMode)
{
	return (::SDL_WM_GrabInput(grabMode ? SDL_GRAB_ON : SDL_GRAB_OFF) == SDL_GRAB_ON);
}

void
Screen::setGrabInputMode (jboolean grabMode)
{
	::SDL_WM_GrabInput(grabMode ? SDL_GRAB_ON : SDL_GRAB_OFF);
}

jboolean 
Screen::getGrabInputMode ()
{
	return (::SDL_WM_GrabInput(SDL_GRAB_QUERY) == SDL_GRAB_ON);
}

