/************************************************************
Copyright (C) 2004-2005  Masahiko SAWAI 
************************************************************/

import sdl4gcj.*;
import sdl4gcj.video.*;
import sdl4gcj.event.*;
import sdl4gcj.input.*;
import gl4gcj.*;
import java.util.Random;

class Sprite extends GL
{
	private static final int MAX_SPEED = 1;
	private static final int SIZE = 32;
	private static final Random random = new Random();

	private int movableW, movableH;

	private int x, y, hw, hh, dx = 0, dy = 0;
	private int texture;

	Sprite(int texture, int screenW, int screenH)
	{
		this.texture = texture;

		// init size
		this.hw = SIZE/2;
		this.hh = SIZE/2;
		this.movableW = screenW - SIZE;
		this.movableH = screenH - SIZE;

		// init location
		this.x = random.nextInt(movableW);
		this.y = random.nextInt(movableH);

		// init speed
		while(dx == 0 && dy == 0)
		{
			dx = random.nextInt(MAX_SPEED * 2 + 1) - MAX_SPEED;
			dy = random.nextInt(MAX_SPEED * 2 + 1) - MAX_SPEED;
		}
	}

	public final void move()
	{
		x += dx;
		if((x < 0) || (x >= movableW))
		{
			dx = -dx;
			x += dx;
		}

		y += dy;
		if((y < 0) || (y >= movableH))
		{
			dy = -dy;
			y += dy;
		}

	}

	public final void draw()
	{
		glLoadIdentity();
		glTranslated(x + hw, y + hh, 0);

		glBindTexture(GL_TEXTURE_2D, this.texture);
		glBegin(GL_POLYGON);
		glTexCoord2i(0,  1);
		glVertex2i(-hw, -hh);
		glTexCoord2i(1,  1);
		glVertex2i( hw, -hh);
		glTexCoord2i(1,  0);
		glVertex2i( hw,  hh);
		glTexCoord2i(0,  0);
		glVertex2i(-hw,  hh);
		glEnd();
	}
}

public class TestSpriteGL extends GL implements SDLConstants
{
	private static final int DEFAULT_NUMBER_OF_SPRITE = 100;
	private static final int DEFAULT_SCREEN_WIDTH = 640;
	private static final int DEFAULT_SCREEN_HEIGHT = 480;

	private Screen screen;
	private int numberOfSprite;

	private int texture;
	private Sprite[] sprites;

	public TestSpriteGL(Screen screen)
	{
		this(screen, DEFAULT_NUMBER_OF_SPRITE);
	}

	public TestSpriteGL(Screen screen, int numberOfSprite)
	{
		this.screen = screen;
		this.numberOfSprite = numberOfSprite;
	}

	public void mainLoop()
	{
		this.setUp();

		EventManager event = new EventManager();
		boolean done = false;
		int frames = 0;
		long totalTicks = Sys.getTicks();
		while (!done)
		{
			while (event.pollEvent() > 0)
			{
				switch (event.type)
				{
					case SDLEvent.SDL_KEYDOWN :
						if ((event.keydown.sym == SDLK_ESCAPE) ||
							(event.keydown.sym == SDLK_q))
							done = true;
						break;
					case SDLEvent.SDL_QUIT:
						done = true;
						break;
					default:
						break;
				}
			}

			this.move();
			this.draw();
			frames++;
		}

		totalTicks = Sys.getTicks() - totalTicks;
		System.out.println(((double)frames * 1000) / totalTicks + " (Frame per Second)");

		this.tearDown();
	}


	protected void setUp()
	{
		screen.showCursor(false);
		// init gl screen
		glViewport(0, 0, screen.getW(), screen.getH());
		glMatrixMode(GL_PROJECTION);
		glOrtho(0.0, screen.getW(), 0.0, screen.getH(), -1.0, 1.0);
		glMatrixMode(GL_MODELVIEW);

		glClearColor(0.0, 0.0, 0.0, 0.0);

		// init alpha blending
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

		// init texture
		/// load icon image and change pixel format
		Surface icon, tmpSurface;
		tmpSurface = Surface.loadBMP("../test/images/icon.bmp");
		tmpSurface.setColorKey(SDL_SRCCOLORKEY|SDL_RLEACCEL);

		if (Sys.byteOrder() == SDL_LIL_ENDIAN)
		{
			icon = Surface.createRGBSurface(Surface.SDL_SWSURFACE,
				tmpSurface.getW(), tmpSurface.getH(), 32,
				0xff, 0xff00, 0xff0000, 0xff000000);
		}
		else
		{
			icon = Surface.createRGBSurface(Surface.SDL_SWSURFACE,
				tmpSurface.getW(), tmpSurface.getH(), 32,
				0xff000000, 0xff0000, 0xff00, 0xff);
		}
		icon.blitSurface(tmpSurface);

		/// create texture from icon image
		glEnable(GL_TEXTURE_2D);
		this.texture = glGenTexture();
		glBindTexture(GL_TEXTURE_2D, this.texture);
		glTexImage2D(GL_TEXTURE_2D, 0,
			GL_RGBA, icon.getW(), icon.getH(), 0,
			GL_RGBA, GL_UNSIGNED_BYTE, icon.getPixels());

		/// free images
		tmpSurface.freeSurface();
		tmpSurface = null;
		icon.freeSurface();
		icon = null;

		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

		// init scprites
		this.sprites = new Sprite[numberOfSprite];
		for (int i = this.sprites.length-1;i >= 0;i--)
			this.sprites[i] = new Sprite(this.texture, this.screen.getW(), this.screen.getH());
	}

	protected void tearDown()
	{
		glDeleteTexture(this.texture);
		this.sprites = null;
	}

	protected void move()
	{
		for (int i = sprites.length-1;i >= 0 ;i--)
		{
			this.sprites[i].move();
		}
	}

	protected void draw()
	{
		glClear(GL_COLOR_BUFFER_BIT);
		for (int i = sprites.length-1;i >= 0 ;i--)
		{
			this.sprites[i].draw();
		}
		this.screen.glSwapBuffers();
	}

	public static void main(String[] args)
	{
		if (Sys.init(Sys.SDL_INIT_VIDEO) != 0)
		{
			System.out.println("Init() fail");
			System.exit(0);
		}

		try
		{
			int bpp = Screen.getVideoInfo().getVideoFormat().getBitsPerPixel();
			SDLOptionParser optParser = new SDLOptionParser(
				DEFAULT_SCREEN_WIDTH, DEFAULT_SCREEN_HEIGHT,
				bpp, SDL_OPENGL);
			args = optParser.parse(args);

			bpp = optParser.getBitsPerPixel();
			int[] rgbSize = new int[3];
			switch (bpp)
			{
			case 8 :
				rgbSize[0] = 3;
				rgbSize[1] = 3;
				rgbSize[2] = 2;
				break;
			case 16 :
				rgbSize[0] = 5;
				rgbSize[1] = 5;
				rgbSize[2] = 5;
				break;
			default:
				rgbSize[0] = 8;
				rgbSize[1] = 8;
				rgbSize[2] = 8;
				break;
			}

			Screen.glSetAttribute(SDL_GL_RED_SIZE, rgbSize[0]);
			Screen.glSetAttribute(SDL_GL_GREEN_SIZE, rgbSize[1]);
			Screen.glSetAttribute(SDL_GL_BLUE_SIZE, rgbSize[2]);
			Screen.glSetAttribute(SDL_GL_DEPTH_SIZE, 16);
			Screen.glSetAttribute(SDL_GL_DOUBLEBUFFER, 1);

			Screen screen = Screen.setVideoMode(
				optParser.getScreenWidth(),
				optParser.getScreenHeight(),
				bpp, optParser.getVideoFlags());
			if (screen == null)
			{
				System.out.println("Screen.setVideoMode() fail");
				System.exit(0);
			}

			int numberOfSprite = DEFAULT_NUMBER_OF_SPRITE;
			if (args.length > 0)
			{
				try
				{
					numberOfSprite = Integer.parseInt(args[0]);
				}
				catch (NumberFormatException e) { }
			}

			TestSpriteGL test = new TestSpriteGL(screen, numberOfSprite);
			System.gc();
			test.mainLoop();
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
		finally
		{
			Sys.quit();
		}
	}
}
