package sdl4gcj.mixer;

import sdl4gcj.Version;

public class Mixer implements MixerConstants
{
	// class methods 

	/**
	* Initialize audio device and mixer.
	* @param frequency Output sampling frequency in samples per second (Hz).
	* @param format Output sample format. Use one of following constants.
	* <ul>
	* <li>AUDIO_U8</li> <li>AUDIO_S8</li> <li>AUDIO_U16LSB</li> <li>AUDIO_S16LSB</li>
	* <li>AUDIO_U16MSB</li> <li>AUDIO_S16MSB</li> <li>AUDIO_U16</li> <li>AUDIO_S16</li>
	* </ul>
	* @param channels 1(mono) or 2(stereo)
	* @param chunksize Bytes used per output sample.
	* @return mixer object.
	**/
	public static native Mixer openAudio(
		int frequency, int format, int channels, int chunksize);

	/**
	* Initialize audio device and mixer.
	* Same as <code>Mixer.openAudio(MIX_DEFAULT_FREQUENCY, MIX_DEFAULT_FORMAT, 
	*		MIX_DEFAULT_CHANNELS, chunksize);</code>
	* @param chunksize Bytes used per output sample.
	* @return mixer object.
	**/
	public static Mixer openAudio(int chunksize)
	{
		return openAudio( MIX_DEFAULT_FREQUENCY, MIX_DEFAULT_FORMAT, 
			MIX_DEFAULT_CHANNELS, chunksize);
	}

	public static Mixer openAudio()
	{
		return openAudio(MIX_DEFAULT_FREQUENCY, MIX_DEFAULT_FORMAT, 
			MIX_DEFAULT_CHANNELS, 1024);
	}

	/**
	* Get version of compiled library.
	* @return version of compiled library 
	**/
	public static native Version compiledVersion();

	/**
	* Get version of linked library .
	* @return version of linked library
	**/
	public static native Version linkedVersion();

	// constractors 
	private Mixer()
	{
	}

	// instance methods 
	/**
	* Close mixer.
	**/
	public native void closeAudio();

	/**
	* Get the actual frequency of opened audio device . 
	* This may or may not match the parameters you passed to openAudio().
	* @return frequency of audio device.
	**/
	public native int getFrequency();

	/**
	* Get the actual format of opened audio device . 
	* This may or may not match the parameters you passed to openAudio().
	* @return frequency of audio device.
	**/
	public native int getFormat();

	/**
	* Get the actual number of audio channels.
	* This may or may not match the parameters you passed to openAudio().
	* @return frequency of audio device.
	**/
	public native int getChannels();


	/**
	* Add which channel to group tag.
	* @param channel Channel number of channels to assign tag to.
	* @param tag A group number Any positive numbers (including zero).
	* -1 is the default group. Use -1 to remove a group tag essentially.
	* @return 1 on success. 0 is returned when the channel specified is invalid.
	**/
	public native int groupChannel(int channel, int tag);

	/**
	* Add which channel to group tag.
	* @param fromChannel First Channel number of channels to assign tag to. Must be less or equal to to.
	* @param toChannel Last Channel number of channels to assign tag to. Must be greater or equal to from.
	* @param tag A group number Any positive numbers (including zero).
	* -1 is the default group. Use -1 to remove a group tag essentially.
	* @return 1 on success. 0 is returned when the channel specified is invalid.
	**/
	public native int groupChannels(int fromChannel, int toChannel, int tag);

	/**
	* Find the first available (not playing) channel in group tag.
	* @param tag A group number Any positive numbers (including zero).
	* -1 is the default group. Use -1 to remove a group tag essentially.
	* @return The channel found on success. -1 is returned when no channels in the group are available.
	**/
	public native int groupAvailable(int tag);

	/**
	* Count the number of channels in group tag.
	* @param tag A group number Any positive numbers (including zero).
	* -1 is the default group. Use -1 to remove a group tag essentially.
	* @return The number of channels found in the group. This function never fails.
	**/
	public native int groupCount(int tag);

	/**
	* Find the oldest actively playing channel in group tag.
	* @param tag A group number Any positive numbers (including zero).
	* -1 is the default group. Use -1 to remove a group tag essentially.
	* @return The channel found on success. -1 is returned when no channels in the group are playing or the group is empty.
	**/
	public native int groupOldest(int tag);

	/**
	* Find the newest, most recently started, actively playing channel in group tag.
	* @param tag A group number Any positive numbers (including zero).
	* -1 is the default group. Use -1 to remove a group tag essentially.
	* @return The channel found on success. -1 is returned when no channels in the group are playing or the group is empty.
	**/
	public native int groupNewer(int tag);

	// extern DECLSPEC int SDLCALL Mix_AllocateChannels(int numchans);
	/**
	* Set the number of channels for this mixer.
	* @param numberOfChannels Number of channels to allocate for mixer.
	* @return The number of channels allocated.
	**/
	public native int allocateChannels(int numberOfChannels);

	/**
	* Check the status of specified channel.
	* If the specified channel is -1, check all channels.
	* @param channel target channel.
	* @return status playing status specified channel
	**/
	public native boolean playing(int channel);

	/**
	* Check the status of all channels. Same as <code>playing(-1)</code>
	**/
	public boolean playing()
	{
		return this.playing(-1);
	}

	/**
	* Check the status of music playing.
	**/
	public native boolean playingMusic();

	/**
	* Play an audio chunk on a specific channel.
	* @param channel a channel number, if channel is -1, paly on the first free channel.
	* @param chunk a chunk data
	* @param loops loop count. if loops is -1, infinity loop.
	* @param ticks playing time milisecond
	**/
	public native int playChannel(int channel, Chunk chunk, int loops, int ticks);

	/**
	* Play an audio chunk on a first free channel.
	* Same as <code>playChannel(-1, chunk, loops, ticks);</code>
	* @param chunk chunk data
	* @param loops loop count. if loops is -1, infinity loop.
	* @param ticks milisecond
	**/
	public int playChannel(Chunk chunk, int loops, int ticks)
	{
		return this.playChannel(-1, chunk, loops, ticks);
	}

	/**
	* Play an audio chunk on a specified channel.
	* Same as <code>playChannel(channel, chunk, loops, -1);</code>
	* @param channel a channel number, if channel is -1, paly on the first free channel.
	* @param chunk chunk data
	* @param loops loop count. if loops is -1, infinity loop.
	**/
	public int playChannel(int channel, Chunk chunk, int loops)
	{
		return this.playChannel(channel, chunk, loops, -1);
	}

	/**
	* Play an audio chunk on a first free channel.
	* Same as <code>playChannel(-1, chunk, loops, -1);</code>
	* @param chunk chunk data
	* @param loops loop count. if loops is -1, infinity loop.
	**/
	public int playChannel(Chunk chunk, int loops)
	{
		return this.playChannel(-1, chunk, loops, -1);
	}

	/**
	* Play an audio chunk on a first free channel. 
	* Same as <code>playChannel(-1, chunk, 0, -1);</code>
	* @param chunk chunk data
	**/
	public int playChannel(Chunk chunk)
	{
		return this.playChannel(-1, chunk, 0, -1);
	}

	/**
	* Play an audio chunk on a specific channel.
	* @param channel a channel number, if channel is -1, paly on the first free channel.
	* @param chunk a chunk data
	* @param loops loop count. if loops is -1, infinity loop.
	* @param fadeTime fade in time (milisecond)
	* @param ticks playing time (milisecond)
	**/
	public native int fadeInChannel(int channel, Chunk chunk, int loops, int fadeTime, int ticks);

	/**
	* Play an audio chunk on a specific channel.
	* Same as <code>fadeInChannel(channel, chunk, loops, fadeTime, -1); </code>
	* @param channel a channel number, if channel is -1, paly on the first free channel.
	* @param chunk a chunk data
	* @param loops loop count. if loops is -1, infinity loop.
	* @param fadeTime fade in time (milisecond)
	**/
	public int fadeInChannel(int channel, Chunk chunk, int loops, int fadeTime)
	{
		return fadeInChannel(channel, chunk, loops, fadeTime, -1);
	}

	/**
	* Play an audio chunk on a specific channel.
	* Same as <code>fadeInChannel(channel, chunk, -1, fadeTime, -1); </code>
	* @param channel a channel number, if channel is -1, paly on the first free channel.
	* @param chunk a chunk data
	* @param fadeTime fade in time (milisecond)
	**/
	public int fadeInChannel(int channel, Chunk chunk, int fadeTime)
	{
		return fadeInChannel(channel, chunk, -1, fadeTime, -1);
	}

	/**
	* Play an audio chunk on a specific channel.
	* Same as <code>fadeInChannel(-1, chunk, -1, fadeTime, -1); </code>
	* @param chunk a chunk data
	* @param fadeTime fade in time (milisecond)
	**/
	public int fadeInChannel(Chunk chunk, int fadeTime)
	{
		return fadeInChannel(-1, chunk, -1, fadeTime, -1);
	}

	/**
	* Play an audio chunk on a first free channel. 
	* @param music music data
	* @param loops loop count if loops is -1, infinity loop.
	**/
	public native void playMusic(Music music, int loops);

	/**
	* Play the music.
	* Same as <code>playMusic(music, -1); </code>
	* @param music music data
	**/
	public void playMusic(Music music)
	{
		this.playMusic(music, -1);
	}

	/**
	* Play the music with fadein effect.
	* @param music music data
	* @param loops loop count
	* @param ticks milisecond
	**/
	public native void fadeInMusic(Music music, int loops, int ticks);

	/**
	* Play the music with fadein effect.
	* same as fadeInMusic(music, -1, ticks) loop inifinitely 
	* @param music music data
	* @param ticks milisecond
	**/
	public void fadeInMusic(Music music, int ticks)
	{
		this.fadeInMusic(music, -1, ticks);
	}

	/**
	* Play the music with fadein effect.
	* @param music music data
	* @param loops loop count
	* @param ticks milisecond
	* @param position position
	* @return 0 on success, or -1 on error.
	**/
	public native void fadeInMusicPos(Music music, int loops, int ticks, double position);

	/**
	* Play the music with Fadein effect.
	* Same as <code>fadeInMusicPos(music, -1, ticks, position);</code>
	* @param music music data
	* @param ticks milisecond
	* @param position position
	* @return 0 on success, or -1 on error.
	**/
	public void fadeInMusicPos(Music music, int ticks, double position)
	{
		this.fadeInMusicPos(music, -1, ticks, position);
	}

	/**
	* Pause specified channel.
	* @param channel target channel
	**/
	public native void pause(int channel);

	/**
	* Resume specified channel.
	* @param channel target channel
	**/
	public native void resume(int channel);

	/**
	* Get pause status of a specified channel.
	* @param channel target channel
	* @return Return the pause status of a specified channel
	**/
	public native boolean paused(int channel);

	/**
	* Pause music stream
	**/
	public native void pauseMusic();

	/**
	* Resume music stream
	**/
	public native void resumeMusic();

	/**
	* Rewind music stream
	**/
	public native void rewindMusic();

	/**
	* Get pause status of music stream
	* @return Return the pause status of music stream.
	**/
	public native boolean pausedMusic();

	/**
	* Set position of playback in stream.
	**/
	public native void setMusicPosition(double position);

	/**
	* Set channel volume. 
	* If the volume is out of range, this method do nothing.
	* @param channel target channel
	* @param volume new volume (0 - 128)
	* @return original volume (0 - 128)
	**/
	public native int volume(int channel, int volume);

	/**
	* Get channel volume. 
	* @param channel target channel
	* @return current volume (0 - 128)
	**/
	public int getVolume(int channel)
	{
		return this.volume(channel, -1);
	}

	/**
	* Set volume for all channels. same as valume(-1, volume);
	* @param volume new volume (0 - 128)
	**/
	public void setVolume(int volume)
	{
		this.volume(-1, volume);
	}


	/**
	* Set music volume. 
	* @param volume new volume (0 - 128)
	* @return original volume (0 - 128)
	**/
	public native int volumeMusic(int volume);

	/**
	* Set music volume. 
	* Same as <code>volumeMusic(volume)</code>
	* @param volume new volume (0 - 128)
	**/
	public void setVolumeMusic(int volume)
	{
		this.volumeMusic(volume);
	}

	/**
	* Get current music volume. 
	* Same as <code>volumeMusic(-1)</code>
	* @return current volume (0 - 128)
	**/
	public int getVolumeMusic()
	{
		return this.volumeMusic(-1);
	}

	/**
	* Halt playing of a specified channel.
	* @param channel target channel
	* @return ??
	**/
	public native int haltChannel(int channel);

	/**
	* Halt playing of a specified group.
	* @param tag target group
	* @return ??
	**/
	public native int haltGroup(int tag);

	/**
	* Halt music playing.
	* @return ??
	**/
	public native int haltMusic();

	/**
	* Change the expiration delay for a particular channel.
	* The sample will stop playing after the 'ticks' milliseconds have elapsed,
	* or remove the expiration if 'ticks' is -1
	* Mix_ExpireChannel(int channel, int ticks);
	* @param channel Target channel, or -1 to fade all channels out.
	* @param ticks new expiration ticks
	* @return Number of channels set to expire. Whether or not they are active.
	**/
	public native int expireChannel(int channel, int ticks);

	/**
	* Set fade out effect for a particular channel.
	* Mix_FadeOutChannel(int channel, int ms);
	* @param channel Channel to fade out, or -1 to fade all channels out.
	* @param ms Milliseconds of time that the fade-out effect should take to go to silence, starting now.
	* @return The number of channels set to fade out.
	**/
	public native int fadeOutChannel(int channel, int ms);

	/**
	* Set fade out effect for a particular group.
	* Mix_FadeOutGroup(int tag, int ms);
	* @param tag Group to fade out.
	* @param ms Milliseconds of time that the fade-out effect should take to go to silence, starting now.
	* @return The number of channels set to fade out.
	**/
	public native int fadeOutGroup(int tag, int ms);

	/**
	* Set fade out effect music stream.
	* <code>Mix_FadeOutMusic(int ms);</code>
	* @param ms Milliseconds of time that the fade-out effect should take to go to silence, starting now.
	* @return 1 on success, 0 on failure.
	**/
	public native int fadeOutMusic(int ms);

	/**
	* Get fading status of music stream.
	* <code>Mix_FadingMusic(void)</code>
	* @return 
	**/
	public native int fadingMusic();

	/**
	* Get fading status of specified channel.
	* <code>Mix_FadingChannel(int channel)</code>
	* @param channel Target channel
	* @return Status of fading effect. It is one of the following. 
	* <ul> <li>MIX_NO_FADING</li> <li>MIX_FADING_OUT</li> <li>MIX_FADING_I</li> </ul>
	**/
	public native int fadingChannel(int channel);

	/**
	* Set panning filer.
	* @param channel target channel
	* @param left left value (0 - 255)
	* @param right right value (0 - 255)
	**/
	public native void setPanning(int channel, short left, short right);

	/**
	* Set position filer.
	* @param channel target channel
	* @param angle angle 0 - 360
	* @param distance distance value between 0 and 255 that specifies the space between the sound and the listener.
	**/
	public native void setPosition(int channel, short angle, short distance);

	/**
	* Set distance filer.
	* @param channel target channel
	* @param distance distance value between 0 and 255 that specifies the space between the sound and the listener.
	**/
	public native void setDistance(int channel, short distance);

	/**
	* Set reverse stereo.
	* @param channel target channel
	* @param flip flip flag
	**/
	public native void setReverseStereo(int channel, boolean flip);

	/**
	* Reserve the first channels (0 - n-1) for the application,
	* @param num number of channels
	* @param flip flip flag
	**/
	public native int reserveChannels(int num);


}
