/**************************************************************************
 Moenizer - Allow to set background image for OmegaT.
 
 Copyright (C) 2013 Yu Tang
               Home page: http://sourceforge.jp/users/yu-tang/
               Support center: http://sourceforge.jp/users/yu-tang/

 This file is part of plugin for OmegaT.
 http://www.omegat.org/

 License: GNU GPL version 3 or (at your option) any later version.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package org.omegat.plugin.moenizer;

import com.sun.java.swing.plaf.windows.WindowsMenuBarUI;
import com.vlsolutions.swing.docking.DockView;
import com.vlsolutions.swing.docking.DockViewTitleBar;
import com.vlsolutions.swing.docking.Dockable;
import com.vlsolutions.swing.docking.DockableState;
import com.vlsolutions.swing.docking.DockingDesktop;
import com.vlsolutions.swing.docking.SplitContainer;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Graphics;
import java.awt.SystemColor;
import java.awt.image.BufferedImage;
import javax.swing.JComponent;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JMenuBar;
import javax.swing.JScrollPane;
import javax.swing.JTextPane;
//import javax.swing.plaf.basic.BasicMenuBarUI;
import org.omegat.core.Core;
import org.omegat.util.gui.UIThreadsUtil;

/**
 *
 * @author Yu-Tang
 */
public class MoeUI {
    
    private static MoeUI moeUI;
    
    private JFrame frame;
    private MoeLayeredPane layeredPane;
    private Container contentPane;
    private JMenuBar menuBar;
    private DockingDesktop desktop;

    static {
        moeUI = null;
    }

    // Singleton. Not allow to Instanciate from outside. Use getMoeUI() to get instance.
    private MoeUI(BufferedImage image) {
        UIThreadsUtil.mustBeSwingThread();

        initUI(image);
    }

     private MoeUI() {
        UIThreadsUtil.mustBeSwingThread();

        initUI(null);
    }
 
    public static MoeUI getMoeUI(BufferedImage image) {
        if (moeUI == null) {
            moeUI = new MoeUI(image);
        }
        return moeUI;
    }
    
    public static MoeUI getMoeUI() {
        if (moeUI == null) {
            moeUI = new MoeUI();
        }
        return moeUI;
    }
    
    public void transparent() {
        UIThreadsUtil.mustBeSwingThread();

        transparent(menuBar);
        transparentRecursive(contentPane);
        transparentInstantStart(desktop);

        frame.repaint();
    }
    
    public void transparentEditor() {
        // Editor に表示されるドキュメントは、以下のケースでまったく異なる。
        //
        // 1. インスタントスタートガイドが表示されている場合（初期状態）
        //    -> view = JTextPane, HTMLDocument
        // 2. プロジェクトをロードまたは新規作成して、分節編集画面が表示されている場合
        //    -> view = JEditorPane, DefaultStyledDocument
        //
        // そのため、2 のタイミングで透過処理をやり直す必要がある。
        // このメソッドは、2 の透過処理専用。

        UIThreadsUtil.mustBeSwingThread();

        JEditorPane editor = getJEditorPaneFromEditor(desktop);
        if (editor == null) {
            return;
        }
        
        /* ここで JEditorPane に半透明の背景色を設定すると、テキストの選択（反転）
         * などの色の変化が正常に更新されず色残りしてしまう。
         * そのため、JEditorPane への処理では単純に背景を透明にして、元画像自体で
         * 色味を調整しておく方針とする。
         * 
        int alpha = 100; // transparent <- 0...255 -> opaque
        Color color = SystemColor.menu;
        editor.setBackground(new Color( color.getRGB() & 0xffffff | 100 << 24, true));
        * */
        editor.setOpaque(false);
        frame.repaint();
    }
    
    public void setBackground(final BufferedImage image) {
        UIThreadsUtil.executeInSwingThread(new Runnable() {
            @Override
            public void run() {
                layeredPane.setBackground(image);
            }
        });
    }

    private void initUI(BufferedImage image) {
        frame = Core.getMainWindow().getApplicationFrame();
        if (image == null) {
            layeredPane = new MoeLayeredPane();
        } else {
            layeredPane = new MoeLayeredPane(image);
        }
        contentPane = frame.getContentPane();
        menuBar = frame.getJMenuBar();
        desktop = getDockingDesktop(contentPane);

        // replace LayeredPane with MoeLayeredPane
        frame.setLayeredPane(layeredPane);
        frame.setContentPane(contentPane);
        frame.setJMenuBar(menuBar);

        frame.validate();
    }
    
    private void transparent(JMenuBar menuBar) {
//        menuBar.setUI ( new BasicMenuBarUI () {
// 上記だと初期表示時に、アクティブメニューがアクティブで描画されない。
// マウス通過などのタイミングで再描画された際にアクティブカラーになるが、
// 見栄えが悪いので、Windows 用の UI を指定する。
// アクティブメニューがアクティブで描画されない問題はこれで解決するが、
// 変わりに Mac や *nix など別 L&F 環境下では違和感があるかもしれない。
        menuBar.setUI ( new WindowsMenuBarUI () {
            @Override
            public void paint ( Graphics g, JComponent c ) {
                int alpha = 100; // transparent <- 0...255 -> opaque
                Color oldColor = g.getColor();
                Color color = SystemColor.menu;
                g.setColor ( new Color( color.getRGB() & 0xffffff | alpha << 24, true));
                g.fillRect ( 0, 0, c.getWidth (), c.getHeight () );
                g.setColor ( oldColor ); // restore
            }
        } );
        menuBar.setOpaque(false);
    }
    
    private void transparentRecursive(Component component) {
        if (component instanceof JComponent) {
            JComponent c = (JComponent) component;
            if (c.isOpaque()) {
                c.setOpaque(false);
            }
        }
        
        if (component instanceof Container) {
            Container container = (Container) component;
            for (Component c: container.getComponents()) {
                transparentRecursive(c);
            }
        }
        
        if (component instanceof DockingDesktop) {
            transparentRecursive((DockingDesktop) component);
        }
    }
        
    private void transparentRecursive(DockingDesktop desktop) {
        //DockingPanel dockingPanel = dockingDesktop.getDockingPanel(); // Scoping NG
        // DockingPanel にアクセスできないので、下位要素から上にさかのぼって処理する。
        // 階層的には、こんな感じになっている。
        // -----------------------
        // DockingDesktop
        //  + DockingPanel                  <-- splitContainer.getParent()
        //    + splitContainer(HORIZONTAL)  <-- splitContainer.getParent()
        //      + splitContainer(VERTICAL)  <-- DockView.getParent()
        //        + DockView                <-- Dockable.getParent()
        //          + Dockable              <-- DockableState.getDockable()
        for (DockableState d: desktop.getDockables()) {
            double width = d.getPosition().getWidth();
            // width (height や x, y でも可) が 0.0 以外の場合はアイコン化されているので、透過処理不要
            // 他に適切な判定方法がありそうだけれども、分からなかったので、とりあえずこれで。
            if (width == 0.0) {
                transparentRecursive(d);
            }
        }
    }

    private void transparentRecursive(DockableState dockableState) {
        Dockable dockable = dockableState.getDockable();

        // DockView
        Container container = dockable.getComponent().getParent();
        DockView view = (DockView) container;
        if (view.isOpaque()) {
            view.setOpaque(false);
        }
        DockViewTitleBar titleBar = view.getTitleBar();
        if (titleBar.isOpaque()) {
            titleBar.setOpaque(false);
        }
        titleBar.setUI(new MoeDockViewTitleBarUI(titleBar));

        // SplitContainer(VERTICAL)
        container = container.getParent();
        if (container == null) {
            return;
        } else if (container.isOpaque()) {
            ((SplitContainer) container).setOpaque(false);
        }

        // SplitContainer(HORIZONTAL)
        container = container.getParent();
        if (container == null) {
            return;
        } else if (container.isOpaque()) {
            ((SplitContainer) container).setOpaque(false);
        }
    }

    private void transparentInstantStart(DockingDesktop desktop) {
        // お手軽スタートは、以下のような HTML で背景色が指定されている。
        // <body ... bgcolor="white" ...>
        // そのため、コンポーネント自体を透過にしても、HTML Body 背景色の
        // 白指定が効いて透過にならない。そこで、背景色指定を削除する。
        for (DockableState d: desktop.getDockables()) {
            Dockable dockable = d.getDockable();
            String key = dockable.getDockKey().getKey();
            if (key.equalsIgnoreCase("EDITOR")) { // found InstantStartGuide
                JScrollPane sp = (JScrollPane) dockable.getComponent();
                JTextPane tp = (JTextPane) sp.getViewport().getView();
                tp.setText(tp.getText().replace(" bgcolor=\"white\"", ""));
                tp.setCaretPosition(0);
                return;
            }
        }
    }
    
    private DockingDesktop getDockingDesktop(Container container) {
        for (Component c: container.getComponents()) {
            if (c instanceof DockingDesktop) {
                return (DockingDesktop) c;
            }
        }
        return null;
    }

    private JEditorPane getJEditorPaneFromEditor(DockingDesktop desktop) {
        for (DockableState d: desktop.getDockables()) {
            Dockable dockable = d.getDockable();
            if (dockable.getDockKey().getKey().equalsIgnoreCase("EDITOR")) {
                JScrollPane sp = (JScrollPane) dockable.getComponent();
                return (JEditorPane) sp.getViewport().getView();
            }
        }
        return null;
    }
}
