﻿using NLog;
using Prism.Navigation;
using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Compression;
using System.Net;
using System.Net.Http;
using System.Text;
using System.Threading.Tasks;

namespace TestNarou3.OuterEdge.Repository.API
{
    internal class HttpClientHelper : IDisposable
    {
        private static readonly ILogger logger = LogManager.GetCurrentClassLogger();

        private const string REQUEST_HEADER_KEY_USER_AGENT = "User-Agent";
        //private const string REQUEST_HEADER_VALUE_USER_AGENT = "Mozilla/5.0 TestNarou/0.1";
        private const string REQUEST_HEADER_VALUE_USER_AGENT = "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/103.0.0.0 Safari/537.36";
        private const string REQUEST_HEADER_KEY_REFERRER = "Referer";

        //private const string RESPONSE_HEADER_KEY_COOKIE = "Set-Cookie";

        private readonly CookieContainer cookies;
        private readonly HttpClientHandler httpClientHandler;
        private readonly HttpClient httpClient;

        private bool disposedValue;

        public HttpClientHelper()
        {
            cookies = new();

            httpClientHandler = new()
            {
                AutomaticDecompression = DecompressionMethods.GZip | DecompressionMethods.Deflate | DecompressionMethods.Brotli,
                CookieContainer = cookies
            };

            httpClient = new(httpClientHandler);

            // dummy cookies
            Uri uri = new("https://syosetu.com");

            (string key, string value)[] keyValues = new[]
            {
#if true
                ("nlist1", "177bb.3-t3he.5s-128dp.1p-ymk6.37-m4t3.2-p1r8.1y-16nyn.2-16fg0.9-p47r.2-kij7.1jj-dp1s.3a-wrsm.9i-y3gs.1-13qar.2-m6bt.8-15amu.20-nsj1.11-sj8f.y-14vo4.1d-gxeh.r-p8az.o-qsmn.19-xhtr.1c-13tin.v-xsc1.z-y2m4.3s-if0o.e-11s5n.1-vwo9.62-yjyd.0"),
                ("nlist3", "11h1y.1-jw85.3-w9yx.3e-134ma.10-eq4y.1o-8s36.25-diya.1m-t2nz.1-ym0j.9-fqfg.1y-zez1.g-14egc.0-uvgq.0-12y59.0-159jy.g-15rq0.0-15ql3.0-12ik0.0-12ijv.0"),
                ("sasieno", "0"),
                ("lineheight", "0"),
                ("fontsize", "0"),
                ("novellayout", "0"),
                ("fix_menu_bar", "1"),
                ("over18", "yes"),
#endif
                ("_gid", "GA1.2.610388633.1659441715"),
                ("_gat_gtag_UA_3637811_4", "1"),
                ("_ga_1TH9CF4FPC", "GS1.1.1659592356.35.1.1659597679.0"),
                ("_ga", "GA1.2.636765982.1649058346"),
            };

            foreach (var (key, value) in keyValues)
            {
                Cookie cookie = new(key, value);
                cookies.Add(uri, cookie);
            }
        }

        #region IDisposable

        protected virtual void Dispose(bool disposing)
        {
            if (!disposedValue)
            {
                if (disposing)
                {
                    httpClient.Dispose();
                    logger.Trace("HttpClientHelper disposed.");
                }

                disposedValue = true;
            }
        }

        public void Dispose()
        {
            Dispose(disposing: true);
            GC.SuppressFinalize(this);
        }

        #endregion

        public async Task<HttpResponseMessage> GetAsync(string url, string referer)
        {
            var request = CreateRequestMessage(url, referer, HttpMethod.Post);

            logger.Trace("GetAsync: URL {0}, referer {1}", url, referer);

            var response = await httpClient.SendAsync(request).ConfigureAwait(false);

            //ShowResponseHeaders(response);
            //ShowCookies();

            logger.Trace("StatusCode: {0}", response.StatusCode);

            return response;
        }

        public async Task<HttpResponseMessage> PostAsync(string url, string referer, IDictionary<string, string> form)
        {
            var content = new FormUrlEncodedContent(form);

            return await PostAsync(url, referer, content).ConfigureAwait(false);
        }

        public async Task<HttpResponseMessage> PostAsync(string url, string referer, FormUrlEncodedContent content)
        {
            var request = CreateRequestMessage(url, referer, HttpMethod.Post);

            request.Content = content;

            logger.Trace("PostAsync: URL {0}, referer {1}", url, referer);

            var response = await httpClient.SendAsync(request).ConfigureAwait(false);

            //ShowResponseHeaders(response);
            //ShowCookies();

            logger.Trace("StatusCode: {0}", response.StatusCode);

            return response;
        }

        private static HttpRequestMessage CreateRequestMessage(string url, string referer, HttpMethod method)
        {
            var request = new HttpRequestMessage(method, url);

            request.Headers.Add(REQUEST_HEADER_KEY_USER_AGENT, REQUEST_HEADER_VALUE_USER_AGENT);

            if (referer != null)
            {
                request.Headers.Add(REQUEST_HEADER_KEY_REFERRER, referer);
            }

            request.Headers.Add("Accept", "text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.9");
            //request.Headers.Add("Accept-Encoding", "gzip, deflate, br");
            request.Headers.Add("Accept-Language", "ja,en;q=0.9,en-US;q=0.8");
            request.Headers.Add("Cache-Control", "no-cache");
            request.Headers.Add("Connection", "keep-alive");
            request.Headers.Add("DNT", "1");
            //request.Headers.Add("Host", "syosetu.com");
            //request.Headers.Add("Origin", "https://syosetu.com");
            request.Headers.Add("Pragma", "no-cache");
            request.Headers.Add("sec-ch-ua", @""".Not/A)Brand"";v=""99"", ""Google Chrome"";v=""103"", ""Chromium"";v=""103""");
            request.Headers.Add("sec-ch-ua-mobile", "?0");
            request.Headers.Add("sec-ch-ua-platform", @"""Windows""");
            request.Headers.Add("Sec-Fetch-Dest", "document");
            request.Headers.Add("Sec-Fetch-Mode", "navigate");
            request.Headers.Add("Sec-Fetch-Site", "same-origin");
            request.Headers.Add("Sec-Fetch-User", "?1");
            request.Headers.Add("Upgrade-Insecure-Requests", "1");

            return request;
        }

        private void ShowCookies()
        {
            foreach (Cookie cookie in cookies.GetAllCookies())
            {
                logger.Trace("Cookie: {0}, {1}, {2}, {3}",
                        cookie.ToString(), cookie.Domain, cookie.Path, cookie.TimeStamp);
            }
        }

        private static void ShowResponseHeaders(HttpResponseMessage response)
        {
            foreach (var header in response.Headers)
            {
                logger.Trace("{0}: {1}", header.Key, string.Join(", ", header.Value));
            }
        }

        public static async Task<string> ReadContentAsync(HttpResponseMessage response)
        {
            return await response.Content.ReadAsStringAsync().ConfigureAwait(false);
        }

        public static string ReadGzipContent(HttpResponseMessage response)
        {
            using var rs = response.Content.ReadAsStream();
            using var ds = new GZipStream(rs, CompressionMode.Decompress);
            using var ws = new MemoryStream();

            ds.CopyTo(ws);

            ws.Position = 0;
            byte[] raw = new byte[ws.Length];
            ws.Read(raw, 0, raw.Length);

            return Encoding.UTF8.GetString(raw);
        }
    }
}
