﻿using NLog;
using Prism.Navigation;
using System.Collections.Generic;
using System.Linq;
using TestNarou3.Domain.Boundary.Repository;
using TestNarou3.Domain.Boundary.Service;
using TestNarou3.Domain.Model.API;
using TestNarou3.Domain.Model.Entity;
using TestNarou3.Domain.Model.Entity.Child;
using TestNarou3.Domain.Translator;

namespace TestNarou3.Domain.Service
{
    internal class NarouService : INarouService, IDisposable
    {
        private static readonly ILogger logger = LogManager.GetCurrentClassLogger();

        private string DataToken { get; set; }

        private BookmarkCategory Categeory { get; }
        private BookmarkDetailList DetailList { get; }

        private readonly INarouRepository repo;

        private readonly IBookmarkCategoryRowTranslator vmCategoryRowTranslator;
        private readonly IBookmarkDetailListRowTranslator vmDetailRowTranslator;

        private bool disposedValue;

        public NarouService(
                INarouRepository repo,
                IBookmarkCategoryRowTranslator vmCategoryRowTranslator,
                IBookmarkDetailListRowTranslator vmDetailRowTranslator
        )
        {
            this.repo = repo;

            this.vmCategoryRowTranslator = vmCategoryRowTranslator;
            this.vmDetailRowTranslator = vmDetailRowTranslator;

            this.Categeory = new();
            this.DetailList = new();
        }

        public void Login(string id, string password)
        {
            this.DataToken = this.repo.Login(id, password);
        }

        public void Logout()
        {
            if (string.IsNullOrEmpty(this.DataToken))
            {
                return;
            }

            this.repo.Logout(this.DataToken);
        }

        public BookmarkCategory GetBookmarkCategory()
        {
            IEnumerable<NarouCategoryInfo> infos = this.repo.GetNarouCategoryList();

            IEnumerable<BookmarkCategoryRow> rows = infos.Zip(
                    Enumerable.Range(1, infos.Count()),
                    (category, no) => (category, no)
            ).Select(
                    pair => this.vmCategoryRowTranslator.Translate(pair.category, pair.no)
            );

            this.Categeory.Refresh(rows);

            return this.Categeory;
        }

        public BookmarkDetailList GetBookmarkDetailList(int categoryNo)
        {
            IEnumerable<NarouNovelDetailInfo> novelInfos =
                    this.repo.GetNovelDetailInfoList(
                            categoryNo, this.Categeory.Rows[categoryNo - 1].Count.Value);

            IEnumerable<BookmarkDetailListRow> detailRows =
                    novelInfos.Skip(1).Select(v => vmDetailRowTranslator.Translate(v));

            this.DetailList.Refresh(detailRows);

            return this.DetailList;
        }

        #region IDisposable

        protected virtual void Dispose(bool disposing)
        {
            if (!disposedValue)
            {
                if (disposing)
                {
                    this.Categeory?.Dispose();
                    this.DetailList?.Dispose();

                    this.repo?.Dispose();

                    logger.Trace("NarouService disposed.");
                }

                disposedValue = true;
            }
        }

        public void Dispose()
        {
            Dispose(disposing: true);
            GC.SuppressFinalize(this);
        }

        #endregion
    }
}
