package gdsearch.vo;

import java.text.*;
import java.util.Date;

import javax.jdo.annotations.IdGeneratorStrategy;
import javax.jdo.annotations.IdentityType;
import javax.jdo.annotations.PersistenceCapable;
import javax.jdo.annotations.Persistent;
import javax.jdo.annotations.PrimaryKey;
import javax.servlet.http.HttpServletRequest;
import com.google.appengine.api.datastore.Query;

/**
 * 検索条件を格納するクラス
 */
@PersistenceCapable(identityType = IdentityType.APPLICATION, detachable="true")
public class GameDataSearchCondition {

    @PrimaryKey
    @Persistent(valueStrategy = IdGeneratorStrategy.IDENTITY)
    private Long id;
    @Persistent
    private String              hard;          // ハード名
    @Persistent
    private String              genre;         // 内容
    @Persistent
    private String              keyword;       // キーワード
    @Persistent
    private Date                saleStartDate; // 発売日（開始）
    @Persistent
    private Date                saleEndDate;   // 発売日（終了）
    @Persistent
    private int                 displayStart;  // 表示開始件数
    @Persistent
    private int                 displayNumber; // 表示件数
    @Persistent
    private String              sortKey;       // ソートキー
    @Persistent
    private Query.SortDirection sortDirection; // ソート順(ASC or DESC)

    // 検索履歴用
    @Persistent
    private String              remoteHost;
    @Persistent
    private String              remoteAddr;
    @Persistent
    private String              userAgent;
    @Persistent
    private Date                searchDate;    // 検索日

    public GameDataSearchCondition() {}

    public GameDataSearchCondition(HttpServletRequest req) throws ParseException {
        setHard        (req.getParameter("hard"   ));
        setGenre       (req.getParameter("genre"  ));
        setKeyword     (req.getParameter("keyword"));

        // 発売年のパラメータから発売日をDate型で設定する
        DateFormat df = new SimpleDateFormat("yyyy/MM/dd");
        if( "" != req.getParameter("saleYearStart")) {
            setSaleStartDate(df.parse(req.getParameter("saleYearStart") +"/01/01"));
        }
        if( "" != req.getParameter("saleYearEnd")) {
            setSaleEndDate  (df.parse(req.getParameter("saleYearEnd"  ) +"/12/31"));
        }

        // 表示開始件数の設定
        setDisplayStart (Integer.parseInt(req.getParameter("DisplayStart")));
        // 表示件数の設定
        setDisplayNumber(Integer.parseInt(req.getParameter("DisplayNumber")));

        // ソートキーの設定
        setSortKey(req.getParameter("SortKey").replaceAll("^(.+) .+$", "$1"));

        // ソート順の設定
        String sortKey = req.getParameter("SortKey").replaceAll("^.+ (.+)$", "$1");
        if(sortKey.equals("DESC")) {
            setSortDirection(Query.SortDirection.DESCENDING);
        } else {
            setSortDirection(Query.SortDirection.ASCENDING);
        }

        // 検索履歴に記録する情報
        setRemoteHost(req.getRemoteHost());
        setRemoteAddr(req.getRemoteAddr());
        setUserAgent (req.getHeader("user-agent"));
        setSearchDate (new Date()); // 検索時の時刻を記録
    }

    public Long                getId()            { return id;            }
    public String              getHard()          { return hard;          }
    public String              getGenre()         { return genre;         }
    public String              getKeyword()       { return keyword;       }
    public Date                getSaleStartDate() { return saleStartDate; }
    public Date                getSaleEndDate()   { return saleEndDate;   }
    public int                 getDisplayStart()  { return displayStart;  }
    public int                 getDisplayNumber() { return displayNumber; }
    public String              getSortKey()       { return sortKey;       }
    public Query.SortDirection getSortDirection() { return sortDirection; }
    public String              getRemoteHost()    { return remoteHost;    }
    public String              getRemoteAddr()    { return remoteAddr;    }
    public String              getUserAgent()     { return userAgent;     }
    public Date                getSearchDate()    { return searchDate;    }
    public String getCacheKey()  {
        String result = "";
        result += hard + ":";
        result += genre + ":";
        result += keyword + ":";
        result += (saleStartDate == null ? "" : saleStartDate) + ":";
        result += (saleEndDate   == null ? "" : saleEndDate  ) + ":";
        result += Integer.toString(getDisplayStart())+ ":";
        result += Integer.toString(getDisplayNumber())+ ":";
        result += getSortKey()+ ":";
        result += getSortDirection();
        return result;
    }

    public void setHard         (String              hard         ) { this.hard          = hard;          }
    public void setGenre        (String              genre        ) { this.genre         = genre;         }
    public void setKeyword      (String              keyword      ) { this.keyword       = keyword;       }
    public void setSaleStartDate(Date                saleStartDate) { this.saleStartDate = saleStartDate; }
    public void setSaleEndDate  (Date                saleEndDate  ) { this.saleEndDate   = saleEndDate;   }
    public void setDisplayStart (int                 displayStart ) { this.displayStart  = displayStart;  }
    public void setDisplayNumber(int                 displayNumber) { this.displayNumber = displayNumber; }
    public void setSortKey      (String              sortKey      ) { this.sortKey       = sortKey;       }
    public void setSortDirection(Query.SortDirection sortDirection) { this.sortDirection = sortDirection; }
    public void setRemoteHost   (String              remoteHost   ) { this.remoteHost    = remoteHost;    }
    public void setRemoteAddr   (String              remoteAddr   ) { this.remoteAddr    = remoteAddr;    }
    public void setUserAgent    (String              userAgent    ) { this.userAgent     = userAgent;     }
    public void setSearchDate   (Date                searchDate   ) { this.searchDate    = searchDate;    }
}
