//! @file d3dgraphics.cpp
//! @brief D3DGraphicsNX̒`

//--------------------------------------------------------------------------------
// 
// OpenXOPS
// Copyright (c) 2014-2015, OpenXOPS Project / [-_-;](mikan) All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
// * Redistributions of source code must retain the above copyright notice, 
//   this list of conditions and the following disclaimer.
// * Redistributions in binary form must reproduce the above copyright notice, 
//   this list of conditions and the following disclaimer in the documentation 
//   and/or other materials provided with the distribution.
// * Neither the name of the OpenXOPS Project nor the@names of its contributors 
//   may be used to endorse or promote products derived from this software 
//   without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL OpenXOPS Project BE LIABLE FOR ANY
// DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
// ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//--------------------------------------------------------------------------------

// ***** OpenGL core only *****
//
// libjpeg
//    Copyright (C) 1991-2014, Thomas G. Lane, Guido Vollbeding.
//    this software is based in part on the work of the Independent JPEG Group
//
// zlib
//    Copyright (C) 1995-2013 Jean-loup Gailly and Mark Adler
//
// libpng
//    Copyright (c) 1998-2014 Glenn Randers-Pehrson
//
// ****************************

#include "d3dgraphics.h"

#ifndef GRAPHICS_OPENGL

//! @brief RXgN^
D3DGraphics::D3DGraphics()
{
	pD3D = NULL;
	pd3dDevice = NULL;
	aspect = 1.0f;
	fullscreenflag = false;
	for(int i=0; i<MAX_MODEL; i++){
		pmesh[i] = NULL;
	}
	for(int i=0; i<MAX_TEXTURE; i++){
		ptextures[i] = NULL;
	}

	blockdata = NULL;
	for(int i=0; i<TOTAL_BLOCKTEXTURE; i++){
		mapTextureID[i] = -1;
	}

	StartRenderFlag = false;

	//ptextsprite = NULL;
	pxmsfont = NULL;
	TextureFont = -1;
}

//! @brief fBXgN^
D3DGraphics::~D3DGraphics()
{
	//\[X
	CleanupD3Dresource();

	if( pd3dDevice != NULL ) pd3dDevice->Release();
	if( pD3D != NULL ) pD3D->Release();
}

//! @brief @n
//! iDirectX 9j
//! @param WindowCtrl WindowControlNX̃|C^
//! @param TextureFontFilename gpeNX`tHg̃t@C
//! @param fullscreen falseFEBhE\@trueFtXN[p\
//! @return F0@sF1
int D3DGraphics::InitD3D(WindowControl *WindowCtrl, char *TextureFontFilename, bool fullscreen)
{
	D3DPRESENT_PARAMETERS d3dpp;
	RECT rec;

	GetClientRect(WindowCtrl->GethWnd(), &rec);

	fullscreenflag = fullscreen;

	//D3D9̍쐬
	pD3D = Direct3DCreate9(D3D_SDK_VERSION);
	if( pD3D == NULL ){
		return 1;
	}

	//D3DfoCX̍쐬
	ZeroMemory(&d3dpp, sizeof(d3dpp));
	if( fullscreenflag == false ){
		d3dpp.Windowed = TRUE;
		d3dpp.BackBufferWidth = rec.right;
		d3dpp.BackBufferHeight = rec.bottom;
		d3dpp.BackBufferFormat = D3DFMT_UNKNOWN;
		d3dpp.SwapEffect = D3DSWAPEFFECT_DISCARD;
		d3dpp.EnableAutoDepthStencil = TRUE;
		d3dpp.AutoDepthStencilFormat = D3DFMT_D24S8;
		d3dpp.FullScreen_RefreshRateInHz = 0;
	}
	else{
		D3DDISPLAYMODE dispmode;
		pD3D->GetAdapterDisplayMode(D3DADAPTER_DEFAULT, &dispmode);

		d3dpp.Windowed = FALSE;
		d3dpp.BackBufferWidth = rec.right;
		d3dpp.BackBufferHeight = rec.bottom;
		d3dpp.BackBufferFormat = dispmode.Format;
		d3dpp.SwapEffect = D3DSWAPEFFECT_DISCARD;
		d3dpp.EnableAutoDepthStencil = TRUE;
		d3dpp.AutoDepthStencilFormat = D3DFMT_D24S8;
		d3dpp.FullScreen_RefreshRateInHz = dispmode.RefreshRate;
	}

	if( FAILED( pD3D->CreateDevice(D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL, WindowCtrl->GethWnd(), D3DCREATE_HARDWARE_VERTEXPROCESSING, &d3dpp, &pd3dDevice) ) ){
		if( FAILED( pD3D->CreateDevice(D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL, WindowCtrl->GethWnd(), D3DCREATE_SOFTWARE_VERTEXPROCESSING, &d3dpp, &pd3dDevice) ) ){
			return 1;
		}
	}

	//eNX`tHgp摜̃t@Cݒ
	strcpy(TextureFontFname, TextureFontFilename);

	//`֌W̏ڍׂȐݒ
	if( InitSubset() != 0){
		return 1;
	}
	

	//AXyNgݒ
	aspect = (float)rec.right / (float)rec.bottom;

	//}EXJ[\
	//ShowCursor(FALSE);


	float aspecth, prx, pry, r;
	aspecth = (float)SCREEN_WIDTH/SCREEN_HEIGHT;

	//HUD_myweapon [s, c, ]

	//HUD_A@ݎĂ镐\W
	prx = DegreeToRadian(-39) * aspecth /2;
	pry = DegreeToRadian(-55) /2;
	r = 7.5f;
	HUD_myweapon_x[0] = cos(pry)*r;
	HUD_myweapon_y[0] = sin(pry)*r;
	HUD_myweapon_z[0] = sin(prx)*r;

	//HUD_A@\̕\W
	prx = DegreeToRadian(-52) * aspecth /2;
	pry = DegreeToRadian(-60) /2;
	r = 16.0f;
	HUD_myweapon_x[1] = cos(pry)*r;
	HUD_myweapon_y[1] = sin(pry)*r;
	HUD_myweapon_z[1] = sin(prx)*r;

	return 0;
}

//! @brief Zbg@n
//! iEBhEŏ̕A@Ȃǁj
//! @param WindowCtrl WindowControlNX̃|C^
//! @return F0@҂F1@sF2
int D3DGraphics::ResetD3D(WindowControl *WindowCtrl)
{
	//tH[JXĂȂ҂
	if( pd3dDevice->TestCooperativeLevel() == D3DERR_DEVICELOST ){
		return 1;
	}

	//\[X
	CleanupD3Dresource();

	D3DPRESENT_PARAMETERS d3dpp;
	RECT rec;

	GetClientRect(WindowCtrl->GethWnd(), &rec);

	//D3DfoCX̍쐬
	ZeroMemory(&d3dpp, sizeof(d3dpp));
	if( fullscreenflag == false ){
		d3dpp.Windowed = TRUE;
		d3dpp.BackBufferWidth = rec.right;
		d3dpp.BackBufferHeight = rec.bottom;
		d3dpp.BackBufferFormat = D3DFMT_UNKNOWN;
		d3dpp.SwapEffect = D3DSWAPEFFECT_DISCARD;
		d3dpp.EnableAutoDepthStencil = TRUE;
		d3dpp.AutoDepthStencilFormat = D3DFMT_D24S8;
		d3dpp.FullScreen_RefreshRateInHz = 0;
	}
	else{
		D3DDISPLAYMODE dispmode;
		pD3D->GetAdapterDisplayMode(D3DADAPTER_DEFAULT, &dispmode);

		d3dpp.Windowed = FALSE;
		d3dpp.BackBufferWidth = rec.right;
		d3dpp.BackBufferHeight = rec.bottom;
		d3dpp.BackBufferFormat = dispmode.Format;
		d3dpp.SwapEffect = D3DSWAPEFFECT_DISCARD;
		d3dpp.EnableAutoDepthStencil = TRUE;
		d3dpp.AutoDepthStencilFormat = D3DFMT_D24S8;
		d3dpp.FullScreen_RefreshRateInHz = dispmode.RefreshRate;
	}

	if( FAILED( pd3dDevice->Reset(&d3dpp) ) ){
		return 2;
	}

	//`֌W̏ڍׂȐݒ
	if( InitSubset() != 0){
		return 2;
	}

	return 0;
}

//! @brief `֌W̍וݒ
//! @attention 1xsĂB
int D3DGraphics::InitSubset()
{
	//Cg
	//pd3dDevice->SetRenderState(D3DRS_AMBIENT, D3DCOLOR_ARGB(0,255,255,255) );
	pd3dDevice->LightEnable(0, FALSE);
	pd3dDevice->SetRenderState(D3DRS_LIGHTING, FALSE);

	//tHO
	float fog_st = 100;
	float fog_end = 800;
	pd3dDevice->SetRenderState(D3DRS_FOGENABLE, TRUE);
	pd3dDevice->SetRenderState(D3DRS_FOGCOLOR, D3DCOLOR_RGBA(0, 0, 0, 0));
	pd3dDevice->SetRenderState(D3DRS_FOGTABLEMODE, D3DFOG_LINEAR);
	pd3dDevice->SetRenderState(D3DRS_FOGVERTEXMODE, D3DFOG_NONE);
	pd3dDevice->SetRenderState(D3DRS_FOGSTART,*(DWORD*)(&fog_st));
	pd3dDevice->SetRenderState(D3DRS_FOGEND,  *(DWORD*)(&fog_end));

	// eNX`tB^g
	pd3dDevice->SetSamplerState(0, D3DSAMP_MINFILTER, D3DTEXF_LINEAR);
	pd3dDevice->SetSamplerState(0, D3DSAMP_MAGFILTER, D3DTEXF_LINEAR);
	pd3dDevice->SetSamplerState(0, D3DSAMP_MIPFILTER, D3DTEXF_LINEAR);

	//~bv}bv̏ڍ׃x (LOD) oCAXw肷B
	float LODBias = -0.2f;
	pd3dDevice->SetSamplerState(0, D3DSAMP_MIPMAPLODBIAS, *((LPDWORD)(&LODBias)) );

	//At@EufBOs
	pd3dDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, TRUE);

	//ߏs
	pd3dDevice->SetRenderState(D3DRS_SRCBLEND, D3DBLEND_SRCALPHA);
	pd3dDevice->SetRenderState(D3DRS_DESTBLEND, D3DBLEND_INVSRCALPHA);

	//At@eXgɑΉĂ邩`FbN
	D3DCAPS9 Caps;
	pd3dDevice->GetDeviceCaps(&Caps);
	if( Caps.AlphaCmpCaps & D3DPCMPCAPS_GREATEREQUAL ){
		//At@eXgݒ
		//@SɓȃsNZ͕`悵Ȃ
		pd3dDevice->SetRenderState(D3DRS_ALPHAREF, (DWORD)0x00000001);
		pd3dDevice->SetRenderState(D3DRS_ALPHATESTENABLE, TRUE); 
		pd3dDevice->SetRenderState(D3DRS_ALPHAFUNC, D3DCMP_GREATEREQUAL);
	}

	//[xobt@r֐
	pd3dDevice->SetRenderState(D3DRS_ZWRITEENABLE, TRUE);
	pd3dDevice->SetRenderState(D3DRS_ZFUNC, D3DCMP_LESS);

	//|S̗E\
	pd3dDevice->SetRenderState(D3DRS_CULLMODE, D3DCULL_CCW);


	//eLXgXvCg
	if( FAILED( D3DXCreateSprite( pd3dDevice, &ptextsprite ) ) ){
		return 1;
	}
	//tHgFlr SVbN@TCYF18
	HRESULT hr = D3DXCreateFont( pd3dDevice, -18, 0, FW_NORMAL, 1, FALSE, DEFAULT_CHARSET, OUT_DEFAULT_PRECIS,
								DEFAULT_QUALITY, DEFAULT_PITCH | FF_DONTCARE, "lr SVbN", &pxmsfont);
	if( FAILED(hr) ) return 1;

	//eNX`tHgp摜擾
	TextureFont = LoadTexture(TextureFontFname, true, false);
	return 0;
}

//! @brief foCX̃\[X
void D3DGraphics::CleanupD3Dresource()
{
	if( TextureFont != -1 ){ CleanupTexture(TextureFont); }
	if( pxmsfont != NULL ){
		pxmsfont->Release();
		pxmsfont = NULL;
	}
	if( ptextsprite != NULL ){ ptextsprite->Release(); }

	CleanupMapdata();

	for(int i=0; i<MAX_MODEL; i++){
		CleanupModel(i);
	}
	for(int i=0; i<MAX_TEXTURE; i++){
		CleanupTexture(i);
	}
}

//! @brief ft@Cǂݍށi.xj
//! @param filename t@C
//! @return FfFԍi0ȏj@sF-1
int D3DGraphics::LoadModel(char* filename)
{
	int id = -1;

	//󂢂ĂvfT
	for(int i=0; i<MAX_MODEL; i++){
		if( pmesh[i] == NULL ){
			id = i;
			break;
		}
	}
	if( id == -1 ){ return -1; }

	LPD3DXBUFFER pD3DXMtrlBuffer;

#ifdef PATH_DELIMITER_SLASH
	//pX؂蕶ϊ
	filename = ChangePathDelimiter(filename);
#endif

	//.xt@Cobt@[ɓǂݍ
	if( FAILED( D3DXLoadMeshFromX( filename, D3DXMESH_SYSTEMMEM, pd3dDevice, NULL, 
				&pD3DXMtrlBuffer, NULL, &nummaterials[id], &pmesh[id] ) ) ) {
		return -1;
	}

	//}eA擾
	D3DXMATERIAL* d3dxMaterials = (D3DXMATERIAL*)pD3DXMtrlBuffer->GetBufferPointer();
	int num = nummaterials[id];
	pmaterials[id] = new D3DMATERIAL9[num];
	if( pmaterials[id]  == NULL ) return -3;

	//\̂ɑ
	for( int i=0; i<num; i=i+1 ){
		pmaterials[id][i] = d3dxMaterials[i].MatD3D;
		pmaterials[id][i].Ambient = pmaterials[id][i].Diffuse;
	}

	//obt@J
	pD3DXMtrlBuffer->Release();

	return id;
}

//! @brief ft@C̒ԃf[^쐬i[tBOj
//! @param idA fA̔Fԍ
//! @param idB fB̔Fԍ
//! @return FVfFԍi0ȏj@sF-1
//! @attention fAƃfB́A_E|SECfbNXłKv܂B
//! @attention ꂼ̃ff[^Ȃ _قȂꍇAsɎs܂B
int D3DGraphics::MorphingModel(int idA, int idB)
{
	//f[^ׂ
	if( (idA < 0)||((MAX_MODEL -1) < idA) ){ return -1; }
	if( pmesh[idA] == NULL ){ return -1; }
	if( (idB < 0)||((MAX_MODEL -1) < idB) ){ return -1; }
	if( pmesh[idB] == NULL ){ return -1; }

	int idN = -1;
	int numvA, numvB;
	LPDIRECT3DVERTEXBUFFER9 pvbA, pvbB, pvbN;
	D3DXVECTOR3 *pVerticesA, *pVerticesB, *pVerticesN;
	int FVFsize;

	//󂢂ĂvfT
	for(int i=0; i<MAX_MODEL; i++){
		if( pmesh[i] == NULL ){
			idN = i;
			break;
		}
	}
	if( idN == -1 ){ return -1; }

	//_擾
	numvA = pmesh[idA]->GetNumVertices();
	numvB = pmesh[idB]->GetNumVertices();

	//_ǂׂ
	if( numvA != numvB ){ return -1; }

	//_f[^Rs[iIɗ̈mۗp̃_~[j
	if( pmesh[idA]->CloneMeshFVF(pmesh[idA]->GetOptions(), pmesh[idA]->GetFVF(), pd3dDevice, &pmesh[idN]) != D3D_OK ){
		return -1;
	}

	//}eARs[
	int num = nummaterials[idA];
	nummaterials[idN] = nummaterials[idA];
	pmaterials[idN] = new D3DMATERIAL9[num];
	if( pmaterials[idN]  == NULL ) return -1;
	for( int i=0; i<num; i=i+1 ){
		pmaterials[idN][i] = pmaterials[idA][i];
	}

	//obt@[擾
	pmesh[idA]->GetVertexBuffer(&pvbA);
	pmesh[idB]->GetVertexBuffer(&pvbB);
	pmesh[idN]->GetVertexBuffer(&pvbN);

	//1_̃oCg擾
	FVFsize = D3DXGetFVFVertexSize(pmesh[idN]->GetFVF());

	//e_ǂݏovZ
	for(int i=0; i<numvA; i++){
		pvbA->Lock(i*FVFsize, sizeof(D3DXVECTOR3), (void**)&pVerticesA, D3DLOCK_READONLY);
		pvbB->Lock(i*FVFsize, sizeof(D3DXVECTOR3), (void**)&pVerticesB, D3DLOCK_READONLY);
		pvbN->Lock(i*FVFsize, sizeof(D3DXVECTOR3), (void**)&pVerticesN, 0);

		//ω
		pVerticesN->x = (pVerticesA->x + pVerticesB->x)/2;
		pVerticesN->y = (pVerticesA->y + pVerticesB->y)/2;
		pVerticesN->z = (pVerticesA->z + pVerticesB->z)/2;

		pvbA->Unlock();
		pvbB->Unlock();
		pvbN->Unlock();
	}

	return idN;
}

//! @brief ft@C
//! @param id fFԍ
void D3DGraphics::CleanupModel(int id)
{
	if( (id < 0)||((MAX_MODEL -1) < id) ){ return; }
	if( pmesh[id] != NULL ){
		delete [] pmaterials[id];

		pmesh[id]->Release();
		pmesh[id] = NULL;
	}
}

//! @brief eNX`ǂݍ
//! @param filename t@C
//! @param texturefont eNX`tHgtO
//! @param BlackTransparent 𓧉߂
//! @return FeNX`Fԍi0ȏj@sF-1
int D3DGraphics::LoadTexture(char* filename, bool texturefont, bool BlackTransparent)
{
	int id = -1;
	D3DXIMAGE_INFO info;
	int MipLevels;

	//󂢂ĂFԍT
	for(int i=0; i<MAX_TEXTURE; i++){
		if( ptextures[i] == NULL ){
			id = i;
			break;
		}
	}
	if( id == -1 ){ return -1; }

#ifdef PATH_DELIMITER_SLASH
	//pX؂蕶ϊ
	filename = ChangePathDelimiter(filename);
#endif

	//t@C擾
	if( D3DXGetImageInfoFromFile(filename, &info) != D3D_OK ){ return -1; }

	//~bv}bvxݒ
	if( texturefont == true ){
		MipLevels = 1;
	}
	else{
		MipLevels = 4;//D3DX_DEFAULT;
	}

	//eNX`ǂݍ
	if( BlackTransparent == false ){
		if( FAILED( D3DXCreateTextureFromFileEx(pd3dDevice, filename, info.Width, info.Height, MipLevels, 0, D3DFMT_UNKNOWN, D3DPOOL_MANAGED, D3DX_DEFAULT, D3DX_DEFAULT, 0x00000000, NULL, NULL, &ptextures[id]) ) ) {
			return -1;
		}
	}
	else{
		if( FAILED( D3DXCreateTextureFromFileEx(pd3dDevice, filename, info.Width, info.Height, MipLevels, 0, D3DFMT_A1R5G5B5, D3DPOOL_MANAGED, D3DX_DEFAULT, D3DX_DEFAULT, D3DCOLOR_ARGB(255, 0, 0, 0), NULL, NULL, &ptextures[id]) ) ) {
			return -1;
		}
	}
	return id;
}

//! @brief eNX`̃TCY擾
//! @param id eNX`Fԍ
//! @param width 󂯎|C^
//! @param height 󂯎|C^
//! @return F0@sF1
//! @attention T[tFCX̃TCY擾܂BGPUɃ[hꂽTCYłAeNX`ijƈقȂꍇ܂B
int D3DGraphics::GetTextureSize(int id, int *width, int *height)
{
	//ȔFԍw肳ĂAԂB
	if( id == -1 ){ return 1; }
	if( ptextures[id] == NULL ){ return 1; }

	IDirect3DSurface9 *surface;
	D3DSURFACE_DESC desc;

	//T[tFCX擾
	ptextures[id]->GetSurfaceLevel(0, &surface);

	//ƍ擾
	surface->GetDesc(&desc);
	*width = desc.Width;
	*height = desc.Height;

	//T[tFCXJ
	surface->Release();

	return 0;
}

//! @brief eNX`
//! @param id eNX`Fԍ
void D3DGraphics::CleanupTexture(int id)
{
	if( (id < 0)||((MAX_TEXTURE -1) < id) ){ return; }
	if( ptextures[id] != NULL ){
		ptextures[id]->Release();
		ptextures[id] = NULL;
	}
}

//! @brief SĂ̕`揈Jn
//! @return F0@sF1
//! @attention `揈̍ŏɌĂяoKv܂B
int D3DGraphics::StartRender()
{
	if( StartRenderFlag == true ){ return 1; }

	//̈
	pd3dDevice->Clear(0, NULL, D3DCLEAR_TARGET | D3DCLEAR_ZBUFFER, D3DCOLOR_XRGB(0,0,0), 1.0f, 0);

	if( SUCCEEDED( pd3dDevice->BeginScene() ) ){
		//Zobt@
		pd3dDevice->SetRenderState(D3DRS_ZENABLE, D3DZB_TRUE);

		//W[n_Ƀ[hϊs
		ResetWorldTransform();

		//`撆̃tO𗧂Ă
		StartRenderFlag = true;
		return 0;
	}

	return 1;
}

//! @brief SĂ̕`揈I
//! @return Ffalse@sFtrue
//! @attention `揈̍ŌɌĂяoKv܂B
bool D3DGraphics::EndRender()
{
	//`撆ȂI
	if( StartRenderFlag == true ){
		pd3dDevice->EndScene();
	}

	HRESULT hr = pd3dDevice->Present(NULL, NULL, NULL, NULL);

	//tO false 
	StartRenderFlag = false;

	if( hr == D3DERR_DEVICELOST ){
		return true;
	}
	return false;
}

//! @brief Zobt@Zbg
void D3DGraphics::ResetZbuffer()
{
	//Zobt@xɂAAēxL
	pd3dDevice->SetRenderState(D3DRS_ZENABLE, D3DZB_FALSE);
	pd3dDevice->Clear(0, NULL, D3DCLEAR_ZBUFFER, D3DCOLOR_XRGB(0,0,0), 1.0f, 0);
	pd3dDevice->SetRenderState(D3DRS_ZENABLE, D3DZB_TRUE);
}

//! @brief [hԂ_i0,0,0jɖ߂@Ȃ
void D3DGraphics::ResetWorldTransform()
{
	D3DXMATRIX matWorld;
	D3DXMatrixIdentity(&matWorld);
	pd3dDevice->SetTransform(D3DTS_WORLD, &matWorld);
}

//! @brief [hԂ̍WEpxEg嗦ݒ
//! @param x XW
//! @param y YW
//! @param z ZW
//! @param rx px
//! @param ry cpx
//! @param size g嗦
void D3DGraphics::SetWorldTransform(float x, float y, float z, float rx, float ry, float size)
{
	SetWorldTransform(x, y, z, rx, ry, 0.0f, size);
}

//! @brief [hԂ̍WEpxEg嗦ݒ
//! @param x XW
//! @param y YW
//! @param z ZW
//! @param rx px
//! @param ry1 cpx
//! @param ry2 cpx
//! @param size g嗦
void D3DGraphics::SetWorldTransform(float x, float y, float z, float rx, float ry1, float ry2, float size)
{
	D3DXMATRIX matWorld;
	D3DXMATRIX matWorld1, matWorld2, matWorld3, matWorld4, matWorld5;

	//s쐬
	D3DXMatrixTranslation(&matWorld1, x, y, z);
	D3DXMatrixRotationY(&matWorld2, rx);
	D3DXMatrixRotationX(&matWorld3, ry1);
	D3DXMatrixRotationZ(&matWorld4, ry2);
	D3DXMatrixScaling(&matWorld5, size, size, size);

	//vZ
	matWorld = matWorld5 * matWorld4 * matWorld3 * matWorld2 * matWorld1;

	//Kp
	pd3dDevice->SetTransform( D3DTS_WORLD, &matWorld );
}

//! @brief [hԂ̍WEpxEg嗦ݒiGtFNgpj
//! @param x XW
//! @param y YW
//! @param z ZW
//! @param rx px
//! @param ry cpx
//! @param rt ]px
//! @param size g嗦
void D3DGraphics::SetWorldTransformEffect(float x, float y, float z, float rx, float ry, float rt, float size)
{
	D3DXMATRIX matWorld;
	D3DXMATRIX matWorld1, matWorld2, matWorld3, matWorld4, matWorld5;

	//s쐬
	D3DXMatrixTranslation(&matWorld1, x, y, z);
	D3DXMatrixRotationY(&matWorld2, rx);
	D3DXMatrixRotationZ(&matWorld3, ry);
	D3DXMatrixRotationX(&matWorld4, rt);
	D3DXMatrixScaling(&matWorld5, size, size, size);

	//vZ
	matWorld = matWorld5 * matWorld4 * matWorld3 * matWorld2 * matWorld1;

	//Kp
	pd3dDevice->SetTransform( D3DTS_WORLD, &matWorld );
}

//! @brief [hԂlꏊɐݒ
//! @param x XW
//! @param y YW
//! @param z ZW
//! @param mx 茳_ɂ fXW
//! @param my 茳_ɂ fYW
//! @param mz 茳_ɂ fZW
//! @param rx px
//! @param ry cpx
//! @param size g嗦
void D3DGraphics::SetWorldTransformHumanWeapon(float x, float y, float z, float mx, float my, float mz, float rx, float ry, float size)
{
	D3DXMATRIX matWorld;
	D3DXMATRIX matWorld1, matWorld2, matWorld3, matWorld4, matWorld5;

	//s쐬
	D3DXMatrixTranslation(&matWorld1, x, y, z);
	D3DXMatrixRotationY(&matWorld2, rx);
	D3DXMatrixRotationX(&matWorld3, ry);
	D3DXMatrixTranslation(&matWorld4, mx, my, mz);
	D3DXMatrixScaling(&matWorld5, size, size, size);

	//vZ
	matWorld = matWorld5 * matWorld4 * matWorld3 * matWorld2 * matWorld1;

	//Kp
	pd3dDevice->SetTransform( D3DTS_WORLD, &matWorld );
}

//! @brief [hԂĂ镐\ꏊɐݒ
//! @param rotation ]
//! @param camera_x JXW
//! @param camera_y JYW
//! @param camera_z JZW
//! @param camera_rx J̉px
//! @param camera_ry J̏cpx
//! @param rx ̂̏cpx
//! @param size \TCY
//! @note rotationEE@trueFݎĂ镐łB@falseF\̕łBirx ͖܂j
//! @todo ʒuTCY̔
void D3DGraphics::SetWorldTransformPlayerWeapon(bool rotation, float camera_x, float camera_y, float camera_z, float camera_rx, float camera_ry, float rx, float size)
{
	D3DXMATRIX matWorld;
	D3DXMATRIX matWorld1, matWorld2, matWorld3, matWorld4, matWorld5, matWorld6;

	size = size * 0.3f;

	//s쐬
	D3DXMatrixTranslation(&matWorld1, camera_x, camera_y, camera_z);
	D3DXMatrixRotationY(&matWorld2, camera_rx *-1);
	D3DXMatrixRotationZ(&matWorld3, camera_ry);
	// matWorld4 = [s, c, ]
	if( rotation == true ){
		D3DXMatrixTranslation(&matWorld4, HUD_myweapon_x[0], HUD_myweapon_y[0], HUD_myweapon_z[0]);
		D3DXMatrixRotationY(&matWorld5, rx);
	}
	else{
		D3DXMatrixTranslation(&matWorld4, HUD_myweapon_x[1], HUD_myweapon_y[1], HUD_myweapon_z[1]);
		D3DXMatrixRotationY(&matWorld5, D3DX_PI);
	}
	D3DXMatrixScaling(&matWorld6, size, size, size);

	//vZ
	matWorld = matWorld6 * matWorld5 * matWorld4 * matWorld3 * matWorld2 * matWorld1;

	//Kp
	pd3dDevice->SetTransform( D3DTS_WORLD, &matWorld );
}

//! @brief [hԂ̍W擾
//! @param *x x󂯎|C^
//! @param *y y󂯎|C^
//! @param *z z󂯎|C^
void D3DGraphics::GetWorldTransformPos(float *x, float *y, float *z)
{
	D3DXMATRIX matWorld;
	pd3dDevice->GetTransform( D3DTS_WORLD, &matWorld );
	*x = matWorld._41;
	*y = matWorld._42;
	*z = matWorld._43;
}

//! @brief tHOݒ
//! @param skynumber ̔ԍ
void D3DGraphics::SetFog(int skynumber)
{
	D3DCOLOR skycolor;

	//̔ԍɂF
	switch(skynumber){
		case 1: skycolor = D3DCOLOR_RGBA(64, 64+16, 64, 0); break;
		case 2: skycolor = D3DCOLOR_RGBA(16, 16, 16, 0); break;
		case 3: skycolor = D3DCOLOR_RGBA(0, 16, 32, 0); break;
		case 4: skycolor = D3DCOLOR_RGBA(32, 16, 16, 0); break;
		case 5: skycolor = D3DCOLOR_RGBA(64, 32, 32, 0); break;
		default: skycolor = D3DCOLOR_RGBA(0, 0, 0, 0); break;
	}

	//tHOݒ
	pd3dDevice->SetRenderState(D3DRS_FOGCOLOR, skycolor);
}

//! @brief Ji_jݒ
//! @param camera_x JXW
//! @param camera_y JYW
//! @param camera_z JZW
//! @param camera_rx J̉px
//! @param camera_ry J̏cpx
//! @param viewangle p
void D3DGraphics::SetCamera(float camera_x, float camera_y, float camera_z, float camera_rx, float camera_ry, float viewangle)
{
	float vUpVecF;
	D3DXMATRIX matWorld;
	D3DXMATRIXA16 matView;

	//camera_ry -PI`PI ̊ԂɐK
	for(; camera_ry>D3DX_PI; camera_ry -= D3DX_PI*2){}
	for(; camera_ry<D3DX_PI*-1; camera_ry += D3DX_PI*2){}

	//J̌
	if( fabs(camera_ry) < D3DX_PI/2 ){
		vUpVecF = 1.0f;
	}
	else{
		vUpVecF = -1.0f;
	}

	D3DXMatrixIdentity(&matWorld);
	pd3dDevice->SetTransform(D3DTS_WORLD, &matWorld);

	//JW
	D3DXVECTOR3 vEyePt( camera_x, camera_y, camera_z );
	D3DXVECTOR3 vLookatPt( cos(camera_rx)*cos(camera_ry) + camera_x, sin(camera_ry) + camera_y, sin(camera_rx)*cos(camera_ry) + camera_z );
	D3DXVECTOR3 vUpVec( 0.0f, vUpVecF, 0.0f );
	D3DXMatrixLookAtLH( &matView, &vEyePt, &vLookatPt, &vUpVec );
	pd3dDevice->SetTransform( D3DTS_VIEW, &matView );

	//Jݒiˉeϊsjviewangle
	D3DXMATRIXA16 matProj;
	D3DXMatrixPerspectiveFovLH( &matProj, viewangle, aspect, CLIPPINGPLANE_NEAR, CLIPPINGPLANE_FAR);
	pd3dDevice->SetTransform(D3DTS_PROJECTION, &matProj);
}

//! @brief }bvf[^荞
//! @param in_blockdata ubNf[^
//! @param directory ubNf[^݂fBNg
void D3DGraphics::LoadMapdata(BlockDataInterface* in_blockdata, char *directory)
{
	//ubNf[^w肳ĂȂ΁AȂB
	if( in_blockdata == NULL ){ return; }

	char fname[MAX_PATH];
	char fnamefull[MAX_PATH];
	//int bs;
	struct blockdata data;
	int vID[4];
	int uvID[4];

	//NXݒ
	blockdata = in_blockdata;

	//eNX`ǂݍ
	for(int i=0; i<TOTAL_BLOCKTEXTURE; i++){
		//eNX`擾
		blockdata->GetTexture(fname, i);

		if( strcmp(fname, "") == 0 ){	//w肳ĂȂ΁AȂ
			mapTextureID[i] = -1;
		}
		else{
			//ufBNg{t@Cv𐶐Aǂݍ
			strcpy(fnamefull, directory);
			strcat(fnamefull, fname);
			mapTextureID[i] = LoadTexture(fnamefull, false, false);
		}
	}

#ifdef BLOCKDATA_GPUMEMORY
	VERTEXTXTA* pVertices;

	//ubN擾
	bs = blockdata->GetTotaldatas();

	//ubÑobt@[쐬
	pd3dDevice->CreateVertexBuffer(bs*6*4*sizeof(VERTEXTXTA),0,D3DFVF_XYZ|D3DFVF_DIFFUSE|D3DFVF_TEX1,D3DPOOL_DEFAULT,&g_pVB,NULL);

	for(int i=0; i<bs; i++){
		//f[^擾
		blockdata->Getdata(&data, i);

		for(int j=0; j<6; j++){
			//ʂ̒_f[^̊֘At擾
			blockdataface(j, &vID[0], &uvID[0]);

			//GPUbNi1ʕj
			g_pVB->Lock((i*6+j)*4*sizeof(VERTEXTXTA), 4*sizeof(VERTEXTXTA), (void**)&pVertices, 0);

			//_WEUVWEFݒ
			pVertices[0].position = D3DXVECTOR3( data.x[ vID[1] ], data.y[ vID[1] ], data.z[ vID[1] ] );
			pVertices[0].tu       = data.material[j].u[ uvID[1] ];
			pVertices[0].tv       = data.material[j].v[ uvID[1] ];
			pVertices[1].position = D3DXVECTOR3( data.x[ vID[2] ], data.y[ vID[2] ], data.z[ vID[2] ] );
			pVertices[1].tu       = data.material[j].u[ uvID[2] ];
			pVertices[1].tv       = data.material[j].v[ uvID[2] ];
			pVertices[2].position = D3DXVECTOR3( data.x[ vID[0] ], data.y[ vID[0] ], data.z[ vID[0] ] );
			pVertices[2].tu       = data.material[j].u[ uvID[0] ];
			pVertices[2].tv       = data.material[j].v[ uvID[0] ];
			pVertices[3].position = D3DXVECTOR3( data.x[ vID[3] ], data.y[ vID[3] ], data.z[ vID[3] ] );
			pVertices[3].tu       = data.material[j].u[ uvID[3] ];
			pVertices[3].tv       = data.material[j].v[ uvID[3] ];
			for(int k=0; k<4; k++){
				pVertices[k].color = D3DCOLOR_COLORVALUE(data.material[j].shadow, data.material[j].shadow, data.material[j].shadow, 1.0f);
			}

			//GPŨbN
			g_pVB->Unlock();
		}
	}
#else
	//ubN擾
	bs = blockdata->GetTotaldatas();

	for(int i=0; i<bs; i++){
		//f[^擾
		blockdata->Getdata(&data, i);

		for(int j=0; j<6; j++){
			//ʂ̒_f[^̊֘At擾
			blockdataface(j, vID, uvID);

			//_WEUVWEFݒ
			g_pVertices[i][j][0].position = D3DXVECTOR3( data.x[ vID[1] ], data.y[ vID[1] ], data.z[ vID[1] ] );
			g_pVertices[i][j][0].tu       = data.material[j].u[ uvID[1] ];
			g_pVertices[i][j][0].tv       = data.material[j].v[ uvID[1] ];
			g_pVertices[i][j][1].position = D3DXVECTOR3( data.x[ vID[2] ], data.y[ vID[2] ], data.z[ vID[2] ] );
			g_pVertices[i][j][1].tu       = data.material[j].u[ uvID[2] ];
			g_pVertices[i][j][1].tv       = data.material[j].v[ uvID[2] ];
			g_pVertices[i][j][2].position = D3DXVECTOR3( data.x[ vID[0] ], data.y[ vID[0] ], data.z[ vID[0] ] );
			g_pVertices[i][j][2].tu       = data.material[j].u[ uvID[0] ];
			g_pVertices[i][j][2].tv       = data.material[j].v[ uvID[0] ];
			g_pVertices[i][j][3].position = D3DXVECTOR3( data.x[ vID[3] ], data.y[ vID[3] ], data.z[ vID[3] ] );
			g_pVertices[i][j][3].tu       = data.material[j].u[ uvID[3] ];
			g_pVertices[i][j][3].tv       = data.material[j].v[ uvID[3] ];
			for(int k=0; k<4; k++){
				g_pVertices[i][j][k].color = D3DCOLOR_COLORVALUE(data.material[j].shadow, data.material[j].shadow, data.material[j].shadow, 1.0f);
			}
		}
	}
#endif
}

//! @brief }bvf[^`
//! @param wireframe C[t[\
void D3DGraphics::DrawMapdata(bool wireframe)
{
	//ubNf[^ǂݍ܂ĂȂ΁AȂB
	if( blockdata == NULL ){ return; }

	struct blockdata data;
	int textureID;

	if( wireframe == true ){
		//C[t[\
		for(int i=0; i<bs; i++){
			blockdata->Getdata(&data, i);
			Drawline(data.x[0], data.y[0], data.z[0], data.x[1], data.y[1], data.z[1]);
			Drawline(data.x[1], data.y[1], data.z[1], data.x[2], data.y[2], data.z[2]);
			Drawline(data.x[2], data.y[2], data.z[2], data.x[3], data.y[3], data.z[3]);
			Drawline(data.x[3], data.y[3], data.z[3], data.x[0], data.y[0], data.z[0]);
			Drawline(data.x[4], data.y[4], data.z[4], data.x[5], data.y[5], data.z[5]);
			Drawline(data.x[5], data.y[5], data.z[5], data.x[6], data.y[6], data.z[6]);
			Drawline(data.x[6], data.y[6], data.z[6], data.x[7], data.y[7], data.z[7]);
			Drawline(data.x[7], data.y[7], data.z[7], data.x[4], data.y[4], data.z[4]);
			Drawline(data.x[0], data.y[0], data.z[0], data.x[4], data.y[4], data.z[4]);
			Drawline(data.x[1], data.y[1], data.z[1], data.x[5], data.y[5], data.z[5]);
			Drawline(data.x[2], data.y[2], data.z[2], data.x[6], data.y[6], data.z[6]);
			Drawline(data.x[3], data.y[3], data.z[3], data.x[7], data.y[7], data.z[7]);
		}
		return;
	}

	//[xobt@r֐ݒ
	//pd3dDevice->SetRenderState(D3DRS_ZFUNC, D3DCMP_LESS);


#ifdef BLOCKDATA_GPUMEMORY
	//f[^ݒ
	pd3dDevice->SetStreamSource(0,g_pVB,0,sizeof(VERTEXTXTA));

	for(textureID=0; textureID<TOTAL_BLOCKTEXTURE; textureID++){
		//eNX`ɓǂݍ߂ĂȂΐݒ
		if( mapTextureID[textureID] == -1 ){
			pd3dDevice->SetFVF(D3DFVF_XYZ|D3DFVF_DIFFUSE);
			pd3dDevice->SetTexture(0, NULL);
		}
		else if( ptextures[ mapTextureID[textureID] ] == NULL ){
			pd3dDevice->SetFVF(D3DFVF_XYZ|D3DFVF_DIFFUSE);
			pd3dDevice->SetTexture(0, NULL);
		}
		else{
			pd3dDevice->SetFVF(D3DFVF_XYZ|D3DFVF_DIFFUSE|D3DFVF_TEX1);
			pd3dDevice->SetTexture(0, ptextures[mapTextureID[textureID]] );
		}

		for(int i=0; i<bs; i++){
			//f[^擾
			blockdata->Getdata(&data, i);

			for(int j=0; j<6; j++){
				//eNX`Fԍ擾
				int ID = data.material[j].textureID;

				if( textureID == ID ){
					//ʂ`
					pd3dDevice->DrawPrimitive(D3DPT_TRIANGLESTRIP, (i*6+j)*4, 2);
				}
			}
		}
	}
#else
	//f[^ݒ
	pd3dDevice->SetFVF(D3DFVF_XYZ|D3DFVF_DIFFUSE|D3DFVF_TEX1);

	for(textureID=0; textureID<TOTAL_BLOCKTEXTURE; textureID++){
		//eNX`ɓǂݍ߂ĂȂΐݒ
		if( mapTextureID[textureID] == -1 ){
			pd3dDevice->SetFVF(D3DFVF_XYZ|D3DFVF_DIFFUSE);
			pd3dDevice->SetTexture(0, NULL);
		}
		else if( ptextures[ mapTextureID[textureID] ] == NULL ){
			pd3dDevice->SetFVF(D3DFVF_XYZ|D3DFVF_DIFFUSE);
			pd3dDevice->SetTexture(0, NULL);
		}
		else{
			pd3dDevice->SetFVF(D3DFVF_XYZ|D3DFVF_DIFFUSE|D3DFVF_TEX1);
			pd3dDevice->SetTexture(0, ptextures[mapTextureID[textureID]] );
		}

		for(int i=0; i<bs; i++){
			//f[^擾
			blockdata->Getdata(&data, i);

			for(int j=0; j<6; j++){
				//eNX`Fԍ擾
				int ID = data.material[j].textureID;

				if( textureID == ID ){
					//ʂ`
					pd3dDevice->DrawPrimitiveUP(D3DPT_TRIANGLESTRIP, 2, g_pVertices[i][j], sizeof(VERTEXTXTA));
				}
			}
		}
	}
#endif

	//[xobt@r֐ɖ߂
	//pd3dDevice->SetRenderState(D3DRS_ZFUNC, D3DCMP_LESSEQUAL);
}

//! @brief }bveNX`擾
//! @param id eNX`ԍ
//! @return eNX`FԍisF-1j
int D3DGraphics::GetMapTextureID(int id)
{
	if( (id < 0)||((TOTAL_BLOCKTEXTURE -1) < id ) ){ return -1; }
	return mapTextureID[id];
}

//! @brief }bvf[^
void D3DGraphics::CleanupMapdata()
{
	//eNX`J
	for(int i=0; i<TOTAL_BLOCKTEXTURE; i++){
		CleanupTexture(mapTextureID[i]);
	}

#ifdef BLOCKDATA_GPUMEMORY
	//_f[^
	if( g_pVB != NULL ){
		g_pVB->Release();
		g_pVB = NULL;
	}
#endif
	bs = 0;

	blockdata = NULL;
}

//! @brief ft@C`
//! @param id_model fFԍ
//! @param id_texture eNX`Fԍ
void D3DGraphics::RenderModel(int id_model, int id_texture)
{
	//Ȉݒ肳ĂΎs
	if( id_model == -1 ){ return; }
	//if( id_texture == -1 ){ return; }

	//w肵fĂȂΎs
	if( pmesh[id_model] == NULL) return;

	//`
	for(int i=0; i<(signed)nummaterials[id_model]; i=i+1){
		pd3dDevice->SetMaterial( &pmaterials[id_model][i] );
		if( id_texture == -1 ){
			pd3dDevice->SetTexture(0, NULL);
		}
		else if( ptextures[id_texture] == NULL ){
			pd3dDevice->SetTexture(0, NULL);
		}
		else{
			pd3dDevice->SetTexture( 0, ptextures[id_texture] );
		}
		pmesh[id_model]->DrawSubset(i);
	}
}

//! @brief `
//! @param id_texture eNX`Fԍ
//! @param alpha x@i0.0`1.0@0.0FSj
void D3DGraphics::RenderBoard(int id_texture, float alpha)
{
	//eNX`ݒ肳ĂȂ΁AȂB
	if( id_texture == -1 ){ return; }

	VERTEXTXTA BoardVertices[4];

	//_WEUVWEF/xݒ
	BoardVertices[0].position = D3DXVECTOR3(0.0f, 0.5f, -0.5f);
	BoardVertices[0].tu       = 1.0f;
	BoardVertices[0].tv       = 0.0f;
	BoardVertices[1].position = D3DXVECTOR3(0.0f, -0.5f, -0.5f);
	BoardVertices[1].tu       = 1.0f;
	BoardVertices[1].tv       = 1.0f;
	BoardVertices[2].position = D3DXVECTOR3(0.0f, 0.5f, 0.5f);
	BoardVertices[2].tu       = 0.0f;
	BoardVertices[2].tv       = 0.0f;
	BoardVertices[3].position = D3DXVECTOR3(0.0f, -0.5f, 0.5f);
	BoardVertices[3].tu       = 0.0f;
	BoardVertices[3].tv       = 1.0f;
	for(int i=0; i<4; i++){
		BoardVertices[i].color = D3DCOLOR_COLORVALUE(1.0f, 1.0f, 1.0f, alpha);
	}

	//At@uhݒ
	pd3dDevice->SetTextureStageState(0, D3DTSS_ALPHAOP, D3DTOP_MODULATE);
	pd3dDevice->SetTextureStageState(0, D3DTSS_ALPHAARG1, D3DTA_TEXTURE);
	pd3dDevice->SetTextureStageState(0, D3DTSS_ALPHAARG2, D3DTA_DIFFUSE);

	//eNX`ƃf[^`ݒ肵`
	pd3dDevice->SetTexture(0, ptextures[id_texture]);
	pd3dDevice->SetFVF(D3DFVF_XYZ|D3DFVF_DIFFUSE|D3DFVF_TEX1);
	pd3dDevice->DrawPrimitiveUP(D3DPT_TRIANGLESTRIP, 2, BoardVertices, sizeof(VERTEXTXTA));

	//At@uhɖ߂
	pd3dDevice->SetTextureStageState(0, D3DTSS_ALPHAOP, D3DTOP_SELECTARG1);
	pd3dDevice->SetTextureStageState(0, D3DTSS_ALPHAARG1, D3DTA_TEXTURE);
}

//! @brief ʂ̖邳ݒ
//! @param Width 
//! @param Height 
//! @param Brightness ʂ̖邳@i0 ŕsρA1 ȏŖ邳̓xj
void D3DGraphics::ScreenBrightness(int Width, int Height, int Brightness)
{
	//邳sςȂ珈Ȃiyʉj
	if( Brightness == 0 ){ return; }

	//xݒ肵A`
	float alpha = 0.02f * Brightness;
	Draw2DBox(0, 0, Width, Height, D3DCOLOR_COLORVALUE(1.0f,1.0f,1.0f,alpha));
}

//! @brief yfobNpzS`
void D3DGraphics::Centerline()
{
	ResetWorldTransform();
	Drawline(100.0f, 0.0f, 0.0f, -100.0f, 0.0f, 0.0f);
	Drawline(0.0f, 100.0f, 0.0f, 0.0f, -100.0f, 0.0f);
	Drawline(0.0f, 0.0f, 100.0f, 0.0f, 0.0f, -100.0f);
}

//! @brief yfobNpzΐ`
void D3DGraphics::Drawline(float x1, float y1, float z1, float x2, float y2, float z2)
{
	VERTEXTXTA mv[2];

	mv[0].position = D3DXVECTOR3(x1, y1, z1);
	mv[1].position = D3DXVECTOR3(x2, y2, z2);
	for(int i=0; i<2; i++){
		mv[i].color = 0xFF00FF00;
		mv[i].tu = 0.0f;
		mv[i].tv = 0.0f;
	}

	pd3dDevice->SetTexture(0, NULL);
	pd3dDevice->SetFVF(D3DFVF_XYZ|D3DFVF_DIFFUSE|D3DFVF_TEX1);
	pd3dDevice->DrawPrimitiveUP(D3DPT_LINELIST, 1, mv, sizeof(VERTEXTXTA));
}

//! @brief 2D VXetHgɂeLXg`Jn
//! @attention DirectX ID3DXSprite Ă܂B
void D3DGraphics::Start2DMSFontTextRender()
{
	ptextsprite->Begin(D3DXSPRITE_ALPHABLEND);
}

//! @brief `iVXetHggpj
//! @param x xW
//! @param y yW
//! @param str @isR[hFj
//! @param color F
//! @warning <b>`͔ɒᑬłB</b>ʓŉxĂяoƃptH[}Xɉe܂B
//! @warningusR[hpxɕ`悷vu{ꂪKvȂ̓eNX`tHgpvȂǂ̑ΉuĂB
//! @attention DirectX ID3DXSprite gpAVXetHgŕ`Ă܂B
//! @attention tHg̎ނTCY͌ŒłB@dɏd˂ė̊oȂƌɂȂ܂B
void D3DGraphics::Draw2DMSFontText(int x, int y, char *str, int color)
{
	//if( ptextsprite == NULL ){ return; }

	//eLXgXvCg
	Start2DMSFontTextRender();

	//Wݒ
	D3DXMATRIX matWorld;
	D3DXMatrixIdentity(&matWorld);
	ptextsprite->SetTransform(&matWorld);

	//`
	RECT rc = {x, y, 0, 0};
	pxmsfont->DrawText(ptextsprite, str, -1, &rc, DT_NOCLIP, color);

	//eLXgXvCg
	End2DMSFontTextRender();
}

//! @brief 𒆉ŕ`iVXetHggpj
//! @param x xW
//! @param y yW
//! @param w ̑傫
//! @param h c̑傫
//! @param str @isR[hFj
//! @param color F
//! @warning <b>`͔ɒᑬłB</b>ʓŉxĂяoƃptH[}Xɉe܂B
//! @warningusR[hpxɕ`悷vu{ꂪKvȂ̓eNX`tHgpvȂǂ̑ΉuĂB
//! @attention DirectX ID3DXSprite gpAVXetHgŕ`Ă܂B
//! @attention tHg̎ނTCY͌ŒłB@dɏd˂ė̊oȂƌɂȂ܂B
void D3DGraphics::Draw2DMSFontTextCenter(int x, int y, int w, int h, char *str, int color)
{
	//if( ptextsprite == NULL ){ return; }

	//eLXgXvCg
	Start2DMSFontTextRender();

	//Wݒ
	D3DXMATRIX matWorld;
	D3DXMatrixIdentity(&matWorld);
	ptextsprite->SetTransform(&matWorld);

	//`
	RECT rc = {x, y, x+w, y+h};
	pxmsfont->DrawText(ptextsprite, str, -1, &rc, DT_CENTER, color);

	//eLXgXvCg
	End2DMSFontTextRender();
}

//! @brief 2D VXetHgɂeLXg`I
//! @attention DirectX ID3DXSprite Ă܂B
void D3DGraphics::End2DMSFontTextRender()
{
	ptextsprite->End();
}

//! @brief 2D`pݒ
void D3DGraphics::Start2DRender()
{
	pd3dDevice->SetTextureStageState(0, D3DTSS_ALPHAOP, D3DTOP_MODULATE);
	pd3dDevice->SetTextureStageState(0, D3DTSS_ALPHAARG1, D3DTA_TEXTURE);
	pd3dDevice->SetTextureStageState(0, D3DTSS_ALPHAARG2, D3DTA_DIFFUSE);

	//[xobt@r֐ݒ
	pd3dDevice->SetRenderState(D3DRS_ZFUNC, D3DCMP_ALWAYS);
}

//! @brief `ieNX`tHggpj
//! @param x xW
//! @param y yW
//! @param str @isR[hF<b>s</b>j
//! @param color F
//! @param fontwidth ꕶ̕
//! @param fontheight ꕶ̍
//! @attention dɏd˂ė̊oȂƌɂȂ܂B
void D3DGraphics::Draw2DTextureFontText(int x, int y, char *str, int color, int fontwidth, int fontheight)
{
	//eNX`tHg̎擾ɎsĂ΁AȂ
	if( TextureFont == -1 ){ return; }

	//2D`pݒKp
	Start2DRender();

	int w;
	float font_u, font_v;
	float t_u, t_v;
	TLVERTX pBoxVertices[4];

	//1UVWvZ
	font_u = 1.0f / 16;
	font_v = 1.0f / 16;

	//[hW_ɖ߂
	ResetWorldTransform();

	//eNX`tHgeNX`ɐݒ
	pd3dDevice->SetTexture( 0, ptextures[TextureFont] );

	//f[^`ݒ
	pd3dDevice->SetFVF(D3DFVF_XYZRHW | D3DFVF_DIFFUSE | D3DFVF_TEX1);

	// ^ꂽ[v
	for(int i=0; i<(int)strlen(str); i++){
		//UVWvZ
		w = str[i];
		if( w < 0 ){ w += 256; }
		t_u = (w % 16) * font_u;
		t_v = (w / 16) * font_v;

		//_WEUVWEFݒ
		pBoxVertices[0].x = (float)x + i*fontwidth;
		pBoxVertices[0].y = (float)y;
		pBoxVertices[0].tu = t_u;
		pBoxVertices[0].tv = t_v;
		pBoxVertices[1].x = (float)x + fontwidth + i*fontwidth;
		pBoxVertices[1].y = (float)y;
		pBoxVertices[1].tu = t_u + font_u;
		pBoxVertices[1].tv = t_v;
		pBoxVertices[2].x = (float)x + i*fontwidth;
		pBoxVertices[2].y = (float)y + fontheight;
		pBoxVertices[2].tu = t_u;
		pBoxVertices[2].tv = t_v + font_v;
		pBoxVertices[3].x = (float)x + fontwidth + i*fontwidth;
		pBoxVertices[3].y = (float)y + fontheight;
		pBoxVertices[3].tu = t_u + font_u;
		pBoxVertices[3].tv = t_v + font_v;
		for(int j=0; j<4; j++){
			pBoxVertices[j].z = 0.0f;
			pBoxVertices[j].rhw = 1.0f;
			pBoxVertices[j].color = color;
		}

		//`
		pd3dDevice->DrawPrimitiveUP(D3DPT_TRIANGLESTRIP, 2, pBoxVertices, sizeof(TLVERTX));
	}

	//2D`pݒ
	End2DRender();
}

//! @brief `
//! @param x1 n_ xW
//! @param y1 n_ yW
//! @param x2 I_ xW
//! @param y2 I_ yW
//! @param color F
void D3DGraphics::Draw2DLine(int x1, int y1, int x2, int y2, int color)
{
	TLVERTX pLineVertices[2];

	//2D`pݒKp
	Start2DRender();

	//[hW_ɖ߂
	ResetWorldTransform();

	//_WƐFȂǂݒ
	pLineVertices[0].x = (float)x1;
	pLineVertices[0].y = (float)y1;
	pLineVertices[1].x = (float)x2;
	pLineVertices[1].y = (float)y2;
	for(int i=0; i<2; i++){
		pLineVertices[i].z = 0.0f;
		pLineVertices[i].rhw = 1.0f;
		pLineVertices[i].color = color;
		pLineVertices[i].tu = 0.0f;
		pLineVertices[i].tv = 0.0f;
	}

	pd3dDevice->SetTexture(0, NULL);

	//f[^`ݒ肵A`B
	pd3dDevice->SetFVF(D3DFVF_XYZRHW | D3DFVF_DIFFUSE | D3DFVF_TEX1);
	pd3dDevice->DrawPrimitiveUP(D3DPT_LINELIST, 1, pLineVertices, sizeof(TLVERTX));

	//2D`pݒ
	End2DRender();
}

//! @brief ~i16p`j`
//! @param x S xW
//! @param y S yW
//! @param r a
//! @param color F
void D3DGraphics::Draw2DCycle(int x, int y, int r, int color)
{
	TLVERTX pLineVertices[16+1];

	//2D`pݒKp
	Start2DRender();

	//[hW_ɖ߂
	ResetWorldTransform();

	//_WƐFȂǂݒ
	for(int i=0; i<16+1; i++){
		pLineVertices[i].x = (float)x + cos(DegreeToRadian((360.0f/16.0f)) * i) * r;
		pLineVertices[i].y = (float)y + sin(DegreeToRadian((360.0f/16.0f)) * i) * r;

		pLineVertices[i].z = 0.0f;
		pLineVertices[i].rhw = 1.0f;
		pLineVertices[i].color = color;
		pLineVertices[i].tu = 0.0f;
		pLineVertices[i].tv = 0.0f;
	}

	pd3dDevice->SetTexture(0, NULL);

	//f[^`ݒ肵A`B
	pd3dDevice->SetFVF(D3DFVF_XYZRHW | D3DFVF_DIFFUSE | D3DFVF_TEX1);
	pd3dDevice->DrawPrimitiveUP(D3DPT_LINESTRIP, 16, pLineVertices, sizeof(TLVERTX));

	//2D`pݒ
	End2DRender();
}

//! @brief lp``
//! @param x1  xW
//! @param y1  yW
//! @param x2 E xW
//! @param y2 E yW
//! @param color F
void D3DGraphics::Draw2DBox(int x1, int y1, int x2, int y2, int color)
{
	TLVERTX pBoxVertices[4];

	//2D`pݒKp
	Start2DRender();

	//[hW_ɖ߂
	ResetWorldTransform();

	//_WƐFȂǂݒ
	pBoxVertices[0].x = (float)x1;
	pBoxVertices[0].y = (float)y1;
	pBoxVertices[1].x = (float)x2;
	pBoxVertices[1].y = (float)y1;
	pBoxVertices[2].x = (float)x1;
	pBoxVertices[2].y = (float)y2;
	pBoxVertices[3].x = (float)x2;
	pBoxVertices[3].y = (float)y2;
	for(int i=0; i<4; i++){
		pBoxVertices[i].z = 0.0f;
		pBoxVertices[i].rhw = 1.0f;
		pBoxVertices[i].color = color;
		pBoxVertices[i].tu = 0.0f;
		pBoxVertices[i].tv = 0.0f;
	}

	pd3dDevice->SetTexture(0, NULL);

	//f[^`ݒ肵A`B
	pd3dDevice->SetFVF(D3DFVF_XYZRHW | D3DFVF_DIFFUSE | D3DFVF_TEX1);
	pd3dDevice->DrawPrimitiveUP(D3DPT_TRIANGLESTRIP, 2, pBoxVertices, sizeof(TLVERTX));

	//2D`pݒ
	End2DRender();
}

//! @brief 摜`
//! @param x xW
//! @param y yW
//! @param id eNX`Fԍ
//! @param width 
//! @param height 
//! @param alpha xi0.0`1.0j
void D3DGraphics::Draw2DTexture(int x, int y, int id, int width, int height, float alpha)
{
	//ȃeNX`ԍw肳ĂΏȂ
	if( id == -1 ){ return; }

	TLVERTX pBoxVertices[4];

	//2D`pݒKp
	Start2DRender();

	//[hW_ɖ߂
	ResetWorldTransform();

	//_WEUVWEFݒ
	pBoxVertices[0].x = (float)x;
	pBoxVertices[0].y = (float)y;
	pBoxVertices[0].tu = 0.0f;
	pBoxVertices[0].tv = 0.0f;
	pBoxVertices[1].x = (float)x + width;
	pBoxVertices[1].y = (float)y;
	pBoxVertices[1].tu = 1.0f;
	pBoxVertices[1].tv = 0.0f;
	pBoxVertices[2].x = (float)x;
	pBoxVertices[2].y = (float)y + height;
	pBoxVertices[2].tu = 0.0f;
	pBoxVertices[2].tv = 1.0f;
	pBoxVertices[3].x = (float)x + width;
	pBoxVertices[3].y = (float)y + height;
	pBoxVertices[3].tu = 1.0f;
	pBoxVertices[3].tv = 1.0f;
	for(int i=0; i<4; i++){
		pBoxVertices[i].z = 0.0f;
		pBoxVertices[i].rhw = 1.0f;
		pBoxVertices[i].color = D3DCOLOR_COLORVALUE(1.0f,1.0f,1.0f,alpha);
	}

	//eNX`ƃf[^`ݒ肵A`
	pd3dDevice->SetTexture( 0, ptextures[id] );
	pd3dDevice->SetFVF(D3DFVF_XYZRHW | D3DFVF_DIFFUSE | D3DFVF_TEX1);
	pd3dDevice->DrawPrimitiveUP(D3DPT_TRIANGLESTRIP, 2, pBoxVertices, sizeof(TLVERTX));

	//2D`pݒ
	End2DRender();
}

//! @brief 2D`pݒ
void D3DGraphics::End2DRender()
{
	pd3dDevice->SetTextureStageState(0, D3DTSS_ALPHAOP, D3DTOP_SELECTARG1);
	pd3dDevice->SetTextureStageState(0, D3DTSS_ALPHAARG1, D3DTA_TEXTURE);

	//[xobt@r֐ɖ߂
	pd3dDevice->SetRenderState(D3DRS_ZFUNC, D3DCMP_LESSEQUAL);
}

//! @brief ʂ̃XN[Vbgۑ
//! @param filename t@C
//! @return Ftrue@sFfalse
bool D3DGraphics::SaveScreenShot(char* filename)
{
	LPDIRECT3DSURFACE9 pSurface = NULL;
	HRESULT hr;

	//T[tF[X쐬Aʂ擾
	pd3dDevice->GetBackBuffer(0, 0, D3DBACKBUFFER_TYPE_MONO, &pSurface);

	//T[tFCX摜ɏo
	hr = D3DXSaveSurfaceToFile(filename, D3DXIFF_BMP, pSurface, NULL, NULL);

	//
	pSurface->Release();

	if( hr == D3D_OK ){
		return true;
	}
	return false;
}

//! @brief J[R[h擾
//! @param red ԁi0.0f`1.0fj
//! @param green ΁i0.0f`1.0fj
//! @param blue i0.0f`1.0fj
//! @param alpha xi0.0f`1.0fj
//! @return J[R[h
int D3DGraphics::GetColorCode(float red, float green, float blue, float alpha)
{
	return D3DCOLOR_COLORVALUE(red, green, blue, alpha);
}

#else	//GRAPHICS_OPENGL

//! @brief RXgN^
D3DGraphics::D3DGraphics()
{
	hGLRC = NULL;
	width = 0;
	height = 0;
	SystemFont = NULL;
	now_SystemFontUStr = new WCHAR [1];
	now_SystemFontUStr[0] = NULL;
	SystemFontListIdx = 0;
	SystemFontListIdxSize = 0;
	now_textureid = -1;

	camera_x = 0.0f;
	camera_y = 0.0f;
	camera_z = 0.0f;
	camera_rx = 0.0f;
	camera_ry = 0.0f;
	viewangle = 0.0f;

	blockdata = NULL;
	for(int i=0; i<TOTAL_BLOCKTEXTURE; i++){
		mapTextureID[i] = -1;
	}

	TextureFont = -1;
}

//! @brief fBXgN^
D3DGraphics::~D3DGraphics()
{
	for(int i=0; i<MAX_MODEL; i++){
		CleanupModel(i);
	}
	for(int i=0; i<MAX_TEXTURE; i++){
		CleanupTexture(i);
	}

	if( SystemFont != NULL ){
		DeleteObject(SystemFont);
	}
	if( now_SystemFontUStr != NULL ){
		delete [] now_SystemFontUStr;
	}
	if( SystemFontListIdx != 0 ){
		glDeleteLists(SystemFontListIdx, SystemFontListIdxSize);
	}

	if( hGLRC != NULL ){ wglDeleteContext(hGLRC); }

	//libjpeg
	jpeg_destroy_decompress(&cinfo);
}

//! @brief @n
//! iOpenGL 1.1j
//! @param WindowCtrl WindowControlNX̃|C^
//! @param TextureFontFilename gpeNX`tHg̃t@C
//! @param fullscreen falseFEBhE\@trueFtXN[p\
//! @return F0@sF1
int D3DGraphics::InitD3D(WindowControl *WindowCtrl, char *TextureFontFilename, bool fullscreen)
{
	hWnd = WindowCtrl->GethWnd();

	RECT prc;
	GetClientRect(hWnd, &prc);
	width = prc.right;
	height = prc.bottom;

	//tXN[
	if( fullscreen == true ){
		DEVMODE devmode;
		ZeroMemory(&devmode, sizeof(devmode));
		devmode.dmSize = sizeof(devmode);
		devmode.dmPelsWidth = width;
		devmode.dmPelsHeight = height;
		devmode.dmFields = DM_PELSWIDTH | DM_PELSHEIGHT;

		if( ChangeDisplaySettings(&devmode, CDS_TEST) != DISP_CHANGE_SUCCESSFUL ){
			return 1;
		}
		ChangeDisplaySettings(&devmode, CDS_FULLSCREEN);
	}



	HDC hDC;
	int pfdID;
	BOOL bResult;

	//sNZtH[}bg
	static PIXELFORMATDESCRIPTOR pfd = {
		sizeof (PIXELFORMATDESCRIPTOR),
		1,
		PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL,
		PFD_TYPE_RGBA,
		24,
		0, 0, 0,
		0, 0, 0,
		0, 0,
		0, 0, 0, 0, 0,
		32,
		0,
		0,
		PFD_MAIN_PLANE,
		0,
		0,
		0,
		0
	};

	//foCXReLXg擾
	hDC = GetDC(hWnd);

	//sNZtH[}bg擾
	pfdID = ChoosePixelFormat(hDC, &pfd);	
	if (pfdID == 0) { return 1; }

	//sNZtH[}bgw
	bResult = SetPixelFormat(hDC, pfdID, &pfd);
	if (bResult == FALSE) { return 1; }

	//ReLXgw
	hGLRC = wglCreateContext(hDC);
	if (hGLRC == NULL) { return 1; }

	//foCXReLXg
	ReleaseDC(hWnd, hDC);

	//VXetHgp
	//tHgFlr SVbN@TCYF18
	SystemFont = CreateFont(18, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, SHIFTJIS_CHARSET, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, FIXED_PITCH | FF_MODERN, "lr SVbN");


	//eNX`tHgp摜̃t@Cݒ
	strcpy(TextureFontFname, TextureFontFilename);

	//eNX`tHgp摜擾
	TextureFont = LoadTexture(TextureFontFname, true, false);


	float aspecth, prx, pry, r;
	aspecth = (float)SCREEN_WIDTH/SCREEN_HEIGHT;

	//HUD_myweapon [s, c, ]

	//HUD_A@ݎĂ镐\W
	prx = (float)M_PI/180*-39 * aspecth /2;
	pry = (float)M_PI/180*-55 /2;
	r = 7.5f;
	HUD_myweapon_x[0] = cos(pry)*r;
	HUD_myweapon_y[0] = sin(pry)*r;
	HUD_myweapon_z[0] = sin(prx)*r;

	//HUD_A@\̕\W
	prx = (float)M_PI/180*-52 * aspecth /2;
	pry = (float)M_PI/180*-60 /2;
	r = 16.0f;
	HUD_myweapon_x[1] = cos(pry)*r;
	HUD_myweapon_y[1] = sin(pry)*r;
	HUD_myweapon_z[1] = sin(prx)*r;


	//libjpeg
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_decompress(&cinfo);

	return 0;
}

//! @brief Zbg@n
//! iEBhEŏ̕A@Ȃǁj
//! @param WindowCtrl WindowControlNX̃|C^
//! @return F0@҂F1@sF2
int D3DGraphics::ResetD3D(WindowControl *WindowCtrl)
{
	hWnd = WindowCtrl->GethWnd();

	return 2;
}

//! @brief ft@Cǂݍށi.xj
//! @param filename t@C
//! @return FfFԍi0ȏj@sF-1
int D3DGraphics::LoadModel(char* filename)
{
	int id = -1;
	FILE *fp;
	char buf[256];
	char str[256];

	int vertexs = 0;
	MODELVDATA *vertex = NULL;
	int polygons = 0;
	int *index = NULL;
	int in_vertexs, in_polygons;
	MODELVDATA *old_vertex;
	int *old_index;

	char stroks[] = " ;,";		//؂镶

	//󂢂ĂFԍT
	for(int i=0; i<MAX_TEXTURE; i++){
		if( pmodel[i].useflag == false ){
			id = i;
			break;
		}
	}
	if( id == -1 ){ return -1; }

	//t@Cǂݍ
	fp = fopen(filename, "r");
	if( fp == NULL ){
		return -1;		//t@Cǂ߂Ȃ
	}

	//}WbNR[h擾
	fgets(buf, 256, fp);
	buf[ strlen("xof 0302txt") ] = 0x00;
	if( strcmp(buf, "xof 0302txt") != 0 ){
		fclose( fp );
		return -1;		//Xt@CłȂ
	}

	while( fgets(buf, 256, fp) != NULL ){
		strcpy(str, buf);
		str[ strlen("Mesh") ] = 0x00;
		if( strcmp(str, "Mesh") == 0 ){
			
			fgets(buf, 256, fp);
			in_vertexs = atoi(buf);

			if( vertexs == 0 ){
				//1ڂ̃bVf[^Ȃ΁Ä쐬邾B
				vertex = new MODELVDATA [in_vertexs];
			}
			else{
				//2ڂ̈ȍ~ȂÄmۂăRs[AÂ͍̈폜B
				old_vertex = vertex;
				vertex = new MODELVDATA [vertexs+in_vertexs];
				memcpy(vertex, old_vertex, sizeof(MODELVDATA)*vertexs);
				delete [] old_vertex;
			}

			for(int i=0; i<in_vertexs; i++){
				fgets(buf, 256, fp);
				vertex[i+vertexs].x = (float)atof( strtok(buf, stroks) ) * -1;
				vertex[i+vertexs].y = (float)atof( strtok(NULL, stroks) );
				vertex[i+vertexs].z = (float)atof( strtok(NULL, stroks) );
			}

			fgets(buf, 256, fp);

			fgets(buf, 256, fp);
			in_polygons = atoi(buf);

			if( polygons == 0 ){
				//1ڂ̃CfbNXf[^Ȃ΁Ä쐬邾B
				index = new int [in_polygons*5];
			}
			else{
				//2ڂ̈ȍ~ȂÄmۂăRs[AÂ͍̈폜B
				old_index = index;
				index = new int [(polygons+in_polygons)*5];
				memcpy(index, old_index, sizeof(int)*polygons*5);
				delete [] old_index;
			}

			for(int i=0; i<in_polygons; i++){
				fgets(buf, 256, fp);
				index[(i+polygons)*5] = atoi( strtok(buf, stroks) );
				for(int j=0; j<index[(i+polygons)*5]; j++){
					index[(i+polygons)*5 + j + 1] = atoi( strtok(NULL, stroks) ) + vertexs;
				}
			}

			while( fgets(buf, 256, fp) != NULL ){
				strcpy(str, buf);
				str[ strlen(" MeshTextureCoords") ] = 0x00;
				if( strcmp(str, " MeshTextureCoords") == 0 ){

					fgets(buf, 256, fp);
					if( atoi(buf) != in_vertexs ){ break; }

					for(int i=0; i<in_vertexs; i++){
						fgets(buf, 256, fp);
						vertex[i+vertexs].u = (float)atof( strtok(buf, stroks) );
						vertex[i+vertexs].v = (float)atof( strtok(NULL, stroks) );
					}

					break;
				}
			}

			vertexs += in_vertexs;
			polygons += in_polygons;
		}
	}

	//t@Cnh
	fclose( fp );

	float *VertexAry = new float [polygons*6*3];
	float *ColorAry = new float [polygons*6*4];
	float *TexCoordAry = new float [polygons*6*2];
	int vid;
	int cnt = 0;

	for(int i=0; i<polygons; i++){
		if( index[i*5] == 3 ){
			//Op`
			vid = index[i*5+1];
			VertexAry[0 + cnt*3] = vertex[vid].x;
			VertexAry[1 + cnt*3] = vertex[vid].y;
			VertexAry[2 + cnt*3] = vertex[vid].z;
			TexCoordAry[0 + cnt*2] = vertex[vid].u;
			TexCoordAry[1 + cnt*2] = vertex[vid].v;

			VertexAry[3 + cnt*3] = vertex[vid].x;
			VertexAry[4 + cnt*3] = vertex[vid].y;
			VertexAry[5 + cnt*3] = vertex[vid].z;
			TexCoordAry[2 + cnt*2] = vertex[vid].u;
			TexCoordAry[3 + cnt*2] = vertex[vid].v;

			vid = index[i*5+3];
			VertexAry[6 + cnt*3] = vertex[vid].x;
			VertexAry[7 + cnt*3] = vertex[vid].y;
			VertexAry[8 + cnt*3] = vertex[vid].z;
			TexCoordAry[4 + cnt*2] = vertex[vid].u;
			TexCoordAry[5 + cnt*2] = vertex[vid].v;

			vid = index[i*5+2];
			VertexAry[9 + cnt*3] = vertex[vid].x;
			VertexAry[10 + cnt*3] = vertex[vid].y;
			VertexAry[11 + cnt*3] = vertex[vid].z;
			TexCoordAry[6 + cnt*2] = vertex[vid].u;
			TexCoordAry[7 + cnt*2] = vertex[vid].v;

			VertexAry[12 + cnt*3] = vertex[vid].x;
			VertexAry[13 + cnt*3] = vertex[vid].y;
			VertexAry[14 + cnt*3] = vertex[vid].z;
			TexCoordAry[8 + cnt*2] = vertex[vid].u;
			TexCoordAry[9 + cnt*2] = vertex[vid].v;

			VertexAry[15 + cnt*3] = vertex[vid].x;
			VertexAry[16 + cnt*3] = vertex[vid].y;
			VertexAry[17 + cnt*3] = vertex[vid].z;
			TexCoordAry[10 + cnt*2] = vertex[vid].u;
			TexCoordAry[11 + cnt*2] = vertex[vid].v;

			cnt += 6;
		}
		else{
			//lp`
			vid = index[i*5+1];
			VertexAry[0 + cnt*3] = vertex[vid].x;
			VertexAry[1 + cnt*3] = vertex[vid].y;
			VertexAry[2 + cnt*3] = vertex[vid].z;
			TexCoordAry[0 + cnt*2] = vertex[vid].u;
			TexCoordAry[1 + cnt*2] = vertex[vid].v;

			VertexAry[3 + cnt*3] = vertex[vid].x;
			VertexAry[4 + cnt*3] = vertex[vid].y;
			VertexAry[5 + cnt*3] = vertex[vid].z;
			TexCoordAry[2 + cnt*2] = vertex[vid].u;
			TexCoordAry[3 + cnt*2] = vertex[vid].v;

			vid = index[i*5+4];
			VertexAry[6 + cnt*3] = vertex[vid].x;
			VertexAry[7 + cnt*3] = vertex[vid].y;
			VertexAry[8 + cnt*3] = vertex[vid].z;
			TexCoordAry[4 + cnt*2] = vertex[vid].u;
			TexCoordAry[5 + cnt*2] = vertex[vid].v;

			vid = index[i*5+2];
			VertexAry[9 + cnt*3] = vertex[vid].x;
			VertexAry[10 + cnt*3] = vertex[vid].y;
			VertexAry[11 + cnt*3] = vertex[vid].z;
			TexCoordAry[6 + cnt*2] = vertex[vid].u;
			TexCoordAry[7 + cnt*2] = vertex[vid].v;

			vid = index[i*5+3];
			VertexAry[12 + cnt*3] = vertex[vid].x;
			VertexAry[13 + cnt*3] = vertex[vid].y;
			VertexAry[14 + cnt*3] = vertex[vid].z;
			TexCoordAry[8 + cnt*2] = vertex[vid].u;
			TexCoordAry[9 + cnt*2] = vertex[vid].v;

			VertexAry[15 + cnt*3] = vertex[vid].x;
			VertexAry[16 + cnt*3] = vertex[vid].y;
			VertexAry[17 + cnt*3] = vertex[vid].z;
			TexCoordAry[10 + cnt*2] = vertex[vid].u;
			TexCoordAry[11 + cnt*2] = vertex[vid].v;

			cnt += 6;
		}
	}

	//Fzp
	ColorAry[0] = 1.0f;
	ColorAry[1] = 1.0f;
	ColorAry[2] = 1.0f;
	ColorAry[3] = 1.0f;
	for(int i=1; i<cnt; i++){
		memcpy(&(ColorAry[i*4]), ColorAry, sizeof(float)*4);
	}

	delete [] vertex;
	delete [] index;

	pmodel[id].useflag = true;
	pmodel[id].polygons = polygons;
	pmodel[id].VertexAry = VertexAry;
	pmodel[id].ColorAry = ColorAry;
	pmodel[id].TexCoordAry = TexCoordAry;
	return id;
}

//! @brief ft@C̒ԃf[^쐬i[tBOj
//! @param idA fA̔Fԍ
//! @param idB fB̔Fԍ
//! @return FVfFԍi0ȏj@sF-1
//! @attention fAƃfB́A_E|SECfbNXłKv܂B
//! @attention ꂼ̃ff[^Ȃ _قȂꍇAsɎs܂B
int D3DGraphics::MorphingModel(int idA, int idB)
{
	return idA;

	/*
	//f[^ׂ
	if( (idA < 0)||((MAX_MODEL -1) < idA) ){ return -1; }
	if( pmodel[idA].useflag == false ){ return; }
	if( (idB < 0)||((MAX_MODEL -1) < idB) ){ return -1; }
	if( pmodel[idB].useflag == false ){ return; }

	int id = -1;

	//󂢂ĂFԍT
	for(int i=0; i<MAX_TEXTURE; i++){
		if( pmodel[i].useflag == false ){
			id = i;
			break;
		}
	}
	if( id == -1 ){ return -1; }

	return -1;
	*/
}

//! @brief ft@C
//! @param id fFԍ
void D3DGraphics::CleanupModel(int id)
{
	if( (id < 0)||((MAX_MODEL -1) < id) ){ return; }
	if( pmodel[id].useflag == false ){ return; }

	delete pmodel[id].VertexAry;
	delete pmodel[id].ColorAry;
	delete pmodel[id].TexCoordAry;
	pmodel[id].useflag = false;
}

//! @brief eNX`ǂݍ
//! @param filename t@C
//! @param texturefont eNX`tHgtO
//! @param BlackTransparent 𓧉߂
//! @return FeNX`Fԍi0ȏj@sF-1
int D3DGraphics::LoadTexture(char* filename, bool texturefont, bool BlackTransparent)
{
	int id = -1;
	char filename2[MAX_PATH];
	int format = 0;

	//󂢂ĂFԍT
	for(int i=0; i<MAX_TEXTURE; i++){
		if( ptextures[i].useflag == false ){
			id = i;
			break;
		}
	}
	if( id == -1 ){ return -1; }

	//t@C֕ϊigqpj
	for(int i=0; i<strlen(filename); i++){
		filename2[i] = (char)tolower(filename[i]);
	}
	filename2[ strlen(filename) ] = NULL;

	//gqŃt@CtH[}bg𔻒
	for(int i=strlen(filename2)-1; i>0; i--){
		if( filename2[i] == '.' ){
			if( strcmp(&(filename2[i]), ".bmp") == 0 ){
				format = 1;
			}
			if( strcmp(&(filename2[i]), ".dds") == 0 ){
				format = 2;
			}
			if( strcmp(&(filename2[i]), ".jpeg") == 0 ){
				format = 3;
			}
			if( strcmp(&(filename2[i]), ".jpg") == 0 ){
				format = 3;
			}
			if( strcmp(&(filename2[i]), ".png") == 0 ){
				format = 4;
			}
			break;
		}
	}

	//ΉĂȂtH[}bg
	if( format == 0 ){ return -1; }

	if( format == 1 ){	// .bmp
		if( LoadBMPTexture(filename, BlackTransparent, &(ptextures[id])) == false ){
			return -1;
		}
	}
	if( format == 2 ){	// .dds
		if( LoadDDSTexture(filename, BlackTransparent, &(ptextures[id])) == false ){
			return -1;
		}
	}
	if( format == 3 ){	// .jpeg
		if( LoadJPEGTexture(filename, BlackTransparent, &(ptextures[id])) == false ){
			return -1;
		}
	}
	if( format == 4 ){	// .png
		if( LoadPNGTexture(filename, BlackTransparent, &(ptextures[id])) == false ){
			return -1;
		}
	}


	//eNX`L
	glEnable(GL_TEXTURE_2D);

	HDC hDC;
	hDC = GetDC(hWnd);
	wglMakeCurrent(hDC, hGLRC);
	glGenTextures(1 , &(textureobjname[id]));
	ReleaseDC(hWnd, hDC);

	glBindTexture(GL_TEXTURE_2D, textureobjname[id]);

	//OpenGLɃZbg
	int width = ptextures[id].width;
	int height = ptextures[id].height;
	unsigned char *data = ptextures[id].data;
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, width, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, data);

	//~bv}bvݒ
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR); 
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

	//Z
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	//eNX`
	glDisable(GL_TEXTURE_2D);

	return id;


	/*
	unsigned char *data = new unsigned char [16*4];

	data[0*4 + 0] = 255;	data[0*4 + 1] = 255;	data[0*4 + 2] = 255;	data[0*4 + 3] = 255;
	data[1*4 + 0] = 0;		data[1*4 + 1] = 0;		data[1*4 + 2] = 0;		data[1*4 + 3] = 255;
	data[2*4 + 0] = 255;	data[2*4 + 1] = 255;	data[2*4 + 2] = 255;	data[2*4 + 3] = 255;
	data[3*4 + 0] = 0;		data[3*4 + 1] = 0;		data[3*4 + 2] = 0;		data[3*4 + 3] = 255;
	data[4*4 + 0] = 255;	data[4*4 + 1] = 0;		data[4*4 + 2] = 0;		data[4*4 + 3] = 255;
	data[5*4 + 0] = 0;		data[5*4 + 1] = 255;	data[5*4 + 2] = 0;		data[5*4 + 3] = 255;
	data[6*4 + 0] = 0;		data[6*4 + 1] = 0;		data[6*4 + 2] = 255;	data[6*4 + 3] = 255;
	data[7*4 + 0] = 0;		data[7*4 + 1] = 0;		data[7*4 + 2] = 0;		data[7*4 + 3] = 255;
	data[8*4 + 0] = 128;	data[8*4 + 1] = 0;		data[8*4 + 2] = 0;		data[8*4 + 3] = 255;
	data[9*4 + 0] = 0;		data[9*4 + 1] = 128;	data[9*4 + 2] = 0;		data[9*4 + 3] = 255;
	data[10*4 + 0] = 0;		data[10*4 + 1] = 0;		data[10*4 + 2] = 128;	data[10*4 + 3] = 255;
	data[11*4 + 0] = 0;		data[11*4 + 1] = 0;		data[11*4 + 2] = 0;		data[11*4 + 3] = 255;
	data[12*4 + 0] = 255;	data[12*4 + 1] = 255;	data[12*4 + 2] = 0;		data[12*4 + 3] = 255;
	data[13*4 + 0] = 255;	data[13*4 + 1] = 0;		data[13*4 + 2] = 255;	data[13*4 + 3] = 255;
	data[14*4 + 0] = 0;		data[14*4 + 1] = 255;	data[14*4 + 2] = 255;	data[14*4 + 3] = 255;
	data[15*4 + 0] = 255;	data[15*4 + 1] = 255;	data[15*4 + 2] = 255;	data[15*4 + 3] = 255;

	ptextures[id].data = data;
	ptextures[id].width = 4;
	ptextures[id].height = 4;

	ptextures[id].useflag = true;

	return id;
	*/
}

//! @brief BMPt@Cǂݍ
//! @param filename t@C
//! @param BlackTransparent 𓧉߂
//! @param ptexture 󂯎TEXTUREDATA\̂̃|C^
//! @return Ftrue@sFfalse
//! @attention ʏ LoadTexture()֐ ĂтƁB
bool D3DGraphics::LoadBMPTexture(char* filename, bool BlackTransparent, TEXTUREDATA *ptexture)
{
	FILE *fp;
	unsigned char header[54];
	unsigned int dataPos;
	unsigned int width, height;
	unsigned int index;

	//t@Cǂݍ
	fp = fopen(filename, "rb");
	if( fp == NULL ){
		return false;		//t@Cǂ߂Ȃ
	}

	//wb_[ǂ
	fread(header, 1, 54, fp);

	if( (header[0x00] != 'B')||(header[0x01] != 'M') ){
		fclose(fp);
		return false;		//.bmpł͂Ȃ
	}

	// oCgz񂩂琮ǂݍ
	dataPos = *(int*)&(header[0x0E]) + 14;
	width = *(int*)&(header[0x12]);
	height = *(int*)&(header[0x16]);
	index = *(int*)&(header[0x1C]);

	//f[^̐擪܂ňړ
	fseek(fp, dataPos, SEEK_SET);

	unsigned char *data = new unsigned char [width*height*4];

	//esNZ8rbgȂA256Fpbg[h
	if( index == 8 ){
		unsigned char pixel;
		unsigned char *pallet = new unsigned char [256*4];
		fread(pallet, 1, 256*4, fp);

		for(int h=height-1; h>=0; h--){
			for(int w=0; w<width; w++){
				fread(&pixel, 1, 1, fp);

				data[(h*width+w)*4 + 0] = pallet[pixel*4 + 2];
				data[(h*width+w)*4 + 1] = pallet[pixel*4 + 1];
				data[(h*width+w)*4 + 2] = pallet[pixel*4 + 0];
				data[(h*width+w)*4 + 3] = 255;

				if( BlackTransparent == true ){
					//ȂΓ߂
					if( (data[(h*width+w)*4 + 0] == 0)&&(data[(h*width+w)*4 + 1] == 0)&&(data[(h*width+w)*4 + 2] == 0) ){
						data[(h*width+w)*4 + 3] = 0;
					}
				}
			}
		}

		delete []pallet;
	}

	//esNZ24rbgȂAtJ[
	if( index == 24 ){
		unsigned char pixel[3];
		for(int h=height-1; h>=0; h--){
			for(int w=0; w<width; w++){
				fread(&pixel, 1, 3, fp);

				data[(h*width+w)*4 + 0] = pixel[2];
				data[(h*width+w)*4 + 1] = pixel[1];
				data[(h*width+w)*4 + 2] = pixel[0];
				data[(h*width+w)*4 + 3] = 255;

				if( BlackTransparent == true ){
					//ȂΓ߂
					if( (data[(h*width+w)*4 + 0] == 0)&&(data[(h*width+w)*4 + 1] == 0)&&(data[(h*width+w)*4 + 2] == 0) ){
						data[(h*width+w)*4 + 3] = 0;
					}
				}
			}
		}
	}

	//t@Cnh
	fclose( fp );

	//\̂ɑ
	ptexture->data = data;
	ptexture->width = width;
	ptexture->height = height;

	ptexture->useflag = true;

	return true;
}

//! @brief DDSt@Cǂݍ
//! @param filename t@C
//! @param BlackTransparent 𓧉߂
//! @param ptexture 󂯎TEXTUREDATA\̂̃|C^
//! @return Ftrue@sFfalse
//! @attention ʏ LoadTexture()֐ ĂтƁB
bool D3DGraphics::LoadDDSTexture(char* filename, bool BlackTransparent, TEXTUREDATA *ptexture)
{
	FILE *fp;
	unsigned char header[124+4];
	unsigned int width, height;
	unsigned int index;

	//t@Cǂݍ
	fp = fopen(filename, "rb");
	if( fp == NULL ){
		return false;		//t@Cǂ߂Ȃ
	}

	//wb_[ǂ
	fread(header, 1, 124+4, fp);

	if( (header[0x00] != 'D')||(header[0x01] != 'D')||(header[0x02] != 'S')||(header[0x03] != ' ') ){
		fclose(fp);
		return false;		//.ddsł͂Ȃ
	}

	// oCgz񂩂琮ǂݍ
	width = *(int*)&(header[0x10]);
	height = *(int*)&(header[0x0C]);
	index = *(int*)&(header[0x58]);

	if( (index != 16)&&(index != 32) ){
		fclose(fp);
		return false;		//ΉĂȂtH[}bg
	}

	unsigned char *data = new unsigned char [width*height*4];

	for(int h=0; h<height; h++){
		for(int w=0; w<width; w++){
			unsigned char pixel[4];
			fread(&pixel, 1, index/8, fp);

			if( index == 16 ){		//esNZ16rbg
				data[(h*width+w)*4 + 0] = (pixel[1]<<4)&0xF0;
				data[(h*width+w)*4 + 1] = pixel[0]&0xF0;
				data[(h*width+w)*4 + 2] = (pixel[0]<<4)&0xF0;
				data[(h*width+w)*4 + 3] = pixel[1]&0xF0;
			}
			if( index == 32 ){		//esNZ32rbg
				data[(h*width+w)*4 + 0] = pixel[2];
				data[(h*width+w)*4 + 1] = pixel[1];
				data[(h*width+w)*4 + 2] = pixel[0];
				data[(h*width+w)*4 + 3] = pixel[3];
			}

			if( BlackTransparent == true ){
				//ȂΓ߂
				if( (data[(h*width+w)*4 + 0] == 0)&&(data[(h*width+w)*4 + 1] == 0)&&(data[(h*width+w)*4 + 2] == 0) ){
					data[(h*width+w)*4 + 3] = 0;
				}
			}
		}
	}

	//t@Cnh
	fclose( fp );

	//\̂ɑ
	ptexture->data = data;
	ptexture->width = width;
	ptexture->height = height;

	ptexture->useflag = true;

	return true;
}

//! @brief JPEGt@Cǂݍ
//! @param filename t@C
//! @param BlackTransparent 𓧉߂
//! @param ptexture 󂯎TEXTUREDATA\̂̃|C^
//! @return Ftrue@sFfalse
//! @attention ʏ LoadTexture()֐ ĂтƁB
bool D3DGraphics::LoadJPEGTexture(char* filename, bool BlackTransparent, TEXTUREDATA *ptexture)
{
	FILE *fp;
	JSAMPARRAY img;
	unsigned int width, height;

	//t@Cǂݍ
	fp = fopen(filename, "rb");
	if( fp == NULL ){
		return false;		//t@Cǂ߂Ȃ
	}
	jpeg_stdio_src(&cinfo, fp);

	//p[^̐ݒ
	jpeg_read_header(&cinfo, true);

	//WJJn
	jpeg_start_decompress(&cinfo);

	//̈m
	img = (JSAMPARRAY)new JSAMPROW [cinfo.output_height];
	for(int i=0; i<cinfo.output_height; i++){
		img[i] = (JSAMPROW)new JSAMPLE [cinfo.output_width * cinfo.out_color_components];
	}

	//WJ
	while( cinfo.output_scanline < cinfo.output_height ) {
		jpeg_read_scanlines(&cinfo, img + cinfo.output_scanline, cinfo.output_height - cinfo.output_scanline);
	}

	//WJI
	jpeg_finish_decompress(&cinfo);


	//t@Cnh
	fclose( fp );


	// oCgz񂩂琮ǂݍ
	width = (int)cinfo.output_width;
	height = (int)cinfo.output_height;

	unsigned char *data = new unsigned char [width*height*4];

	for(int h=0; h<height; h++){
		//1C擾
		JSAMPROW p = img[h];

		for(int w=0; w<width; w++){
			data[(h*width+w)*4 + 0] = p[w*3 + 0];
			data[(h*width+w)*4 + 1] = p[w*3 + 1];
			data[(h*width+w)*4 + 2] = p[w*3 + 2];
			data[(h*width+w)*4 + 3] = 255;

			if( BlackTransparent == true ){
				//ȂΓ߂
				if( (data[(h*width+w)*4 + 0] == 0)&&(data[(h*width+w)*4 + 1] == 0)&&(data[(h*width+w)*4 + 2] == 0) ){
					data[(h*width+w)*4 + 3] = 0;
				}
			}

		}
	}

	//̈
	for(int i=0; i<cinfo.output_height; i++){
		delete [] img[i];
	}
	delete [] img;

	//\̂ɑ
	ptexture->data = data;
	ptexture->width = width;
	ptexture->height = height;

	ptexture->useflag = true;

	return true;
}

//! @brief PNGt@Cǂݍ
//! @param filename t@C
//! @param BlackTransparent 𓧉߂
//! @param ptexture 󂯎TEXTUREDATA\̂̃|C^
//! @return Ftrue@sFfalse
//! @attention ʏ LoadTexture()֐ ĂтƁB
bool D3DGraphics::LoadPNGTexture(char* filename, bool BlackTransparent, TEXTUREDATA *ptexture)
{
	FILE *fp;
	png_byte sig[4];
	png_structp pPng;
    png_infop pInfo;
	unsigned int width, height;

	//t@Cǂݍ
	fp = fopen(filename, "rb");
	if( fp == NULL ){
		return false;		//t@Cǂ߂Ȃ
	}

	//PNGt@C
	fread(sig, 4, 1, fp);
	if( png_sig_cmp(sig, 0, 4) != 0 ){
		fclose(fp);
		return false;	//.pngł͂Ȃ
	}

	//\̂
	pPng = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
    pInfo = png_create_info_struct(pPng);

	//擾
	png_init_io(pPng, fp);
	png_set_sig_bytes(pPng, 4);
	png_read_info(pPng, pInfo);

	//eNX`̑傫擾
	width = png_get_image_width(pPng, pInfo);
	height =  png_get_image_height(pPng, pInfo);

	//C^[X
	if( png_get_interlace_type(pPng, pInfo) != PNG_INTERLACE_NONE ){
		png_destroy_read_struct(&pPng, &pInfo, (png_infopp)NULL);
		fclose(fp);
		return false;	//C^[Xɂ͑ΉȂB
	}

	//At@`l
	png_set_add_alpha(pPng, 0xff, PNG_FILLER_AFTER);

	// tRNS`N΁AAt@`lɕϊ
	if (png_get_valid(pPng, pInfo, PNG_INFO_tRNS)) {
		png_set_tRNS_to_alpha(pPng);
	}

	unsigned char *data = new unsigned char [width*height*4];

	//1C̍Ɨ̈m
	png_bytep buf = new png_byte[width*4];

	for(int h=0; h<height; h++){
		//1C擾
		png_read_row(pPng,buf,NULL);

		for(int w=0; w<width; w++){
			data[(h*width+w)*4 + 0] = buf[w*4 + 0];
			data[(h*width+w)*4 + 1] = buf[w*4 + 1];
			data[(h*width+w)*4 + 2] = buf[w*4 + 2];
			data[(h*width+w)*4 + 3] = buf[w*4 + 3];

			if( BlackTransparent == true ){
				//ȂΓ߂
				if( (data[(h*width+w)*4 + 0] == 0)&&(data[(h*width+w)*4 + 1] == 0)&&(data[(h*width+w)*4 + 2] == 0) ){
					data[(h*width+w)*4 + 3] = 0;
				}
			}
		}
	}

	//1C̍Ɨ̈
	delete [] buf;

	//
	png_read_end(pPng, NULL);
	png_destroy_read_struct(&pPng, &pInfo, (png_infopp)NULL);

	//t@Cnh
	fclose( fp );

	//\̂ɑ
	ptexture->data = data;
	ptexture->width = width;
	ptexture->height = height;

	ptexture->useflag = true;

	return true;
}

//! @brief eNX`̃TCY擾
//! @param id eNX`Fԍ
//! @param width 󂯎|C^
//! @param height 󂯎|C^
//! @return F0@sF1
//! @attention T[tFCX̃TCY擾܂BGPUɃ[hꂽTCYłAeNX`ijƈقȂꍇ܂B
int D3DGraphics::GetTextureSize(int id, int *width, int *height)
{
	//ȔFԍw肳ĂAԂB
	if( id == -1 ){ return 1; }
	if( ptextures[id].useflag == false ){ return 1; }

	*width = ptextures[id].width;
	*height = ptextures[id].height;

	return 0;
}

//! @brief eNX`w
//! @param TextureID eNX`Fԍ
void D3DGraphics::SetTexture(int TextureID)
{
	if( now_textureid == TextureID ){
		return;
	}

	//OpenGLɃZbg
	glBindTexture(GL_TEXTURE_2D, textureobjname[TextureID]);

	now_textureid = TextureID;
}

//! @brief eNX`
//! @param id eNX`Fԍ
void D3DGraphics::CleanupTexture(int id)
{
	if( (id < 0)||((MAX_TEXTURE -1) < id) ){ return; }
	if( ptextures[id].useflag == false ){ return; }

	delete ptextures[id].data;
	glDeleteTextures(1 , &(textureobjname[id]));
	ptextures[id].useflag = false;
}

//! @brief SĂ̕`揈Jn
//! @return F0@sF1
//! @attention `揈̍ŏɌĂяoKv܂B
int D3DGraphics::StartRender()
{
	HDC hDC;

	hDC = BeginPaint(hWnd, &Paint_ps);

	//ReLXgw
	wglMakeCurrent(hDC, hGLRC);

	//
	glClearColor(0.0f, 0.0f, 0.0f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glEnable(GL_DEPTH_TEST);
	glEnable(GL_CULL_FACE);

	//iߗLj
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	//At@eXg
	glAlphaFunc(GL_GREATER, 0.0f);
	glEnable(GL_ALPHA_TEST);

	return 0;
}

//! @brief SĂ̕`揈I
//! @return Ffalse@sFtrue
//! @attention `揈̍ŌɌĂяoKv܂B
bool D3DGraphics::EndRender()
{
	glFlush();

	wglMakeCurrent(NULL, NULL);

	EndPaint(hWnd, &Paint_ps);

	return false;
}

//! @brief Zobt@Zbg
void D3DGraphics::ResetZbuffer()
{
	glClear(GL_DEPTH_BUFFER_BIT);
}

//! @brief [hԂ_i0,0,0jɖ߂@Ȃ
void D3DGraphics::ResetWorldTransform()
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(viewangle*(180.0f/(float)M_PI), (float)width/height, CLIPPINGPLANE_NEAR, CLIPPINGPLANE_FAR);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	gluLookAt(camera_x*-1, camera_y, camera_z, camera_x*-1 + cos(camera_rx*-1 + (float)M_PI)*cos(camera_ry), camera_y + sin(camera_ry), camera_z + sin(camera_rx*-1 + (float)M_PI)*cos(camera_ry), 0.0f, 1.0f, 0.0f);
}

//! @brief [hԂ̍WEpxEg嗦ݒ
//! @param x XW
//! @param y YW
//! @param z ZW
//! @param rx px
//! @param ry cpx
//! @param size g嗦
void D3DGraphics::SetWorldTransform(float x, float y, float z, float rx, float ry, float size)
{
	SetWorldTransform(x, y, z, rx, ry, 0.0f, size);
}

//! @brief [hԂ̍WEpxEg嗦ݒ
//! @param x XW
//! @param y YW
//! @param z ZW
//! @param rx px
//! @param ry1 cpx
//! @param ry2 cpx
//! @param size g嗦
void D3DGraphics::SetWorldTransform(float x, float y, float z, float rx, float ry1, float ry2, float size)
{
	ResetWorldTransform();

	glMatrixMode(GL_MODELVIEW);

	glTranslatef(x*-1, y, z);
	glRotatef(rx*-1*(180.0f/(float)M_PI), 0.0f, 1.0f, 0.0f);
	glRotatef(ry1*(180.0f/(float)M_PI), 1.0f, 0.0f, 0.0f);
	glRotatef(ry2*-1*(180.0f/(float)M_PI), 0.0f, 0.0f, 1.0f);
	glScalef(size, size, size);
}

//! @brief [hԂ̍WEpxEg嗦ݒiGtFNgpj
//! @param x XW
//! @param y YW
//! @param z ZW
//! @param rx px
//! @param ry cpx
//! @param rt ]px
//! @param size g嗦
void D3DGraphics::SetWorldTransformEffect(float x, float y, float z, float rx, float ry, float rt, float size)
{
	ResetWorldTransform();

	glMatrixMode(GL_MODELVIEW);

	glTranslatef(x*-1, y, z);
	glRotatef(rx*-1*(180.0f/(float)M_PI), 0.0f, 1.0f, 0.0f);
	glRotatef(ry*-1*(180.0f/(float)M_PI), 0.0f, 0.0f, 1.0f);
	glRotatef(rt*(180.0f/(float)M_PI), 1.0f, 0.0f, 0.0f);
	glScalef(size, size, size);
}

//! @brief [hԂlꏊɐݒ
//! @param x XW
//! @param y YW
//! @param z ZW
//! @param mx 茳_ɂ fXW
//! @param my 茳_ɂ fYW
//! @param mz 茳_ɂ fZW
//! @param rx px
//! @param ry cpx
//! @param size g嗦
void D3DGraphics::SetWorldTransformHumanWeapon(float x, float y, float z, float mx, float my, float mz, float rx, float ry, float size)
{
	ResetWorldTransform();

	glMatrixMode(GL_MODELVIEW);

	glTranslatef(x*-1, y, z);
	glRotatef(rx*-1*(180.0f/(float)M_PI), 0.0f, 1.0f, 0.0f);
	glRotatef(ry*(180.0f/(float)M_PI), 1.0f, 0.0f, 0.0f);
	glTranslatef(mx*-1, my, mz);
	glScalef(size, size, size);
}

//! @brief [hԂĂ镐\ꏊɐݒ
//! @param rotation ]
//! @param camera_x JXW
//! @param camera_y JYW
//! @param camera_z JZW
//! @param camera_rx J̉px
//! @param camera_ry J̏cpx
//! @param rx ̂̏cpx
//! @param size \TCY
//! @note rotationEE@trueFݎĂ镐łB@falseF\̕łBirx ͖܂j
//! @todo ʒuTCY̔
void D3DGraphics::SetWorldTransformPlayerWeapon(bool rotation, float camera_x, float camera_y, float camera_z, float camera_rx, float camera_ry, float rx, float size)
{
	size = size * 0.3f;

	ResetWorldTransform();

	glMatrixMode(GL_MODELVIEW);

	glTranslatef(camera_x*-1, camera_y, camera_z);
	glRotatef(camera_rx*(180.0f/(float)M_PI), 0.0f, 1.0f, 0.0f);
	glRotatef(camera_ry*-1*(180.0f/(float)M_PI), 0.0f, 0.0f, 1.0f);

	if( rotation == true ){
		glTranslatef(HUD_myweapon_x[0]*-1, HUD_myweapon_y[0], HUD_myweapon_z[0]);
		glRotatef(rx*-1*(180.0f/(float)M_PI), 0.0f, 1.0f, 0.0f);
	}
	else{
		glTranslatef(HUD_myweapon_x[1]*-1, HUD_myweapon_y[1], HUD_myweapon_z[1]);
		glRotatef(180, 0.0f, 1.0f, 0.0f);
	}
	glScalef(size, size, size);
}

//! @brief [hԂ̍W擾
//! @param *x x󂯎|C^
//! @param *y y󂯎|C^
//! @param *z z󂯎|C^
void D3DGraphics::GetWorldTransformPos(float *x, float *y, float *z)
{
	GLfloat model[16];
	glMatrixMode(GL_MODELVIEW);
	glGetFloatv(GL_MODELVIEW_MATRIX, model);
	*x = model[12];
	*y = model[13];
	*z = model[14];
}

//! @brief tHOݒ
//! @param skynumber ̔ԍ
void D3DGraphics::SetFog(int skynumber)
{
	float fogColor[4];

	//̔ԍɂF
	switch(skynumber){
		case 1:		fogColor[0] = 0.25f;	fogColor[1] = 0.25f+0.0625;	fogColor[2] = 0.25f;	fogColor[3] = 1.0f;		break;
		case 2:		fogColor[0] = 0.0625;	fogColor[1] = 0.0625;		fogColor[2] = 0.0625;	fogColor[3] = 1.0f;		break;
		case 3:		fogColor[0] = 0.0f;		fogColor[1] = 0.0625;		fogColor[2] = 0.125;	fogColor[3] = 1.0f;		break;
		case 4:		fogColor[0] = 0.125;	fogColor[1] = 0.0625;		fogColor[2] = 0.0625;	fogColor[3] = 1.0f;		break;
		case 5:		fogColor[0] = 0.25f;	fogColor[1] = 0.125;		fogColor[2] = 0.125;	fogColor[3] = 1.0f;		break;
		default:	fogColor[0] = 0.0f;		fogColor[1] = 0.0f;			fogColor[2] = 0.0f;		fogColor[3] = 1.0f;		break;
	}

	float fog_st = 100;
	float fog_end = 800;
	glFogi(GL_FOG_MODE, GL_LINEAR);
	glFogfv(GL_FOG_COLOR, fogColor);
	glHint(GL_FOG_HINT, GL_NICEST);
	glFogf(GL_FOG_START, fog_st);
	glFogf(GL_FOG_END, fog_end);

	glEnable(GL_FOG);
}

//! @brief Ji_jݒ
//! @param in_camera_x JXW
//! @param in_camera_y JYW
//! @param in_camera_z JZW
//! @param in_camera_rx J̉px
//! @param in_camera_ry J̏cpx
//! @param in_viewangle p
void D3DGraphics::SetCamera(float in_camera_x, float in_camera_y, float in_camera_z, float in_camera_rx, float in_camera_ry, float in_viewangle)
{
	glViewport(0, 0, width, height);
	
	camera_x = in_camera_x;
	camera_y = in_camera_y;
	camera_z = in_camera_z;
	camera_rx = in_camera_rx;
	camera_ry = in_camera_ry;
	viewangle = in_viewangle;

	ResetWorldTransform();
}

//! @brief }bvf[^荞
//! @param in_blockdata ubNf[^
//! @param directory ubNf[^݂fBNg
void D3DGraphics::LoadMapdata(BlockDataInterface* in_blockdata, char *directory)
{
	//ubNf[^w肳ĂȂ΁AȂB
	if( in_blockdata == NULL ){ return; }

	char fname[MAX_PATH];
	char fnamefull[MAX_PATH];

	//NXݒ
	blockdata = in_blockdata;

	//ubN擾
	bs = blockdata->GetTotaldatas();

	//eNX`ǂݍ
	for(int i=0; i<TOTAL_BLOCKTEXTURE; i++){
		//eNX`擾
		blockdata->GetTexture(fname, i);

		if( strcmp(fname, "") == 0 ){	//w肳ĂȂ΁AȂ
			mapTextureID[i] = -1;
		}
		else{
			//ufBNg{t@Cv𐶐Aǂݍ
			strcpy(fnamefull, directory);
			strcat(fnamefull, fname);
			mapTextureID[i] = LoadTexture(fnamefull, false, false);
		}
	}
}

//! @brief }bvf[^`
//! @param wireframe C[t[\
void D3DGraphics::DrawMapdata(bool wireframe)
{
	//ubNf[^ǂݍ܂ĂȂ΁AȂB
	if( blockdata == NULL ){ return; }

	struct blockdata data;
	int textureID;
	int vID[4];
	int uvID[4];
	float *VertexAry = new float [bs*6 * 6*3];
	float *ColorAry = new float [bs*6 * 6*4];
	float *TexCoordAry = new float [bs*6 * 6*2];

	if( wireframe == true ){
		//C[t[\
		for(int i=0; i<bs; i++){
			blockdata->Getdata(&data, i);
			Drawline(data.x[0], data.y[0], data.z[0], data.x[1], data.y[1], data.z[1]);
			Drawline(data.x[1], data.y[1], data.z[1], data.x[2], data.y[2], data.z[2]);
			Drawline(data.x[2], data.y[2], data.z[2], data.x[3], data.y[3], data.z[3]);
			Drawline(data.x[3], data.y[3], data.z[3], data.x[0], data.y[0], data.z[0]);
			Drawline(data.x[4], data.y[4], data.z[4], data.x[5], data.y[5], data.z[5]);
			Drawline(data.x[5], data.y[5], data.z[5], data.x[6], data.y[6], data.z[6]);
			Drawline(data.x[6], data.y[6], data.z[6], data.x[7], data.y[7], data.z[7]);
			Drawline(data.x[7], data.y[7], data.z[7], data.x[4], data.y[4], data.z[4]);
			Drawline(data.x[0], data.y[0], data.z[0], data.x[4], data.y[4], data.z[4]);
			Drawline(data.x[1], data.y[1], data.z[1], data.x[5], data.y[5], data.z[5]);
			Drawline(data.x[2], data.y[2], data.z[2], data.x[6], data.y[6], data.z[6]);
			Drawline(data.x[3], data.y[3], data.z[3], data.x[7], data.y[7], data.z[7]);
		}
		return;
	}

	//zL
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);

	for(textureID=0; textureID<TOTAL_BLOCKTEXTURE; textureID++){
		int cnt = 0;

		//eNX`ɓǂݍ߂ĂȂΐݒ
		if( mapTextureID[textureID] == -1 ){
			//eNX`
			glDisable(GL_TEXTURE_2D);
		}
		else if( ptextures[ mapTextureID[textureID] ].useflag == false ){
			//eNX`
			glDisable(GL_TEXTURE_2D);
		}
		else{
			//eNX`L
			glEnable(GL_TEXTURE_2D);

			//eNX`Zbg
			SetTexture(mapTextureID[textureID]);
		}

		for(int i=0; i<bs; i++){
			//f[^擾
			blockdata->Getdata(&data, i);

			for(int j=0; j<6; j++){
				//eNX`Fԍ擾
				int ID = data.material[j].textureID;

				if( textureID == ID ){
					//ʂ̒_f[^̊֘At擾
					blockdataface(j, &vID[0], &uvID[0]);

					//_zp
					VertexAry[0 + cnt*18] = data.x[ vID[1] ]*-1;		VertexAry[1 + cnt*18] = data.y[ vID[1] ];	VertexAry[2 + cnt*18] = data.z[ vID[1] ];
					VertexAry[3 + cnt*18] = data.x[ vID[1] ]*-1;		VertexAry[4 + cnt*18] = data.y[ vID[1] ];	VertexAry[5 + cnt*18] = data.z[ vID[1] ];
					VertexAry[6 + cnt*18] = data.x[ vID[0] ]*-1;		VertexAry[7 + cnt*18] = data.y[ vID[0] ];	VertexAry[8 + cnt*18] = data.z[ vID[0] ];
					VertexAry[9 + cnt*18] = data.x[ vID[2] ]*-1;		VertexAry[10 + cnt*18] = data.y[ vID[2] ];	VertexAry[11 + cnt*18] = data.z[ vID[2] ];
					VertexAry[12 + cnt*18] = data.x[ vID[3] ]*-1;		VertexAry[13 + cnt*18] = data.y[ vID[3] ];	VertexAry[14 + cnt*18] = data.z[ vID[3] ];
					VertexAry[15 + cnt*18] = data.x[ vID[3] ]*-1;		VertexAry[16 + cnt*18] = data.y[ vID[3] ];	VertexAry[17 + cnt*18] = data.z[ vID[3] ];

					//Fzp
					ColorAry[0 + cnt*24] = data.material[j].shadow;
					ColorAry[1 + cnt*24] = data.material[j].shadow;
					ColorAry[2 + cnt*24] = data.material[j].shadow;
					ColorAry[3 + cnt*24] = 1.0f;
					for(int i=1; i<6; i++){
						memcpy(&(ColorAry[i*4 + cnt*24]), ColorAry, sizeof(float)*4);
					}

					//UVWzp
					TexCoordAry[0 + cnt*12] = data.material[j].u[ uvID[1] ];	TexCoordAry[1 + cnt*12] = data.material[j].v[ uvID[1] ];
					TexCoordAry[2 + cnt*12] = data.material[j].u[ uvID[1] ];	TexCoordAry[3 + cnt*12] = data.material[j].v[ uvID[1] ];
					TexCoordAry[4 + cnt*12] = data.material[j].u[ uvID[0] ];	TexCoordAry[5 + cnt*12] = data.material[j].v[ uvID[0] ];
					TexCoordAry[6 + cnt*12] = data.material[j].u[ uvID[2] ];	TexCoordAry[7 + cnt*12] = data.material[j].v[ uvID[2] ];
					TexCoordAry[8 + cnt*12] = data.material[j].u[ uvID[3] ];	TexCoordAry[9 + cnt*12] = data.material[j].v[ uvID[3] ];
					TexCoordAry[10 + cnt*12] = data.material[j].u[ uvID[3] ];	TexCoordAry[11 + cnt*12] = data.material[j].v[ uvID[3] ];

					cnt += 1;
				}
			}
		}

		//`
		glVertexPointer(3, GL_FLOAT, 0, VertexAry);
		glColorPointer(4, GL_FLOAT, 0, ColorAry);
		glTexCoordPointer(2, GL_FLOAT, 0, TexCoordAry);
		glDrawArrays(GL_TRIANGLE_STRIP, 1, 6*cnt-2);
	}

	//z񖳌
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

	delete [] VertexAry;
	delete [] ColorAry;
	delete [] TexCoordAry;
}

//! @brief }bveNX`擾
//! @param id eNX`ԍ
//! @return eNX`FԍisF-1j
int D3DGraphics::GetMapTextureID(int id)
{
	if( (id < 0)||((TOTAL_BLOCKTEXTURE -1) < id ) ){ return -1; }
	return mapTextureID[id];
}

//! @brief }bvf[^
void D3DGraphics::CleanupMapdata()
{
	//eNX`J
	for(int i=0; i<TOTAL_BLOCKTEXTURE; i++){
		CleanupTexture(mapTextureID[i]);
	}

	bs = 0;

	blockdata = NULL;
}

//! @brief ft@C`
//! @param id_model fFԍ
//! @param id_texture eNX`Fԍ
void D3DGraphics::RenderModel(int id_model, int id_texture)
{
	//Ȉݒ肳ĂΎs
	if( id_model == -1 ){ return; }
	if( pmodel[id_model].useflag == false ){ return; }

	//eNX`ɓǂݍ߂ĂȂΐݒ
	if( id_texture == -1 ){
		//eNX`
		glDisable(GL_TEXTURE_2D);
	}
	else if( ptextures[id_texture].useflag == false ){
		//eNX`
		glDisable(GL_TEXTURE_2D);
	}
	else{
		//eNX`L
		glEnable(GL_TEXTURE_2D);

		//eNX`Zbg
		SetTexture(id_texture);
	}

	//zL
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);

	//`
	glVertexPointer(3, GL_FLOAT, 0, pmodel[id_model].VertexAry);
	glColorPointer(4, GL_FLOAT, 0, pmodel[id_model].ColorAry);
	glTexCoordPointer(2, GL_FLOAT, 0, pmodel[id_model].TexCoordAry);
	glDrawArrays(GL_TRIANGLE_STRIP, 1, pmodel[id_model].polygons*6-2);

	//z񖳌
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

	/*
	Drawline(1.0f, 0.0f, 0.0f, -1.0f, 0.0f, 0.0f);
	Drawline(0.0f, 1.0f, 0.0f, 0.0f, -1.0f, 0.0f);
	Drawline(0.0f, 0.0f, 1.0f, 0.0f, 0.0f, -1.0f);
	*/
}

//! @brief `
//! @param id_texture eNX`Fԍ
//! @param alpha x@i0.0`1.0@0.0FSj
void D3DGraphics::RenderBoard(int id_texture, float alpha)
{
	//eNX`ݒ肳ĂȂ΁AȂB
	if( id_texture == -1 ){ return; }
	if( ptextures[id_texture].useflag == false ){ return; }

	float VertexAry[4*3];
	float ColorAry[4*4];
	float TexCoordAry[4*2];

	//eNX`L
	glEnable(GL_TEXTURE_2D);

	//eNX`Zbg
	SetTexture(id_texture);

	//_zp
	VertexAry[0] = 0.0f;	VertexAry[1] = 0.5f;	VertexAry[2] = 0.5f;
	VertexAry[3] = 0.0f;	VertexAry[4] = -0.5f;	VertexAry[5] = 0.5f;
	VertexAry[6] = 0.0f;	VertexAry[7] = 0.5f;	VertexAry[8] = -0.5f;
	VertexAry[9] = 0.0f;	VertexAry[10] = -0.5f;	VertexAry[11] = -0.5f;

	//Fzp
	ColorAry[0] = 1.0f;
	ColorAry[1] = 1.0f;
	ColorAry[2] = 1.0f;
	ColorAry[3] = alpha;
	for(int i=1; i<4; i++){
		memcpy(&(ColorAry[i*4]), ColorAry, sizeof(float)*4);
	}

	//UVWzp
	TexCoordAry[0] = 1.0f;	TexCoordAry[1] = 0.0f;
	TexCoordAry[2] = 1.0f;	TexCoordAry[3] = 1.0f;
	TexCoordAry[4] = 0.0f;	TexCoordAry[5] = 0.0f;
	TexCoordAry[6] = 0.0f;	TexCoordAry[7] = 1.0f;

	//zL
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);

	//`
	glVertexPointer(3, GL_FLOAT, 0, VertexAry);
	glColorPointer(4, GL_FLOAT, 0, ColorAry);
	glTexCoordPointer(2, GL_FLOAT, 0, TexCoordAry);
	glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

	//z񖳌
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
}

//! @brief ʂ̖邳ݒ
//! @param Width 
//! @param Height 
//! @param Brightness ʂ̖邳@i0 ŕsρA1 ȏŖ邳̓xj
void D3DGraphics::ScreenBrightness(int Width, int Height, int Brightness)
{
	//邳sςȂ珈Ȃiyʉj
	if( Brightness == 0 ){ return; }

	//xݒ肵A`
	float alpha = 0.02f * Brightness;
	Draw2DBox(0, 0, Width, Height, GetColorCode(1.0f,1.0f,1.0f,alpha));
}

//! @brief yfobNpzS`
void D3DGraphics::Centerline()
{
	ResetWorldTransform();
	Drawline(100.0f, 0.0f, 0.0f, -100.0f, 0.0f, 0.0f);
	Drawline(0.0f, 100.0f, 0.0f, 0.0f, -100.0f, 0.0f);
	Drawline(0.0f, 0.0f, 100.0f, 0.0f, 0.0f, -100.0f);
}

//! @brief yfobNpzΐ`
void D3DGraphics::Drawline(float x1, float y1, float z1, float x2, float y2, float z2)
{
	float VertexAry[2*3];
	unsigned char ColorAry[2*4];

	//eNX`
	glDisable(GL_TEXTURE_2D);

	//_zp
	VertexAry[0] = (float)x1*-1;	VertexAry[1] = (float)y1;	VertexAry[2] = (float)z1;
	VertexAry[3] = (float)x2*-1;	VertexAry[4] = (float)y2;	VertexAry[5] = (float)z2;

	//Fzp
	ColorAry[0] = 0;
	ColorAry[1] = 255;
	ColorAry[2] = 0;
	ColorAry[3] = 255;
	memcpy(&(ColorAry[4]), ColorAry, sizeof(unsigned char)*4);

	//zL
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);

	//`
	glVertexPointer(3, GL_FLOAT, 0, VertexAry);
	glColorPointer(4, GL_UNSIGNED_BYTE, 0, ColorAry);
	glDrawArrays(GL_LINE_STRIP, 0, 2);

	//z񖳌
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
}

//! @brief `iVXetHggpj
//! @param x xW
//! @param y yW
//! @param str @isR[hFj
//! @param color F
//! @warning <b>`͔ɒᑬłB</b>ʓŉxĂяoƃptH[}Xɉe܂B
//! @warningusR[hpxɕ`悷vu{ꂪKvȂ̓eNX`tHgpvȂǂ̑ΉuĂB
//! @attention tHg̎ނTCY͌ŒłB@dɏd˂ė̊oȂƌɂȂ܂B
//! @todo dɏd˂ƁA㉺֌WȂB
//! @todo 1ڂꍇB
void D3DGraphics::Draw2DMSFontText(int x, int y, char *str, int color)
{
	int len = strlen(str);
	WCHAR *ustr;

	Start2DRender();

	//eNX`
	glDisable(GL_TEXTURE_2D);

	//Unicode֕ϊ
	ustr = new WCHAR [len+1];
	MultiByteToWideChar(CP_ACP,	0, str, -1, ustr, len + 1);

	//VȕȂA\[X蒼
	if( lstrcmpW(ustr, now_SystemFontUStr) != 0 ){
		GLuint listIdx;
		HDC hDC;

		//Âf[^폜
		glDeleteLists(SystemFontListIdx, SystemFontListIdxSize);
		delete [] now_SystemFontUStr;

		//foCXReLXgݒ
		hDC = GetDC(hWnd);
		wglMakeCurrent(hDC, hGLRC);
		SelectObject(hDC, SystemFont);

		//fBXvCXg쐬
		listIdx = glGenLists(len);
		for(int i=0; i<lstrlenW(ustr); i++){
			wglUseFontBitmapsW(hDC, ustr[i], 1, listIdx+i);
		}

		//foCXReLXgp
		ReleaseDC(hWnd, hDC);

		//ݒL^
		now_SystemFontUStr = new WCHAR [len+1];
		lstrcpyW(now_SystemFontUStr, ustr);
		SystemFontListIdx = listIdx;
		SystemFontListIdxSize = len;
	}

	//WƐFݒ
	glBitmap(0, 0, 0, 0, 10, 0, NULL);
	glRasterPos2i(x, y);
	glColor4ub((color>>24)&0xFF, (color>>16)&0xFF, (color>>8)&0xFF, color&0xFF);

	for(int i=0; i<lstrlenW(ustr); i++){
		if( ustr[i] == '\n' ){
			//s
			y += 19;
			glRasterPos2i(x, y);
		}
		else{
			//fBXvCXg`
			glCallList(SystemFontListIdx + i);
		}
	}

	//Unicode̔p
	delete [] ustr;

	End2DRender();
}

//! @brief 𒆉ŕ`iVXetHggpj
//! @param x xW
//! @param y yW
//! @param w ̑傫
//! @param h c̑傫
//! @param str @isR[hFj
//! @param color F
//! @warning <b>ɂȂ܂B</b>
void D3DGraphics::Draw2DMSFontTextCenter(int x, int y, int w, int h, char *str, int color)
{
	Draw2DMSFontText(x, y, str, color);
}

//! @brief 2D`pݒ
void D3DGraphics::Start2DRender()
{
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	glOrtho(0, width, height, 0, -1, 1);
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	gluLookAt(0, 0, 1, 0, 0, 0, 0, 1, 0);

	glDisable(GL_DEPTH_TEST);
	glDisable(GL_CULL_FACE);
}

//! @brief `ieNX`tHggpj
//! @param x xW
//! @param y yW
//! @param str @isR[hF<b>s</b>j
//! @param color F
//! @param fontwidth ꕶ̕
//! @param fontheight ꕶ̍
//! @attention dɏd˂ė̊oȂƌɂȂ܂B
void D3DGraphics::Draw2DTextureFontText(int x, int y, char *str, int color, int fontwidth, int fontheight)
{
	//eNX`tHg̎擾ɎsĂ΁AȂ
	if( TextureFont == -1 ){ return; }

	int strlens = (int)strlen(str);

	float *VertexAry = new float [strlens*6*2];
	unsigned char *ColorAry = new unsigned char [strlens*6*4];
	float *TexCoordAry = new float [strlens*6*2];

	//2D`pݒKp
	Start2DRender();

	int w;
	float font_u, font_v;
	float t_u, t_v;

	//1UVWvZ
	font_u = 1.0f / 16;
	font_v = 1.0f / 16;

	//eNX`L
	glEnable(GL_TEXTURE_2D);

	//eNX`Zbg
	SetTexture(TextureFont);

	//zL
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);

	//Fzp
	ColorAry[0] = (color>>24)&0xFF;
	ColorAry[1] = (color>>16)&0xFF;
	ColorAry[2] = (color>>8)&0xFF;
	ColorAry[3] = color&0xFF;
	for(int i=1; i<strlens*6; i++){
		memcpy(&(ColorAry[i*4]), ColorAry, sizeof(unsigned char)*4);
	}

	// ^ꂽ[v
	for(int i=0; i<strlens; i++){
		//UVWvZ
		w = str[i];
		if( w < 0 ){ w += 256; }
		t_u = (w % 16) * font_u;
		t_v = (w / 16) * font_v;

		VertexAry[0 + i*12] = (float)x + i*fontwidth;			VertexAry[1 + i*12] = (float)y;
		VertexAry[2 + i*12] = (float)x + i*fontwidth;			VertexAry[3 + i*12] = (float)y;
		VertexAry[4 + i*12] = (float)x + fontwidth + i*fontwidth;	VertexAry[5 + i*12] = (float)y;
		VertexAry[6 + i*12] = (float)x + i*fontwidth;			VertexAry[7 + i*12] = (float)y + fontheight;
		VertexAry[8 + i*12] = (float)x + fontwidth + i*fontwidth;	VertexAry[9 + i*12] = (float)y + fontheight;
		VertexAry[10 + i*12] = (float)x + fontwidth + i*fontwidth;	VertexAry[11 + i*12] = (float)y + fontheight;
		TexCoordAry[0 + i*12] = t_u;		TexCoordAry[1 + i*12] = t_v;
		TexCoordAry[2 + i*12] = t_u;		TexCoordAry[3 + i*12] = t_v;
		TexCoordAry[4 + i*12] = t_u + font_u;	TexCoordAry[5 + i*12] = t_v;
		TexCoordAry[6 + i*12] = t_u;		TexCoordAry[7 + i*12] = t_v + font_v;
		TexCoordAry[8 + i*12] = t_u + font_u;	TexCoordAry[9 + i*12] = t_v + font_v;
		TexCoordAry[10 + i*12] = t_u + font_u;	TexCoordAry[11 + i*12] = t_v + font_v;
	}

	//`
	glVertexPointer(2, GL_FLOAT, 0, VertexAry);
	glColorPointer(4, GL_UNSIGNED_BYTE, 0, ColorAry);
	glTexCoordPointer(2, GL_FLOAT, 0, TexCoordAry);
	glDrawArrays(GL_TRIANGLE_STRIP, 1, strlens*6-2);

	//z񖳌
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

	//2D`pݒ
	End2DRender();

	delete [] VertexAry;
	delete [] ColorAry;
	delete [] TexCoordAry;
}

//! @brief `
//! @param x1 n_ xW
//! @param y1 n_ yW
//! @param x2 I_ xW
//! @param y2 I_ yW
//! @param color F
void D3DGraphics::Draw2DLine(int x1, int y1, int x2, int y2, int color)
{
	float VertexAry[2*2];
	unsigned char ColorAry[2*4];

	//2D`pݒKp
	Start2DRender();

	//eNX`
	glDisable(GL_TEXTURE_2D);

	//_zp
	VertexAry[0] = (float)x1;	VertexAry[1] = (float)y1;
	VertexAry[2] = (float)x2;	VertexAry[3] = (float)y2;

	//Fzp
	ColorAry[0] = (color>>24)&0xFF;
	ColorAry[1] = (color>>16)&0xFF;
	ColorAry[2] = (color>>8)&0xFF;
	ColorAry[3] = color&0xFF;
	memcpy(&(ColorAry[4]), ColorAry, sizeof(unsigned char)*4);

	//zL
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);

	//`
	glVertexPointer(2, GL_FLOAT, 0, VertexAry);
	glColorPointer(4, GL_UNSIGNED_BYTE, 0, ColorAry);
	glDrawArrays(GL_LINE_STRIP, 0, 2);

	//z񖳌
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);

	//2D`pݒ
	End2DRender();
}

//! @brief ~i16p`j`
//! @param x S xW
//! @param y S yW
//! @param r a
//! @param color F
void D3DGraphics::Draw2DCycle(int x, int y, int r, int color)
{
	float VertexAry[(16+1)*2];
	unsigned char ColorAry[(16+1)*4];

	//2D`pݒKp
	Start2DRender();

	//eNX`
	glDisable(GL_TEXTURE_2D);

	//_Wݒ
	for(int i=0; i<16+1; i++){
		float x2, y2;
		x2 = (float)x + cos((float)M_PI*2/16 * i) * r;
		y2 = (float)y + sin((float)M_PI*2/16 * i) * r;
		VertexAry[i*2] = x2;	VertexAry[i*2+1] = y2;
	}

	//Fzp
	ColorAry[0] = (color>>24)&0xFF;
	ColorAry[1] = (color>>16)&0xFF;
	ColorAry[2] = (color>>8)&0xFF;
	ColorAry[3] = color&0xFF;
	for(int i=1; i<16+1; i++){
		memcpy(&(ColorAry[i*4]), ColorAry, sizeof(unsigned char)*4);
	}

	//zL
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);

	//`
	glVertexPointer(2, GL_FLOAT, 0, VertexAry);
	glColorPointer(4, GL_UNSIGNED_BYTE, 0, ColorAry);
	glDrawArrays(GL_LINE_STRIP, 0, 16+1);

	//z񖳌
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);

	//2D`pݒ
	End2DRender();
}

//! @brief lp``
//! @param x1  xW
//! @param y1  yW
//! @param x2 E xW
//! @param y2 E yW
//! @param color F
void D3DGraphics::Draw2DBox(int x1, int y1, int x2, int y2, int color)
{
	float VertexAry[4*2];
	unsigned char ColorAry[4*4];

	//2D`pݒKp
	Start2DRender();

	//eNX`
	glDisable(GL_TEXTURE_2D);

	//_zp
	VertexAry[0] = (float)x1;	VertexAry[1] = (float)y1;
	VertexAry[2] = (float)x2;	VertexAry[3] = (float)y1;
	VertexAry[4] = (float)x1;	VertexAry[5] = (float)y2;
	VertexAry[6] = (float)x2;	VertexAry[7] = (float)y2;

	//Fzp
	ColorAry[0] = (color>>24)&0xFF;
	ColorAry[1] = (color>>16)&0xFF;
	ColorAry[2] = (color>>8)&0xFF;
	ColorAry[3] = color&0xFF;
	for(int i=1; i<4; i++){
		memcpy(&(ColorAry[i*4]), ColorAry, sizeof(unsigned char)*4);
	}

	//zL
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);

	//`
	glVertexPointer(2, GL_FLOAT, 0, VertexAry);
	glColorPointer(4, GL_UNSIGNED_BYTE, 0, ColorAry);
	glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

	//z񖳌
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);

	//2D`pݒ
	End2DRender();
}

//! @brief 摜`
//! @param x xW
//! @param y yW
//! @param id eNX`Fԍ
//! @param width 
//! @param height 
//! @param alpha xi0.0`1.0j
void D3DGraphics::Draw2DTexture(int x, int y, int id, int width, int height, float alpha)
{
	//ȃeNX`ԍw肳ĂΏȂ
	if( id == -1 ){ return; }

	float VertexAry[4*2];
	float ColorAry[4*4];
	float TexCoordAry[4*2];

	//2D`pݒKp
	Start2DRender();

	//eNX`L
	glEnable(GL_TEXTURE_2D);

	//eNX`Zbg
	SetTexture(id);

	//_zp
	VertexAry[0] = (float)x;		VertexAry[1] = (float)y;
	VertexAry[2] = (float)x+width;	VertexAry[3] = (float)y;
	VertexAry[4] = (float)x;		VertexAry[5] = (float)y+height;
	VertexAry[6] = (float)x+width;	VertexAry[7] = (float)y+height;

	//Fzp
	ColorAry[0] = 1.0f;
	ColorAry[1] = 1.0f;
	ColorAry[2] = 1.0f;
	ColorAry[3] = alpha;
	for(int i=1; i<4; i++){
		memcpy(&(ColorAry[i*4]), ColorAry, sizeof(float)*4);
	}

	//UVWzp
	TexCoordAry[0] = 0.0f;	TexCoordAry[1] = 0.0f;
	TexCoordAry[2] = 1.0f;	TexCoordAry[3] = 0.0f;
	TexCoordAry[4] = 0.0f;	TexCoordAry[5] = 1.0f;
	TexCoordAry[6] = 1.0f;	TexCoordAry[7] = 1.0f;

	//zL
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);

	//`
	glVertexPointer(2, GL_FLOAT, 0, VertexAry);
	glColorPointer(4, GL_FLOAT, 0, ColorAry);
	glTexCoordPointer(2, GL_FLOAT, 0, TexCoordAry);
	glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

	//z񖳌
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

	//2D`pݒ
	End2DRender();
}

//! @brief 2D`pݒ
void D3DGraphics::End2DRender()
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glPopMatrix();

	glEnable(GL_DEPTH_TEST);
	glEnable(GL_CULL_FACE);
}

//! @brief ʂ̃XN[Vbgۑ
//! @param filename t@C
//! @return Ftrue@sFfalse
bool D3DGraphics::SaveScreenShot(char* filename)
{
	return false;
}

//! @brief J[R[h擾
//! @param red ԁi0.0f`1.0fj
//! @param green ΁i0.0f`1.0fj
//! @param blue i0.0f`1.0fj
//! @param alpha xi0.0f`1.0fj
//! @return J[R[h
int D3DGraphics::GetColorCode(float red, float green, float blue, float alpha)
{
	unsigned char red2, green2, blue2, alpha2;
	red2 = (unsigned char)(red*255);
	green2 = (unsigned char)(green*255);
	blue2 = (unsigned char)(blue*255);
	alpha2 = (unsigned char)(alpha*255);

	return (red2 << 24) | (green2 << 16) | (blue2 << 8) | alpha2;
}

#endif	//GRAPHICS_OPENGL