//! @file		pf_gpio.c
//! @brief		プラットフォーム(GPIO)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "nrf52833.h"
#include "pf_gpio.h"

//! @brief		ポート数
#define PF_GPIO_PORT_MAX				((u4)2U)

//! @brief		未使用ピン数
//! @details	外乱ノイズの影響を避ける目的で、内蔵プルアップ抵抗を使用する
#define PF_GPIO_NC_PIN_MAX				((u4)9U)

//! @brief		入出力方向
typedef enum PF_GPIO_DIR_Tag
{
	PF_GPIO_DIR_INPUT = 0x0000,					//!< 入力方向
	PF_GPIO_DIR_OUTPUT = 0x0001,				//!< 出力方向
} PF_GPIO_DIR;

//! @brief		入力バッファ
typedef enum PF_GPIO_INBUF_Tag
{
	PF_GPIO_INBUF_CONNECT = 0x0000,				//!< 入力バッファ接続
	PF_GPIO_INBUF_DISCONNECT = 0x0002,			//!< 入力バッファ未接続
} PF_GPIO_INBUF;

//! @brief		プルアップ・プルダウン
typedef enum PF_GPIO_PULL_Tag
{
	PF_GPIO_PULL_NONE = 0x0000,					//!< プルアップなし・プルダウンなし
	PF_GPIO_PULL_UP = 0x0004,					//!< プルアップあり
	PF_GPIO_PULL_DOWN = 0x000C,					//!< プルダウンあり
} PF_GPIO_PULL;

//! @brief		駆動特性
typedef enum PF_GPIO_DRIVE_Tag
{
	PF_GPIO_DRIVE_S0S1 = 0x0000,				//!< '0'=標準/'1'=標準
	PF_GPIO_DRIVE_H0S1 = 0x0100,				//!< '0'=高駆動/'1'=標準
	PF_GPIO_DRIVE_S0H1 = 0x0200,				//!< '0'=標準/'1'=高駆動
	PF_GPIO_DRIVE_H0H1 = 0x0300,				//!< '0'=高駆動/'1'=高駆動
	PF_GPIO_DRIVE_D0S1 = 0x0400,				//!< '0'=未接続/'1'=標準
	PF_GPIO_DRIVE_D0H1 = 0x0500,				//!< '0'=未接続/'1'=高駆動
	PF_GPIO_DRIVE_S0D1 = 0x0600,				//!< '0'=標準/'1'=未接続
	PF_GPIO_DRIVE_H0D1 = 0x0700,				//!< '0'=高駆動/'1'=未接続
} PF_GPIO_DRIVE;

//! @brief		GPIOピン初期化情報構造体
typedef struct PF_GPIO_INIT_Tag
{
	u4				port;						//!< ポート(0～1)
	u4				pin;						//!< ピン(0～31)
	BOOL			level;						//!< 初期出力レベル
	PF_GPIO_DIR		dir;						//!< 入出力方向
	PF_GPIO_INBUF	inbuf;						//!< 入力バッファ
	PF_GPIO_PULL	pull;						//!< プルアップ・プルダウン
	PF_GPIO_DRIVE	drive;						//!< 駆動特性
} PF_GPIO_INIT;

//! @brief		GPIOピン初期化情報テーブル
static const PF_GPIO_INIT pf_gpio_init_table[PF_GPIO_ID_MAX] =
{
	// PF_GPIO_ID_UART_TXD
	{
		0,										// ポート
		6,										// ピン
		TRUE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0S1,						// 駆動特性
	},

	// PF_GPIO_ID_UART_RXD
	{
		1,										// ポート
		8,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_INPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0S1,						// 駆動特性
	},

	// PF_GPIO_ID_ROW1
	{
		0,										// ポート
		21,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_H0H1,						// 駆動特性
	},

	// PF_GPIO_ID_ROW2
	{
		0,										// ポート
		22,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_H0H1,						// 駆動特性
	},

	// PF_GPIO_ID_ROW3
	{
		0,										// ポート
		15,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_H0H1,						// 駆動特性
	},

	// PF_GPIO_ID_ROW4
	{
		0,										// ポート
		24,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_H0H1,						// 駆動特性
	},

	// PF_GPIO_ID_ROW5
	{
		0,										// ポート
		19,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_H0H1,						// 駆動特性
	},


	// PF_GPIO_ID_COL1
	{
		0,										// ポート
		28,										// ピン
		TRUE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_H0H1,						// 駆動特性
	},

	// PF_GPIO_ID_COL2
	{
		0,										// ポート
		11,										// ピン
		TRUE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_H0H1,						// 駆動特性
	},

	// PF_GPIO_ID_COL3
	{
		0,										// ポート
		31,										// ピン
		TRUE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_H0H1,						// 駆動特性
	},

	// PF_GPIO_ID_COL4
	{
		1,										// ポート
		5,										// ピン
		TRUE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_H0H1,						// 駆動特性
	},

	// PF_GPIO_ID_COL5
	{
		0,										// ポート
		30,										// ピン
		TRUE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_H0H1,						// 駆動特性
	},

	// PF_GPIO_ID_BUTTON_A
	{
		0,										// ポート
		14,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_INPUT,						// 入出力方向
		PF_GPIO_INBUF_CONNECT,					// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0S1,						// 駆動特性
	},

	// PF_GPIO_ID_BUTTON_B
	{
		0,										// ポート
		23,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_INPUT,						// 入出力方向
		PF_GPIO_INBUF_CONNECT,					// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0S1,						// 駆動特性
	},

	// PF_FPIO_ID_INTERNAL_IRQ
	{
		0,										// ポート
		25,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_INPUT,						// 入出力方向
		PF_GPIO_INBUF_CONNECT,					// 入力バッファ
		PF_GPIO_PULL_UP,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0S1,						// 駆動特性
	},

	// PF_GPIO_ID_INTERNAL_SCL
	{
		0,										// ポート
		8,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_INPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0D1,						// 駆動特性
	},

	// PF_GPIO_ID_INTERNAL_SDA
	{
		0,										// ポート
		16,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_INPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0D1,						// 駆動特性
	},

	// PF_GPIO_ID_EXTERNAL_SCL
	{
		0,										// ポート
		26,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_INPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0D1,						// 駆動特性
	},

	// PF_GPIO_ID_EXTERNAL_SDA
	{
		1,										// ポート
		0,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_INPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0D1,						// 駆動特性
	},

	// PF_GPIO_ID_INTERNAL_SPEAKER
	{
		0,										// ポート
		0,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_H0H1,						// 駆動特性
	},

	// PF_GPIO_ID_MAQUEEN_SPEAKER
	{
		0,										// ポート
		2,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0S1,						// 駆動特性
	},

	// PF_GPIO_ID_MAQUEEN_LED_L
	{
		0,										// ポート
		10,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0S1,						// 駆動特性
	},

	// PF_GPIO_ID_MAQUEEN_LED_R
	{
		0,										// ポート
		12,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0S1,						// 駆動特性
	},

	// PF_GPIO_ID_MAQUEEN_PATROL_L
	{
		0,										// ポート
		17,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_INPUT,						// 入出力方向
		PF_GPIO_INBUF_CONNECT,					// 入力バッファ
		PF_GPIO_PULL_UP,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0S1,						// 駆動特性
	},

	// PF_GPIO_ID_MAQUEEN_PATROL_R
	{
		0,										// ポート
		1,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_INPUT,						// 入出力方向
		PF_GPIO_INBUF_CONNECT,					// 入力バッファ
		PF_GPIO_PULL_UP,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0S1,						// 駆動特性
	},

	// PF_GPIO_ID_MAQUEEN_TRIG
	{
		0,										// ポート
		3,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0S1,						// 駆動特性
	},

	// PF_GPIO_ID_MAQUEEN_ECHO
	{
		0,										// ポート
		4,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_INPUT,						// 入出力方向
		PF_GPIO_INBUF_CONNECT,					// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0S1,						// 駆動特性
	},

	// PF_GPIO_ID_MAQUEEN_NEOPIXEL
	{
		0,										// ポート
		13,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_OUTPUT,						// 入出力方向
		PF_GPIO_INBUF_DISCONNECT,				// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0S1,						// 駆動特性
	},

	// PF_GPIO_ID_MAQUEEN_NECIR
	{
		1,										// ポート
		2,										// ピン
		FALSE,									// 初期出力レベル
		PF_GPIO_DIR_INPUT,						// 入出力方向
		PF_GPIO_INBUF_CONNECT,					// 入力バッファ
		PF_GPIO_PULL_NONE,						// プルアップ・プルダウン
		PF_GPIO_DRIVE_S0S1,						// 駆動特性
	},
};

//! @brief		GPIOピン未使用情報構造体
typedef struct PF_GPIO_NC_Tag
{
	u4				port;						//!< ポート(0～1)
	u4				pin;						//!< ピン(0～31)
} PF_GPIO_NC;

//! @brief		GPIOピン未使用情報テーブル
static const PF_GPIO_NC pf_gpio_nc_table[PF_GPIO_NC_PIN_MAX] =
{
	// P0.07
	{
		0,
		7,
	},

	// P0.18
	{
		0,
		18,
	},

	// P0.27
	{
		0,
		27,
	},

	// P0.29
	{
		0,
		29,
	},

	// P1.01
	{
		1,
		1,
	},

	// P1.03
	{
		1,
		3,
	},

	// P1.06
	{
		1,
		6,
	},

	// P1.07
	{
		1,
		7,
	},

	// P1.09
	{
		1,
		9,
	},
};

//! @brief		GPIOポート→デバイステーブル
static NRF_GPIO_Type* const pf_gpio_port_to_dev[PF_GPIO_PORT_MAX] =
{
	NRF_P0,									//!< ポート0
	NRF_P1,									//!< ポート1
};

//! @brief		GPIO初期化(未使用ピン)
static void pf_gpio_init_nc(void)
{
	u4 loop;
	NRF_GPIO_Type *dev;
	u4 cnf;

	// オート変数初期化
	loop = 0;
	dev = NULL;
	cnf = PF_GPIO_DIR_INPUT | PF_GPIO_INBUF_DISCONNECT | PF_GPIO_PULL_UP | PF_GPIO_DRIVE_S0S1;

	for (loop = 0; loop < PF_GPIO_NC_PIN_MAX; loop++)
	{
		// ポートからデバイスを取得
		dev = pf_gpio_port_to_dev[pf_gpio_nc_table[loop].port];

		// PIN_CNFレジスタを設定
		dev->PIN_CNF[pf_gpio_nc_table[loop].pin] = cnf;
	}
}

//! @brief		GPIO初期化(単一のID)
//! @param		[in] id			GPIOピンのID
static void pf_gpio_init_id(PF_GPIO_ID id)
{
	NRF_GPIO_Type *dev;

	// オート変数初期化
	dev = NULL;

	// 出力方向の場合、先に出力値を設定する
	if (PF_GPIO_DIR_OUTPUT == pf_gpio_init_table[id].dir)
	{
		pf_gpio_output(id, pf_gpio_init_table[id].level);
	}

	// IDからデバイスを取得
	dev = pf_gpio_port_to_dev[pf_gpio_init_table[id].port];

	// コンフィグ設定
	dev->PIN_CNF[pf_gpio_init_table[id].pin] = (u4)(pf_gpio_init_table[id].dir
					| pf_gpio_init_table[id].inbuf
					| pf_gpio_init_table[id].pull
					| pf_gpio_init_table[id].drive);
}

//! @brief		GPIO初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_gpio_init(void)
{
	PF_GPIO_ID id;

	// オート変数初期化
	id = 0;

	// 未使用ピンを初期化
	pf_gpio_init_nc();

	// すべてのIDをループ
	for (id = 0; id < PF_GPIO_ID_MAX; id++)
	{
		// 1つのIDを初期化
		pf_gpio_init_id(id);
	}
}

//! @brief		GPIOポート番号を取得
//! @param		[in] id			GPIOピンのID
//! @return		GPIOポート番号(0 or 1)
u4 pf_gpio_get_port(PF_GPIO_ID id)
{
	u4 port;

	// オート変数初期化
	port = 0;

	// パラメータチェック
	if (id < PF_GPIO_ID_MAX)
	{
		port = pf_gpio_init_table[id].port;
	}

	return port;
}

//! @brief		GPIOピン番号を取得
//! @param		[in] id			GPIOピンのID
//! @return		GPIOピン番号(0～31)
u4 pf_gpio_get_pin(PF_GPIO_ID id)
{
	u4 pin;

	// オート変数初期化
	pin = 0;

	// パラメータチェック
	if (id < PF_GPIO_ID_MAX)
	{
		pin = pf_gpio_init_table[id].pin;
	}

	return pin;
}

//! @brief		GPIOから入力
//! @param		[in] id			GPIOピンのID
//! @return		入力レベル(TRUE='H'レベル/FALSE='L'レベル)
BOOL pf_gpio_input(PF_GPIO_ID id)
{
	NRF_GPIO_Type *dev;
	u4 in;
	BOOL result;

	// オート変数初期化
	dev = NULL;
	in = 0;
	result = FALSE;

	// パラメータチェック
	if (id < PF_GPIO_ID_MAX)
	{
		// IDからデバイスを取得
		dev = pf_gpio_port_to_dev[pf_gpio_init_table[id].port];

		// 当該ピンのみを取り出した情報を取得
		in = dev->IN & (1 << pf_gpio_init_table[id].pin);

		// 0で'L'レベル、1で'H'レベル
		if (0 == in)
		{
			// 'L'レベル
			result = FALSE;
		}
		else
		{
			// 'H'レベル
			result = TRUE;
		}
	}

	return result;
}

//! @brief		GPIOへ出力
//! @param		[in] id			GPIOピンのID
//! @param		[in] level		出力レベル
void pf_gpio_output(PF_GPIO_ID id, BOOL level)
{
	NRF_GPIO_Type *dev;

	// オート変数初期化
	dev = NULL;

	// パラメータチェック
	if (id < PF_GPIO_ID_MAX)
	{
		// IDからデバイスを取得
		dev = pf_gpio_port_to_dev[pf_gpio_init_table[id].port];

		// 出力
		if (FALSE == level)
		{
			// 'L'レベル
			dev->OUTCLR = 1 << pf_gpio_init_table[id].pin;
		}
		else
		{
			// 'H'レベル
			dev->OUTSET = 1 << pf_gpio_init_table[id].pin;
		}
	}
}
