//! @file		pf_interrupt.c
//! @brief		プラットフォーム(割り込み)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "nrf52833.h"
#include "pf_interrupt.h"

//! @brief		グローバル割り込み禁止
//! @return		直前のグローバル割り込み禁止状態(1:割り込み禁止/0:割り込み許可)
u4 pf_interrupt_global_disable(void)
{
	u4 primask;

	// オート変数初期化
	primask = 0;

	// 現在のPRIMASKビットを取得
	primask = (u4)__get_PRIMASK();

	// 割り込み禁止
	__disable_irq();

	// メモリバリア・命令バリア(念のため)
	__DSB();
	__ISB();

	return primask;
}

//! @brief		グローバル割り込み復元
//! @param		[in] primask	pf_interrupt_global_disable()の返り値
void pf_interrupt_global_restore(u4 primask)
{
	// 直前の状態が割り込み許可の場合のみ
	if (0 == primask)
	{
		// 割り込み許可
		__enable_irq();
	}
}

//! @brief		割り込み番号を取得
//! @param		[in] pri		割り込み優先度
//! @return		割り込み番号(priが不正の場合、UARTE1_IRQn)
static IRQn_Type pf_interrupt_get_irq(PF_INTERRUPT_PRI pri)
{
	IRQn_Type irq;

	// オート変数初期化
	irq = UARTE1_IRQn;

	// 割り込み優先度から割り込み番号を取得
	switch (pri)
	{
	// UART
	case PF_INTERRUPT_PRI_UART:
		irq = UARTE0_UART0_IRQn;
		break;

	// Display Timer
	case PF_INTERRUPT_PRI_DISPLAY_TIMER:
		irq = TIMER0_IRQn;
		break;

	// I2C(内部)
	case PF_INTERRUPT_PRI_I2C_INT:
		irq = SPIM0_SPIS0_TWIM0_TWIS0_SPI0_TWI0_IRQn;
		break;

	// I2C(外部)
	case PF_INTERRUPT_PRI_I2C_EXT:
		irq = SPIM1_SPIS1_TWIM1_TWIS1_SPI1_TWI1_IRQn;
		break;

	// 音楽演奏
	case PF_INTERRUPT_PRI_MUSIC:
		irq = TIMER1_IRQn;
		break;

	default:
		break;
	}

	return irq;
}

//! @brief		ローカル割り込み禁止
//! @param		[in] pri		割り込み優先度
//! @return		直前のローカル割り込み禁止状態(1:割り込み禁止/0:割り込み許可)
u4 pf_interrupt_local_disable(PF_INTERRUPT_PRI pri)
{
	u4 enable;
	IRQn_Type irq;

	// オート変数初期化
	enable = 0;
	irq = UARTE1_IRQn;

	// SysTickは別制御(NVICで割り込み許可/割り込み禁止を管理していないため)
	if (PF_INTERRUPT_PRI_SYSTICK == pri)
	{
		enable = pf_interrupt_global_disable();
	}
	else
	{
		// 割り込み優先度から割り込み番号を取得
		irq = pf_interrupt_get_irq(pri);

		// 特定できた場合のみ、割り込み禁止を行う
		if (UARTE1_IRQn != irq)
		{
			enable = NVIC_GetEnableIRQ(irq);
			NVIC_DisableIRQ(irq);
		}
	}

	return enable;
}

//! @brief		ローカル割り込み復元
//! @param		[in] pri		割り込み優先度
//! @param		[in] enable		pf_interrupt_local_disable(pri)の返り値
void pf_interrupt_local_restore(PF_INTERRUPT_PRI pri, u4 enable)
{
	IRQn_Type irq;

	// オート変数初期化
	irq = UARTE1_IRQn;

	// SysTickは別制御(NVICで割り込み許可/割り込み禁止を管理していないため)
	if (PF_INTERRUPT_PRI_SYSTICK == pri)
	{
		pf_interrupt_global_restore(enable);
	}
	else
	{
		// 割り込み優先度から割り込み番号を取得
		irq = pf_interrupt_get_irq(pri);

		// 特定できた場合のみ、割り込み復元を行う
		if (UARTE1_IRQn != irq)
		{
			// 直前の割り込み禁止状態が割り込み許可の場合のみ
			if (0 != enable)
			{
				NVIC_EnableIRQ(irq);
			}
		}
	}
}
