//! @file		pf_power.c
//! @brief		プラットフォーム(Power)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "nrf52833.h"
#include "nrf52833_bitfields.h"
#include "pf_interrupt.h"
#include "pf_display.h"
#include "pf_systick.h"
#include "pf_power.h"

//! @brief		VDD電圧低下警告有無フラグ
static BOOL pf_power_warning = FALSE;

//! @brief		インタフェースMCU初期化待ちのためのデモ(アニメーション)
//! @details	NRF_POWER->RESETREAS==0の場合に限り呼び出される。所要時間1000msで設計している
static void pf_power_greetings(void)
{
	u4 base;
	u4 freerun;
	u4 brightness;
	u4 target;

	// オート変数初期化
	base = pf_systick_freerun();
	freerun = base;
	brightness = PF_DISPLAY_BRIGHTNESS_MIN;
	target = brightness;

	// 初期表示
	pf_display_brightness(brightness);
	pf_display_id(PF_DISPLAY_ID_HAPPY);

	// 460msループ(輝度を上げる)
	while ((u4)(freerun - base) <= ((PF_DISPLAY_BRIGHTNESS_MAX - PF_DISPLAY_BRIGHTNESS_MIN) * 5))
	{
		// 目標輝度を算出
		target = (u4)(freerun - base);
		target /= 5;
		target += PF_DISPLAY_BRIGHTNESS_MIN;

		// brightnessと異なったら設定
		if (target != brightness)
		{
			pf_display_brightness(target);
			brightness = target;
		}

		// freerunを再取得
		freerun = pf_systick_freerun();
	}

	// baseを加算
	base += ((PF_DISPLAY_BRIGHTNESS_MAX - PF_DISPLAY_BRIGHTNESS_MIN) * 5);

	// 80msループ(輝度を保つ)
	while ((u4)(freerun - base) <= 80)
	{
		// freerunを再取得
		freerun = pf_systick_freerun();
	}

	// baseを加算
	base += 80;

	// 460msループ(輝度を下げる)
	while ((u4)(freerun - base) <= ((PF_DISPLAY_BRIGHTNESS_MAX - PF_DISPLAY_BRIGHTNESS_MIN) * 5))
	{
		// 目標輝度を算出
		target = (u4)(freerun - base);
		target /= 5;
		target = PF_DISPLAY_BRIGHTNESS_MAX - target;

		// brightnessと異なったら設定
		if (target != brightness)
		{
			pf_display_brightness(target);
			brightness = target;
		}

		// freerunを再取得
		freerun = pf_systick_freerun();
	}

	// 表示OFF
	pf_display_off();
}

//! @brief		Power初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
//! @attention	Display初期化の後で呼び出すこと
void pf_power_init(void)
{
	// VDD電圧低下警告なし
	pf_power_warning = FALSE;

	// VDD電圧低下警告(VDD、VDDHとも2.8Vで警告)
	NRF_POWER->POFCON = (POWER_POFCON_POF_Enabled << POWER_POFCON_POF_Pos)
			| (POWER_POFCON_THRESHOLD_V28 << POWER_POFCON_THRESHOLD_Pos)
			| (POWER_POFCON_THRESHOLDVDDH_V28 << POWER_POFCON_THRESHOLDVDDH_Pos);

	// パワーオンリセットの場合、インタフェースMCUの起動待ちのためのデモを行う
	if (0 == NRF_POWER->RESETREAS)
	{
		pf_power_greetings();
	}
}

//! @brief		Power定期タスク
//! @remarks	プラットフォーム定期タスク(入力系)処理から呼び出すこと
void pf_power_task(void)
{
	// VDD電圧低下警告なしの場合
	if (FALSE == pf_power_warning)
	{
		// POFWARN
		if (POWER_EVENTS_POFWARN_EVENTS_POFWARN_Generated << POWER_EVENTS_POFWARN_EVENTS_POFWARN_Pos
						== NRF_POWER->EVENTS_POFWARN)
		{
			// VDD電圧低下警告あり
			pf_power_warning = TRUE;

			// ディスプレイ表示
			pf_display_powerdown();
		}
	}
}
