//! @file		pf_neopixel.c
//! @brief		プラットフォーム(NeoPixel)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "pf_gpio.h"
#include "pf_neopixel.h"

//! @brief		NepPixelあたりの送信バイト数
#define PF_NEOPIXEL_SEND_BYTES			((u4)24U)

//! @brief		R/G/Bの階調マスク値
#define PF_NEOPIXEL_COLOR_MASK			((u1)0x7FU)

//! @brief		外部関数(アセンブラ)定義
//! @details	pin=13<<1, num_bytes=(Gx8,Bx8,Rx8)xLEDs=96を指定する
extern void sendNeopixelBuffer(uint32_t pin, uint8_t* data_address, uint16_t num_bytes);

//! @brief		NeoPixel動作情報構造体
typedef struct PF_NEOPIXEL_INFO_Tag
{
	PF_NEOPIXEL_COLOR	current[PF_NEOPIXEL_ID_MAX];	//!< 現在のRGB
	PF_NEOPIXEL_COLOR	target[PF_NEOPIXEL_ID_MAX];		//!< 目的とするRGB
	u1 buf[PF_NEOPIXEL_ID_MAX][PF_NEOPIXEL_SEND_BYTES];	//!< 送信バッファ
} PF_NEOPIXEL_INFO;

//! @brief		NeoPixel動作情報
static PF_NEOPIXEL_INFO pf_neopixel_info;

//! @brief		NeoPixel初期化(単一のID)
//! @param		[in] id			NeoPixelのID
static void pf_neopixel_init_id(PF_NEOPIXEL_ID id)
{
	// 現在の状態は消灯
	pf_neopixel_info.current[id].r = 0U;
	pf_neopixel_info.current[id].g = 0U;
	pf_neopixel_info.current[id].b = 0U;

	// 目的とする状態は消灯
	pf_neopixel_info.target[id].r = 0U;
	pf_neopixel_info.target[id].g = 0U;
	pf_neopixel_info.target[id].b = 0U;
}

//! @brief		NeoPixel初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_neopixel_init(void)
{
	PF_NEOPIXEL_ID id;

	// オート変数初期化
	id = 0;

	// すべてのIDをループ
	for (id = 0; id < PF_NEOPIXEL_ID_MAX; id++)
	{
		// 1つのIDを初期化
		pf_neopixel_init_id(id);
	}
}

//! @brief		NeoPixel定期タスク
//! @remarks	プラットフォーム定期タスク(出力系)処理から呼び出すこと
void pf_neopixel_task(void)
{

}

//! @brief		NeoPixel制御
//! @param		[in] id			NeoPixelのID
//! @param		[in] color		NeoPixelカラー構造体へのポインタ
void pf_neopixel_ctrl(PF_NEOPIXEL_ID id, const PF_NEOPIXEL_COLOR *color)
{
	// パラメータチェック
	if (id < PF_NEOPIXEL_ID_MAX)
	{
		// RGBカラーをセット
		pf_neopixel_info.target[id].r = color->r & PF_NEOPIXEL_COLOR_MASK;
		pf_neopixel_info.target[id].g = color->g & PF_NEOPIXEL_COLOR_MASK;
		pf_neopixel_info.target[id].b = color->b & PF_NEOPIXEL_COLOR_MASK;
	}
}
