//! @file		pf_ultra.c
//! @brief		プラットフォーム(超音波)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "nrf52833.h"
#include "nrf52833_bitfields.h"
#include "pf_gpio.h"
#include "pf_systick.h"
#include "pf_interrupt.h"
#include "pf_timer.h"
#include "pf_ultra.h"

//! @brief		測定間隔[ms]
//! @details	データシートより200ms以上が必要
#define PF_ULTRA_MEASURE_CYCLE_MS		((u4)200U)

//! @brief		TRIGホールド時間[us]
//! @details	データシートより10us以上が必要(少しマージンを取る)
#define PF_ULTRA_TRIG_HOLD_US			((u4)14U)

//! @brief		測定温度
//! @details	温度センサと組み合わせるのが理想だが、一律25℃で固定する
#define PF_ULTRA_TEMPERATURE			((f4)25.0f)

//! @brief		GPIOTE立ち上がり検出チャネル
#define PF_ULTRA_GPIOTE_RISING_CH		((u4)0U)

//! @brief		GPIOTE立ち下がり検出チャネル
#define PF_ULTRA_GPIOTE_FALLING_CH		((u4)1U)

//! @brief		PPI立ち上がり接続チャネル
#define PF_ULTRA_PPI_RISING_CH			((u4)0U)

//! @brief		PPI立ち下がり接続チャネル
#define PF_ULTRA_PPI_FALLING_CH			((u4)1U)

//! @brief		超音波動作情報構造体
typedef struct PF_ULTRA_INFO_Tag
{
	BOOL			enable;					//!< 測定有効フラグ
	u4				cm;						//!< 前方までの距離(エラー時はPF_ULTRA_UPPER_LIMIT)
	u4				count;					//!< 周期カウンタ
	BOOL			measure;				//!< 測定中フラグ
} PF_ULTRA_INFO;

//! @brief		超音波動作情報
static PF_ULTRA_INFO pf_ultra_info;

//! @brief		GPIOTE初期化
static void pf_ultra_init_gpiote(void)
{
	u4 port;
	u4 pin;

	// オート変数初期化
	port = 0;
	pin = 0;

	// GPIOより、ECHOのポートとピンを取得
	port = pf_gpio_get_port(PF_GPIO_ID_MAQUEEN_ECHO);
	pin = pf_gpio_get_pin(PF_GPIO_ID_MAQUEEN_ECHO);

	// 立ち上がり検出チャネルコンフィグ(Eventモード、Rising Edgeで使用)
	NRF_GPIOTE->CONFIG[PF_ULTRA_GPIOTE_RISING_CH] = (GPIOTE_CONFIG_MODE_Event
					<< GPIOTE_CONFIG_MODE_Pos) | (pin << GPIOTE_CONFIG_PSEL_Pos)
					| (port << GPIOTE_CONFIG_PORT_Pos)
					| (GPIOTE_CONFIG_POLARITY_LoToHi << GPIOTE_CONFIG_POLARITY_Pos);

	// 立ち下がり検出チャネルコンフィグ(Eventモード、Falling Edgeで使用)
	NRF_GPIOTE->CONFIG[PF_ULTRA_GPIOTE_FALLING_CH] = (GPIOTE_CONFIG_MODE_Event
					<< GPIOTE_CONFIG_MODE_Pos) | (pin << GPIOTE_CONFIG_PSEL_Pos)
					| (port << GPIOTE_CONFIG_PORT_Pos)
					| (GPIOTE_CONFIG_POLARITY_HiToLo << GPIOTE_CONFIG_POLARITY_Pos);

	// 割り込みは使用しない
	NRF_GPIOTE->INTENSET = 0;
}

//! @brief		PPI初期化
static void pf_ultra_init_ppi(void)
{
	// 立ち上がりチャネル(GPIOTEのEVENTS_IN[PF_ULTRA_PPI_RISING_CH]をTIMERのTASK_STARTに接続)
	NRF_PPI->CH[PF_ULTRA_PPI_RISING_CH].EEP = (u4)&(NRF_GPIOTE->EVENTS_IN[PF_ULTRA_PPI_RISING_CH]);
	NRF_PPI->CH[PF_ULTRA_PPI_RISING_CH].TEP = pf_timer_get_tasks_start(PF_TIMER_ID_ULTRA);
	NRF_PPI->FORK[PF_ULTRA_PPI_RISING_CH].TEP = 0;
	NRF_PPI->CHENSET = 1U << PF_ULTRA_PPI_RISING_CH;

	// 立ち下がりチャネル(GPIOTEのEVENTS_IN[PF_ULTRA_PPI_FALLING_CH]をTIMERのTASK_CAPTUREに接続)
	NRF_PPI->CH[PF_ULTRA_PPI_FALLING_CH].EEP =
					(u4)&(NRF_GPIOTE->EVENTS_IN[PF_ULTRA_GPIOTE_FALLING_CH]);
	NRF_PPI->CH[PF_ULTRA_PPI_FALLING_CH].TEP = pf_timer_get_tasks_capture(PF_TIMER_ID_ULTRA);
	NRF_PPI->FORK[PF_ULTRA_PPI_FALLING_CH].TEP = 0;
	NRF_PPI->CHENSET = 1U << PF_ULTRA_PPI_FALLING_CH;
}

//! @brief		超音波初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_ultra_init(void)
{
	// 周辺機能を初期化
	pf_ultra_init_gpiote();
	pf_ultra_init_ppi();

	// 動作情報を初期化
	pf_ultra_info.enable = FALSE;
	pf_ultra_info.cm = PF_ULTRA_UPPER_LIMIT;
	pf_ultra_info.count = (PF_ULTRA_MEASURE_CYCLE_MS) / (PF_SYSTICK_SYNC_MS) - 1;
	pf_ultra_info.measure = FALSE;
}

//! @brief		usウェイト
//! @param		[in] us			待ち時間[us]
static void pf_ultra_wait_us(u4 us)
{
	u4 diff;
	PF_SYSTICK_TIME time[2];

	// オート変数初期化
	diff = 0;
	time[0].sec = 0;
	time[0].ms = 0;
	time[0].us = 0;
	time[1].sec = 0;
	time[1].ms = 0;
	time[1].us = 0;

	// 現在のSysTick時刻を取得
	pf_systick_time(&time[0]);

	// 差分が引数を超えるまでループ
	while (diff < us)
	{
		// time[1]を取得
		pf_systick_time(&time[1]);

		// msが変化なしか？
		if (time[0].ms == time[1].ms)
		{
			// msが同じであれば、usの差異を得る
			diff = time[1].us - time[0].us;
		}
		else
		{
			// msが異なっているなら、us差異に1000加算する
			diff = 1000U + time[1].us - time[0].us;
		}
	}
}

//! @brief		超音波測定開始
static void pf_ultra_start(void)
{
	// 測定開始
	pf_ultra_info.measure = TRUE;

	// TRIGピンをHighレベルにする
	pf_gpio_output(PF_GPIO_ID_MAQUEEN_TRIG, TRUE);

	// TRIGピンをHighのままホールドする
	pf_ultra_wait_us(PF_ULTRA_TRIG_HOLD_US);

	// GPIOTEイベントをクリア(前回エラー時を考慮)
	NRF_GPIOTE->EVENTS_IN[PF_ULTRA_GPIOTE_RISING_CH] = GPIOTE_EVENTS_PORT_EVENTS_PORT_NotGenerated
					<< GPIOTE_EVENTS_PORT_EVENTS_PORT_Pos;
	NRF_GPIOTE->EVENTS_IN[PF_ULTRA_GPIOTE_FALLING_CH] = GPIOTE_EVENTS_PORT_EVENTS_PORT_NotGenerated
					<< GPIOTE_EVENTS_PORT_EVENTS_PORT_Pos;

	// TRIGピンをLowレベルにする
	pf_gpio_output(PF_GPIO_ID_MAQUEEN_TRIG, FALSE);
}

//! @brief		測定距離キャプチャ
static void pf_ultra_capture(void)
{
	u4 capture;
	f4 speed_of_sound;
	f4 duration;
	f4 distance;

	// オート変数初期化
	capture = 0;
	speed_of_sound = 331.5f + 0.6f * PF_ULTRA_TEMPERATURE;
	duration = 0.0f;
	distance = 0.0f;

	// GPIOTEイベントをクリア
	NRF_GPIOTE->EVENTS_IN[PF_ULTRA_GPIOTE_RISING_CH] = GPIOTE_EVENTS_PORT_EVENTS_PORT_NotGenerated
					<< GPIOTE_EVENTS_PORT_EVENTS_PORT_Pos;
	NRF_GPIOTE->EVENTS_IN[PF_ULTRA_GPIOTE_FALLING_CH] = GPIOTE_EVENTS_PORT_EVENTS_PORT_NotGenerated
					<< GPIOTE_EVENTS_PORT_EVENTS_PORT_Pos;

	// タイマのキャプチャカウンタ(単位:us)を得る
	capture = pf_timer_get_capture(PF_TIMER_ID_ULTRA);

	// キャプチャカウンタから変換
	duration = (f4)capture;
	duration /= 2.0f;
	distance = (duration * speed_of_sound) / 10000.0f;

	// 測定結果を格納
	pf_ultra_info.cm = (u4)distance;

	// 上限チェック
	if (pf_ultra_info.cm >= PF_ULTRA_UPPER_LIMIT)
	{
		// 上限に抑える
		pf_ultra_info.cm = PF_ULTRA_UPPER_LIMIT;
	}

	// 測定完了
	pf_ultra_info.measure = FALSE;
}

//! @brief		完了ポーリング
static void pf_ultra_poll(void)
{
	u4 generated;

	// オート変数初期化
	generated = GPIOTE_EVENTS_PORT_EVENTS_PORT_Generated << GPIOTE_EVENTS_PORT_EVENTS_PORT_Pos;

	// GPIOTEイベントが共に発生していれば、測定完了
	if (generated == (NRF_GPIOTE->EVENTS_IN[PF_ULTRA_GPIOTE_RISING_CH] & generated))
	{
		if (generated == (NRF_GPIOTE->EVENTS_IN[PF_ULTRA_GPIOTE_FALLING_CH] & generated))
		{
			// 測定距離キャプチャ
			pf_ultra_capture();
		}
	}
}

//! @brief		超音波定期タスク
//! @remarks	プラットフォーム定期タスク(入力系)処理から呼び出すこと
void pf_ultra_task(void)
{
	// カウントアップ
	pf_ultra_info.count++;

	// 測定周期チェック
	if (((PF_ULTRA_MEASURE_CYCLE_MS) / (PF_SYSTICK_SYNC_MS)) >= pf_ultra_info.count)
	{
		// カウンタリセット
		pf_ultra_info.count = 0;

		// この時点で測定中であれば、ダミー読み出し後、エラーにする
		if (TRUE == pf_ultra_info.measure)
		{
			(void)pf_timer_get_capture(PF_TIMER_ID_ULTRA);
			pf_ultra_info.cm = PF_ULTRA_UPPER_LIMIT;
			pf_ultra_info.measure = FALSE;
		}

		// 測定が有効なら
		if (TRUE == pf_ultra_info.enable)
		{
			// 測定開始
			pf_ultra_start();
		}
	}
	else
	{
		// 測定中なら
		if (TRUE == pf_ultra_info.measure)
		{
			// 測定完了をポーリング
			pf_ultra_poll();
		}
	}
}

//! @brief		超音波測定の有効化・無効化
//! @param		[in] enable		有効フラグ(TRUE=有効化/FALSE=無効化)
void pf_ultra_enable(BOOL enable)
{
	// 無効化→有効化の場合、結果をリセット
	if ((TRUE == enable) && (FALSE == pf_ultra_info.enable))
	{
		pf_ultra_info.cm = PF_ULTRA_UPPER_LIMIT;
	}

	pf_ultra_info.enable = enable;
}

//! @brief		距離取得
//! @return		前方障害物までの距離(単位:cm)
u4 pf_ultra_get(void)
{
	return pf_ultra_info.cm;
}
