//! @file		pf_uart.h
//! @brief		プラットフォーム(UART)ヘッダファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#ifndef PF_UART_H
#define PF_UART_H

#include "pf_types.h"

//! @brief		UARTエラー情報構造体
typedef struct PF_UART_ERROR_Tag
{
	u4				overrun;				//!< オーバーランエラー発生回数
	u4				parity;					//!< パリティエラー発生回数
	u4				framing;				//!< フレーミングエラー発生回数
	u4				brk;					//!< ブレーク信号受信回数
} PF_UART_ERROR;

//! @brief		UART初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
//! @attention	GPIO初期化の後で呼び出すこと
void pf_uart_init(void);

//! @brief		UART定期タスク
//! @remarks	プラットフォーム定期タスク(入力系)処理から呼び出すこと
void pf_uart_task(void);

//! @brief		UART送信
//! @param		[in] buf		送信バッファへのポインタ
//! @param		[in] bytes		送信バイト数
void pf_uart_send(const u1 *buf, u4 bytes);

//! @brief		UART受信
//! @param		[out] buf		受信バッファへのポインタ
//! @param		[in] bytes		受信バッファの最大数
//! @return		受信したバイト数
u4 pf_uart_recv(u1 *buf, u4 bytes);

//! @brief		UARTエラー情報取得
//! @param		[out] error		エラー情報構造体へのポインタ
//! @remarks	プラットフォーム内部のエラー情報はクリアされる
void pf_uart_error(PF_UART_ERROR *error);

//! @brief		UART1文字出力
//! @param		[in] ch			送信する文字(0xFF以外)
void pf_uart_putc(u1 ch);

//! @brief		UART1文字入力
//! @return		受信した文字(受信していない場合、0xFF)
u1 pf_uart_getc(void);

//! @brief		UARTログ出力
//! @param		[in] format		printf形式の書式指定文字列(末尾の改行はあってもなくてもよい)
//! @attention	グローバル割り込み禁止状態で呼び出さないこと
void pf_uart_log(const char *format, ...);

//! @brief		UART割り込みハンドラ
//! @attention	データ競合(割り込み干渉)に注意する
void UARTE0_UART0_IRQHandler(void);

#endif  // PF_UART_H
