package mirrg.simulation.cart.almandine.mods.vanilla.parts;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Stroke;
import java.awt.geom.Ellipse2D;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.stream.Stream;

import mirrg.simulation.cart.almandine.GameAlmandine;
import mirrg.simulation.cart.almandine.facroty.property.IDialogProperty;
import mirrg.simulation.cart.almandine.factory.Factory;
import mirrg.simulation.cart.almandine.factory.IllegalEntityIdException;
import mirrg.simulation.cart.almandine.factory.Part;
import mirrg.simulation.cart.almandine.mods.vanilla.primaries.Cart;
import mirrg.simulation.cart.almandine.mods.vanilla.primaries.Cart.PositionRail;
import mirrg.simulation.cart.almandine.mods.vanilla.primaries.Cart.PositionStation;
import mirrg.simulation.cart.almandine.mods.vanilla.primaries.RailBase;

public abstract class StationBase extends Part
{

	@Deprecated
	public StationBase()
	{
		super();
	}

	public int x;
	public int y;
	public int radius;

	public StationBase(GameAlmandine game, Point point, int radius)
	{
		super(game);
		this.x = point.x;
		this.y = point.y;
		this.radius = radius;
	}

	@Override
	public boolean isHover(int x, int y)
	{
		int dist2 = (x - this.x) * (x - this.x) + (y - this.y) * (y - this.y);

		if (dist2 < (radius - 4) * (radius - 4)) return false;
		if (dist2 < (radius + 4) * (radius + 4)) return true;
		return false;
	}

	public boolean isContained(Rectangle rectangle) throws IllegalEntityIdException
	{
		return new Bound(x - radius, y - radius, radius * 2, radius * 2).contained(rectangle);
	}

	@Override
	public void draw(Graphics2D graphics)
	{
		Stroke stroke = graphics.getStroke();
		graphics.setStroke(new BasicStroke(3));

		graphics.setColor(new Color(getColor()));
		graphics.draw(new Ellipse2D.Double(x - radius, y - radius, radius * 2, radius * 2));

		graphics.setStroke(stroke);
	}

	protected abstract int getColor();

	@Override
	public void drawHover(Graphics2D graphics)
	{
		int radius = this.radius + 4;

		Stroke stroke = graphics.getStroke();
		graphics.setStroke(new BasicStroke(3));

		graphics.setColor(new Color(0x7bfc82));
		graphics.draw(new Ellipse2D.Double(x - radius, y - radius, radius * 2, radius * 2));

		graphics.setStroke(stroke);
	}

	public void move(int dx, int dy)
	{
		x += dx;
		y += dy;
	}

	@Override
	public Point getPoint() throws IllegalEntityIdException
	{
		return new Point(x, y);
	}

	/**
	 * @return 列の最後尾の直後を表す順序番号
	 */
	public int freeOrder()
	{
		OptionalInt max = getCarts()
			.mapToInt(cart -> ((PositionStation) cart.position).order)
			.max();
		return max.isPresent() ? max.getAsInt() + 1 : 0;
	}

	@Override
	protected void addProperty(IDialogProperty dialogProperty)
	{
		super.addProperty(dialogProperty);

		dialogProperty.addPropertyInt("X", "px", () -> x, x -> {
			this.x = x;
			return true;
		});
		dialogProperty.addPropertyInt("Y", "px", () -> y, y -> {
			this.y = y;
			return true;
		});
		dialogProperty.addPropertyInt("Radius", "px", () -> radius, radius -> {
			if (radius < 0) return false;
			this.radius = radius;
			return true;
		});
	}

	public static enum Direction
	{
		UNDEFINED(Math.PI * -0),
		RIGHT(Math.PI * -0 /* Y座標の定義が逆なため */),
		UP_RIGHT(Math.PI * -0.25),
		UP(Math.PI * -0.5),
		UP_LEFT(Math.PI * -0.75),
		LEFT(Math.PI * -1),
		DOWN_LEFT(Math.PI * -1.25),
		DOWN(Math.PI * -1.5),
		DOWN_RIGHT(Math.PI * -1.75), ;

		public final double angle;

		private Direction(double angle)
		{
			this.angle = angle;
		}

	}

	public Optional<RailBase> getRail(Direction direction)
	{
		return getRails()
			.min((a, b) -> (int) Math.signum(getRailAngleGap(a, direction)
				- getRailAngleGap(b, direction)));
	}

	private double getRailAngleGap(RailBase rail, Direction direction)
	{
		try {
			double angle = rail.getAngle();
			if (rail.idEnd == getId()) angle += Math.PI;
			angle -= direction.angle;
			while (angle < -Math.PI) {
				angle += Math.PI * 2;
			}
			while (angle > Math.PI) {
				angle -= Math.PI * 2;
			}
			return Math.abs(angle);
		} catch (IllegalEntityIdException e) {
			return Math.PI * 2;
		}
	}

	public Stream<RailBase> getRails()
	{
		return Factory.filterClass(game.factory.getEntities(), RailBase.class)
			.filter(rail -> rail.idBegin == getId() || rail.idEnd == getId());
	}

	public void emitCart(Cart cart, RailBase rail)
	{
		if (rail.idBegin == getId()) {
			cart.position = new PositionRail(rail.getId(), 0, true);
		} else {
			cart.position = new PositionRail(rail.getId(), 1, false);
		}

		updateOrder();
	}

	protected void updateOrder()
	{
		int[] order = {
			0,
		};

		getCarts()
			.forEach(cart -> {
				((PositionStation) cart.position).order = order[0];
				order[0]++;
			});
	}

	/**
	 * この駅にいるカートを先頭から順番に返す。
	 */
	public Stream<Cart> getCarts()
	{
		return Factory.filterClass(game.factory.getEntities(), Cart.class)
			.filter(cart -> cart.position instanceof PositionStation)
			.filter(cart -> ((PositionStation) cart.position).idStation == getId())
			.sorted((a, b) -> ((PositionStation) a.position).order - ((PositionStation) b.position).order);
	}

	public Optional<Cart> getCartPrimary()
	{
		return getCarts()
			.findFirst();
	}

}
