package project.web.generic;

import java.io.IOException;

import javax.servlet.http.HttpServletResponse;

import org.apache.logging.log4j.LogManager;

import com.fasterxml.jackson.databind.ObjectMapper;

import core.exception.LogicalException;
import online.context.check.InputCheck;
import online.struts.action.RestAction;
import online.struts.action.UniForm;
import project.check.range.LengthRangeCheck;
import project.svc.generic.QueryService;
import project.web.InstanceFactory;

/**
 * 選択肢アクション
 *
 * @author Tadashi Nakayama
 * @version 1.0.0
 */
public final class OptionAction implements RestAction {

	/**
	 * 取得処理
	 *
	 * @param model 汎用データモデル
	 * @param response レスポンス
	 * @return 処理結果
	 */
	public String get(final UniForm model, final HttpServletResponse response) {
		if (!model.hasQueryString()) {
			throw new LogicalException(HttpServletResponse.SC_BAD_REQUEST);
		}

		// 入力チェック
		checkInput(model);

		//DB情報取得
		final var ql = InstanceFactory.create(QueryService.class, model);
		if (!ql.search()) {
			throw new LogicalException(HttpServletResponse.SC_NOT_FOUND);
		}

		try {
			response.setContentType("application/json; charset=UTF-8");

			final var um = ql.getResultModel();
			final String[][] body = new String[][]{
				um.getStringArray(getOptionLabel(model)),
				um.getStringArray(getOptionValue(model))
			};
			new ObjectMapper().writeValue(response.getWriter(), body);

			response.flushBuffer();
		} catch (final IOException ex) {
			LogManager.getLogger().info(ex.getMessage());
		}

		return null;
	}

	/**
	 * ラベル項目名取得
	 * @param model 汎用モデル
	 * @return ラベル項目名
	 */
	private String getOptionLabel(final UniForm model) {
		final var label = model.getString("OptionLabelName");
		return label.isEmpty() ? "Label" : label;
	}

	/**
	 * 値項目名取得
	 * @param model 汎用モデル
	 * @return 値項目名
	 */
	private String getOptionValue(final UniForm model) {
		final var value = model.getString("OptionValueName");
		return value.isEmpty() ? "Value" : value;
	}

	/**
	 * 入力チェック
	 *
	 * @param model 汎用モデル
	 */
	private void checkInput(final UniForm model) {
		final var ic = InstanceFactory.create(InputCheck.class, model);
		final var pm = model.getActionParameter().getParameter();
		pm.keySet().forEach(k -> ic.add(k, new LengthRangeCheck(0, 1024)));
		ic.populate();
	}
}
