/*
 * (c) Copyright Sysdeo SA 2001, 2002.
 * All Rights Reserved.
 */

package com.sysdeo.eclipse.tomcat;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.JavaCore;


/**
 * See %TOMCAT6_HOME%/bin/catalina.bat
 */
public class Tomcat6Bootstrap extends TomcatBootstrap {

	/** isServerXmlMode */
	private final boolean serverXml;
	/** TomcatConfigFile */
	private final String configFile;
	/** isSecurityManagerEnabled */
	private final boolean security;
	/** TomcatClasspathVariable */
	private final IPath tomcatHomePath;

	/**
	 * constructor
	 * @param xml isServerXmlMode
	 * @param cf TomcatConfigFile
	 * @param manager isSecurityManagerEnabled
	 * @param home TomcatClasspathVariable
	 */
	public Tomcat6Bootstrap(final boolean xml, final String cf, final boolean manager, final IPath home) {
		this.serverXml = xml;
		this.configFile = cf;
		this.security = manager;
		this.tomcatHomePath = home;
	}

	/**
	 * @see TomcatBootstrap#getClasspath()
	 */
	@Override
	public String[] getClasspath() {
		ArrayList<String> classpath = new ArrayList<>();
		classpath.add(getTomcatDir() + File.separator + "bin" + File.separator + "bootstrap.jar");
		// Add tools.jar JDK file to classpath
		String toolsJarLocation = super.getVmInstalled().getInstallLocation()
						+ File.separator + "lib" + File.separator + "tools.jar";
		if (new File(toolsJarLocation).exists()) {
			classpath.add(toolsJarLocation);
		}
		return classpath.toArray(new String[classpath.size()]);
	}

	/**
	 * @see TomcatBootstrap#getMainClass()
	 */
	@Override
	public String getMainClass() {
		return "org.apache.catalina.startup.Bootstrap";
	}

	/**
	 * @see TomcatBootstrap#getStartCommand()
	 */
	@Override
	public String getStartCommand() {
		return "start";
	}

	/**
	 * @see TomcatBootstrap#getStopCommand()
	 */
	@Override
	public String getStopCommand() {
		return "stop";
	}

	/**
	 *
	 * @param command String
	 * @return String[]
	 */
	@Override
	public String[] getPrgArgs(final String command) {
		String[] prgArgs;
		if (this.serverXml) {
			prgArgs = new String[3];
			prgArgs[0] = "-config";
			prgArgs[1] = "\"" + this.configFile + "\"";
			prgArgs[2] = command;
		} else {
			prgArgs = new String[1];
			prgArgs[0] = command;
		}
		return prgArgs;
	}

	/**
	 * @see TomcatBootstrap#getVmArgs()
	 */
	@Override
	public String[] getVmArgs() {
		ArrayList<String> vmArgs = new ArrayList<>();
		vmArgs.add("-Dcatalina.home=\"" + getTomcatDir() + "\"");

		String version = System.getProperty("java.specification.version");
		if (Integer.parseInt(version) <= 8) {
			String endorsedDir = getTomcatDir() + File.separator + "endorsed";
			vmArgs.add("-Djava.endorsed.dirs=\"" + endorsedDir + "\"");
		}

		String catalinaBase = getTomcatBase();
		if (catalinaBase.isEmpty()) {
			catalinaBase = getTomcatDir();
		}

		vmArgs.add("-Dcatalina.base=\"" + catalinaBase + "\"");
		vmArgs.add("-Djava.io.tmpdir=\"" + catalinaBase + File.separator + "temp\"");

		if (this.security) {
			vmArgs.add("-Djava.security.manager");
			String securityPolicyFile = catalinaBase + File.separator + "conf" + File.separator + "catalina.policy";
			vmArgs.add("-Djava.security.policy=\"" + securityPolicyFile + "\"");
		}

		return vmArgs.toArray(new String[vmArgs.size()]);
	}

	/**
	 * @see TomcatBootstrap#getXMLTagAfterContextDefinition()
	 */
	@Override
	public String getXMLTagAfterContextDefinition() {
		return "</Host>";
	}

	/**
	 * @see com.sysdeo.eclipse.tomcat.TomcatBootstrap#getJasperJarPath()
	 */
	@Override
	public IPath getJasperJarPath() {
		return new Path("lib").append("jasper.jar");
	}

	/**
	 * @see com.sysdeo.eclipse.tomcat.TomcatBootstrap#getServletJarPath()
	 */
	@Override
	public IPath getServletJarPath() {
		return new Path("lib").append("servlet-api.jar");
	}

	/**
	 * @see com.sysdeo.eclipse.tomcat.TomcatBootstrap#getJSPJarPath()
	 */
	@Override
	public IPath getJSPJarPath() {
		return new Path("lib").append("jsp-api.jar");
	}

	/**
	 * @see com.sysdeo.eclipse.tomcat.TomcatBootstrap#getElJarPath()
	 */
	@Override
	public IPath getElJarPath() {
		return new Path("lib").append("el-api.jar");
	}

	/**
	 * @see com.sysdeo.eclipse.tomcat.TomcatBootstrap#getAnnotationsJarPath()
	 */
	@Override
	public IPath getAnnotationsJarPath() {
		return new Path("lib").append("annotations-api.jar");
	}

	/**
	 *
	 * @return Collection<IClasspathEntry>
	 */
	@Override
	public Collection<IClasspathEntry> getTomcatJars(final IPath homePath) {
		ArrayList<IClasspathEntry> jars = new ArrayList<>(super.getTomcatJars(homePath));
		jars.add(JavaCore.newVariableEntry(this.tomcatHomePath.append(getElJarPath()), null, null));
		jars.add(JavaCore.newVariableEntry(this.tomcatHomePath.append(getAnnotationsJarPath()), null, null));
		return jars;
	}

	/**
	 * @see TomcatBootstrap#getLabel()
	 */
	@Override
	public String getLabel() {
		return "Tomcat 6.x";
	}

	/**
	 * @see com.sysdeo.eclipse.tomcat.TomcatBootstrap#getContextWorkDir(java.lang.String)
	 */
	@Override
	public String getContextWorkDir(final String workFolder) {
		StringBuilder workDir = new StringBuilder("workDir=");
		workDir.append('"');
		workDir.append(workFolder);
		workDir.append('"');
		return workDir.toString();
	}
}
