/*
 * $Id: ConfigHelper.java 471754 2006-11-06 14:55:09Z husted $
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.struts.config;

import org.apache.struts.Globals;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionFormBean;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.apache.struts.action.ActionMessages;
import org.apache.struts.upload.MultipartRequestWrapper;
import org.apache.struts.util.MessageResources;
import org.apache.struts.util.RequestUtils;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

/**
 * <p> NOTE: THIS CLASS IS UNDER ACTIVE DEVELOPMENT. THE CURRENT CODE IS
 * WRITTEN FOR CLARITY NOT EFFICIENCY. NOT EVERY API FUNCTION HAS BEEN
 * IMPLEMENTED YET. </p><p> A helper object to expose the Struts shared
 * resources, which are be stored in the application, session, or request
 * contexts, as appropriate. </p><p> An instance should be created for each
 * request processed. The  methods which return resources from the request or
 * session contexts are not thread-safe. </p><p> Provided for use by other
 * servlets in the application so they can easily access the Struts shared
 * resources. </p><p> The resources are stored under attributes in the
 * application, session, or request contexts. </p><p> The ActionConfig methods
 * simply return the resources from under the context and key used by the
 * Struts ActionServlet when the resources are created. </p>
 *
 * @version $Rev: 471754 $ $Date: 2005-05-14 02:09:06 -0400 (Sat, 14 May 2005)
 *          $
 * @since Struts 1.1
 */
public class ConfigHelper implements ConfigHelperInterface {

    // --------------------------------------------------------  Properties

    /**
     * <p> The application associated with this instance. </p>
     */
    private ServletContext application = null;

    /**
     * <p> The session associated with this instance. </p>
     */
    private HttpSession session = null;

    /**
     * <p> The request associated with this instance. </p>
     */
    private HttpServletRequest request = null;

    /**
     * <p> The response associated with this instance. </p>
     */
    private HttpServletResponse response = null;

    /**
     * <p> The forward associated with this instance. </p>
     */
    private ActionForward forward = null;

    /**
     * constructor
     */
    public ConfigHelper() {
        super();
    }

    /**
     * @param ctx ServletContext
     * @param req HttpServletRequest
     * @param res HttpServletResponse
     */
    public ConfigHelper(final ServletContext ctx, final HttpServletRequest req,
            final HttpServletResponse res) {
        super();
        this.setResources(ctx, req, res);
    }

    /**
     * <p> Set the application associated with this instance.
     * [servlet.getServletContext()] </p>
     * @param ctx ServletContext
     */
    public void setApplication(final ServletContext ctx) {
        this.application = ctx;
    }

    /**
     * <p> Set the session associated with this instance. </p>
     * @param val HttpSession
     */
    public void setSession(final HttpSession val) {
        this.session = val;
    }

    /**
     * <p> Set the request associated with this object. Session object is also
     * set or cleared. </p>
     * @param req HttpServletRequest
     */
    public void setRequest(final HttpServletRequest req) {
        this.request = req;

        if (this.request == null) {
            setSession(null);
        } else {
            setSession(this.request.getSession(false));
        }
    }

    /**
     * <p> Set the response associated with this instance. Session object is
     * also set or cleared. </p>
     * @param res HttpServletResponse
     */
    public void setResponse(final HttpServletResponse res) {
        this.response = res;
    }

    /**
     * Set the forward associated with this instance.
     * @param val ActionForward
     */
    public void setForward(final ActionForward val) {
        this.forward = val;
    }

    /**
     * <p> Set the application and request for this object instance. The
     * ServletContext can be set by any servlet in the application. The
     * request should be the instant request. Most of the other methods
     * retrieve their own objects by reference to the application, request, or
     * session attributes. Do not call other methods without setting these
     * first! This is also called by the convenience constructor. </p>
     *
     * @param ctx - The associated ServletContext.
     * @param req     - The associated HTTP request.
     * @param res    - The associated HTTP response.
     */
    public void setResources(final ServletContext ctx,
        final HttpServletRequest req, final HttpServletResponse res) {
        setApplication(ctx);
        setRequest(req);
        setResponse(res);
    }

    // ------------------------------------------------ Application Context

    /**
     * @see org.apache.struts.config.ConfigHelperInterface#getActionMessages()
     */
    @Override
    public ActionMessages getActionMessages() {
        if (this.application == null) {
            return null;
        }

        return (ActionMessages) this.application.getAttribute(Globals.MESSAGE_KEY);
    }

    /**
     * <p> The application resources for this application. </p>
     */
    @Override
    public MessageResources getMessageResources() {
        if (this.application == null) {
            return null;
        }

        return (MessageResources) this.application.getAttribute(Globals.MESSAGES_KEY);
    }

    /**
     * <p> The path-mapped pattern (<code>/action/*</code>) or extension
     * mapped pattern ((<code>*.do</code>) used to determine our Action URIs
     * in this application. </p>
     */
    @Override
    public String getServletMapping() {
        if (this.application == null) {
            return null;
        }

        return (String) this.application.getAttribute(Globals.SERVLET_KEY);
    }

    // ---------------------------------------------------- Session Context

    /**
     * <p> The transaction token stored in this session, if it is used. </p>
     */
    @Override
    public String getToken() {
        if (this.session == null) {
            return null;
        }

        return (String) this.session.getAttribute(Globals.TRANSACTION_TOKEN_KEY);
    }

    // ---------------------------------------------------- Request Context

    /**
     * <p> The runtime JspException that may be been thrown by a Struts tag
     * extension, or compatible presentation extension, and placed in the
     * request. </p>
     */
    @Override
    public Throwable getException() {
        if (this.request == null) {
            return null;
        }

        return (Throwable) this.request.getAttribute(Globals.EXCEPTION_KEY);
    }

    /**
     * <p> The multipart object for this request. </p>
     */
    @Override
    public MultipartRequestWrapper getMultipartRequestWrapper() {
        if (this.request == null) {
            return null;
        }

        return (MultipartRequestWrapper) this.request.getAttribute(Globals.MULTIPART_KEY);
    }

    /**
     * <p> The <code>org.apache.struts.ActionMapping</code> instance for this
     * request. </p>
     */
    @Override
    public ActionMapping getMapping() {
        if (this.request == null) {
            return null;
        }

        return (ActionMapping) this.request.getAttribute(Globals.MAPPING_KEY);
    }

    // ---------------------------------------------------- Utility Methods

    /**
     * <p> Return true if a message string for the specified message key is
     * present for the user's Locale. </p>
     *
     * @param key Message key
     */
    @Override
    public boolean isMessage(final String key) {
        // Look up the requested MessageResources
        MessageResources resources = getMessageResources();

        if (resources == null) {
            return false;
        }

        // Return the requested message presence indicator
        return resources.isPresent(RequestUtils.getUserLocale(this.request, null), key);
    }

    /**
     * <p>
     * Retrieve and return the <code>ActionForm</code> bean associated with
     * this mapping, creating and stashing one if necessary.  If there is no
     * form bean associated with this mapping, return <code>null</code>.
     * </p>
     */
    @Override
    public ActionForm getActionForm() {
        // Is there a mapping associated with this request?
        ActionMapping mapping = getMapping();

        if (mapping == null) {
            return null;
        }

        // Is there a form bean associated with this mapping?
        String attribute = mapping.getAttribute();

        if (attribute == null) {
            return null;
        }

        // Look up the existing form bean, if any
        ActionForm instance = null;

        if ("request".equals(mapping.getScope())) {
            instance = (ActionForm) this.request.getAttribute(attribute);
        } else if (this.session != null) {
            instance = (ActionForm) this.session.getAttribute(attribute);
        }

        return instance;
    }

    /**
     * <p> Return the form bean definition associated with the specified
     * logical name, if any; otherwise return <code>null</code>. </p>
     *
     * @param name Logical name of the requested form bean definition
     */
    @Override
    public ActionFormBean getFormBean(final String name) {
        return null;
    }

    /**
     * <p> Return the forwarding associated with the specified logical name,
     * if any; otherwise return <code>null</code>. </p>
     *
     * @param name Logical name of the requested forwarding
     */
    @Override
    public ActionForward getActionForward(final String name) {
        return null;
    }

    /**
     * <p> Return the mapping associated with the specified request path, if
     * any; otherwise return <code>null</code>. </p>
     *
     * @param path Request path for which a mapping is requested
     */
    @Override
    public ActionMapping getActionMapping(final String path) {
        return null;
    }

    /**
     * <p> Return the form action converted into an action mapping path.  The
     * value of the <code>action</code> property is manipulated as follows in
     * computing the name of the requested mapping:</p>
     *
     * <ul>
     *
     * <li>Any filename extension is removed (on the theory that extension
     * mapping is being used to select the controller servlet).</li>
     *
     * <li>If the resulting value does not start with a slash, then a slash is
     * prepended.</li>
     *
     * </ul>
     */
    @Override
    public String getActionMappingName(final String action) {
        String value = action;
        int question = action.indexOf('?');

        if (question >= 0) {
            value = value.substring(0, question);
        }

        int slash = value.lastIndexOf('/');
        int period = value.lastIndexOf('.');

        if ((period >= 0) && (period > slash)) {
            value = value.substring(0, period);
        }

        if (value.startsWith("/")) {
            return value;
        }
        return "/" + value;

    }

    /**
     * <p> Return the form action converted into a server-relative URL. </p>
     */
    @Override
    public String getActionMappingURL(final String action) {
        StringBuilder value = new StringBuilder(this.request.getContextPath());

        // Use our servlet mapping, if one is specified
        String servletMapping = getServletMapping();

        if (servletMapping != null) {
            String queryString = null;
            int question = action.indexOf('?');

            if (question >= 0) {
                queryString = action.substring(question);
            }

            String actionMapping = getActionMappingName(action);

            if (servletMapping.startsWith("*.")) {
                value.append(actionMapping);
                value.append(servletMapping.substring(1));
            } else if (servletMapping.endsWith("/*")) {
                value.append(servletMapping, 0, servletMapping.length() - 2);
                value.append(actionMapping);
            }

            if (queryString != null) {
                value.append(queryString);
            }
        } else {
            // Otherwise, assume extension mapping is in use and extension is
            // already included in the action property
            if (!action.startsWith("/")) {
                value.append("/");
            }

            value.append(action);
        }

        // Return the completed value
        return value.toString();
    }

    /**
     * <p> Return the url encoded to maintain the user session, if any. </p>
     */
    @Override
    public String getEncodeURL(final String url) {
        if ((this.session != null) && (this.response != null)) {
            boolean redirect = false;

            if (this.forward != null) {
                redirect = this.forward.getRedirect();
            }

            if (redirect) {
                return this.response.encodeRedirectURL(url);
            }
            return this.response.encodeURL(url);

        }
        return url;
    }

    // ------------------------------------------------ Presentation API

    /**
     * <p> Renders the reference for a HTML <base> element </p>
     */
    @Override
    public String getOrigRef() {

        if (this.request == null) {
            return null;
        }

        StringBuilder result = RequestUtils.requestToServerUriStringBuilder(this.request);

        return result.toString();
    }

    /**
     * <p> Renders the reference for a HTML <base> element. </p>
     */
    @Override
    public String getBaseRef() {
        if (this.request == null) {
            return null;
        }

        StringBuilder result = RequestUtils.requestToServerStringBuilder(this.request);
        String path;

        if (this.forward == null) {
            path = this.request.getRequestURI();
        } else {
            path = this.request.getContextPath() + this.forward.getPath();
        }

        result.append(path);

        return result.toString();
    }

    /**
     * <p> Return the path for the specified forward, otherwise return
     * <code>null</code>. </p>
     *
     * @param name Name given to local or global forward.
     */
    @Override
    public String getLink(final String name) {
        ActionForward af = getActionForward(name);

        if (af == null) {
            return null;
        }

        // :TODO: What about runtime parameters?
        return getEncodeURL(this.request.getContextPath() + af.getPath());
    }

    /**
     * <p> Return the localized message for the specified key, otherwise
     * return <code>null</code>. </p>
     * @param key Message key
     * @return String
     */
    @Override
    public String getMessage(final String key) {
        MessageResources resources = getMessageResources();

        if (resources == null) {
            return null;
        }

        return resources.getMessage(
                RequestUtils.getUserLocale(this.request, null), key);
    }

    /**
     * <p> Look up and return a message string, based on the specified
     * parameters. </p>
     * @param key  Message key to be looked up and returned
     * @param args Replacement parameters for this message
     * @return String
     */
    @Override
    public String getMessage(final String key, final Object... args) {
        MessageResources resources = getMessageResources();

        if (resources == null) {
            return null;
        }

        // Return the requested message
        if (args == null) {
            return resources.getMessage(
                    RequestUtils.getUserLocale(this.request, null), key);
        }
        return resources.getMessage(
                RequestUtils.getUserLocale(this.request, null), key, args);

    }

    /**
     * <p> Return the URL for the specified ActionMapping, otherwise return
     * <code>null</code>. </p>
     * @param path Name given to local or global forward.
     * @return String
     */
    @Override
    public String getAction(final String path) {
        return getEncodeURL(getActionMappingURL(path));
    }

    // --------------------------------------------- Presentation Wrappers

    /**
     * <p> Wrapper for getLink(String) </p>
     * @param name Name given to local or global forward.
     * @return String
     */
    public String link(final String name) {
        return getLink(name);
    }

    /**
     * <p> Wrapper for getMessage(String) </p>
     * @param key Message key
     * @return String
     */
    public String message(final String key) {
        return getMessage(key);
    }

    /**
     * <p> Wrapper for getMessage(String,Object[]) </p>
     * @param key  Message key to be looked up and returned
     * @param args Replacement parameters for this message
     * @return String
     */
    public String message(final String key, final Object[] args) {
        return getMessage(key, args);
    }

    /**
     * <p> Wrapper for getAction(String) </p>
     * @param path Name given to local or global forward.
     * @return String
     */
    public String action(final String path) {
        return getAction(path);
    }
}
