/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2.base;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.chain2.Processing;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

/**
 * @version $Id$
 */
public class CopyCommandTestCase {

    /** CopyCommand */
    private CopyCommand<String, Object, Map<String, Object>> command;
    /** Context */
    private Map<String, Object> context = null;
    /** originalContext */
    private Map<String, Object> originalContext;

    /**
     * Setup
     */
    @Before
    public void setUp() {
        this.command = new CopyCommand<>();

        this.context = new HashMap<>();
        this.context.put("one", "one");
        this.context.put("two", "two");
        this.context.put("three", "three");

        this.originalContext = Collections.unmodifiableMap(new HashMap<>(this.context));
    }

    /**
     * Teardown
     */
    @After
    public void tearDown() {
        this.command = null;
        this.context = null;
    }

    /**
     * executeWithEmptyContextReturnsFalse
     */
    @Test
    public void executeWithEmptyContextReturnsFalse() {
        this.context.clear();
        execute();
    }

    /**
     * noToKeyDoesNotAlterContext
     */
    @Test
    public void noToKeyDoesNotAlterContext() {
        this.command.setFromKey("one");
        execute();

        Assert.assertEquals(this.originalContext, this.context);
    }

    /**
     * noToFromKeyDoesNotAlterContext
     */
    @Test
    public void noToFromKeyDoesNotAlterContext() {
        this.command.setToKey("one");
        execute();

        Assert.assertEquals(this.originalContext, this.context);
    }

    /**
     * bothKeysExistsValueIsCopied
     */
    @Test
    public void bothKeysExistsValueIsCopied() {
        this.command.setFromKey("one");
        this.command.setToKey("two");

        execute();

        Assert.assertEquals("one", this.context.get("one"));
        Assert.assertEquals("one", this.context.get("two"));
        Assert.assertEquals("three", this.context.get("three"));
        Assert.assertEquals(3, this.context.size());
    }

    /**
     * execute
     */
    private void execute() {
        // make sure execute always returns continue
        Assert.assertEquals(Processing.CONTINUE, this.command.execute(this.context));
    }

}
