/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2.testutils;

import java.util.stream.Stream;

import org.apache.commons.chain2.Catalog;
import org.apache.commons.chain2.Context;
import org.hamcrest.Description;
import org.hamcrest.Factory;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

/**
 * A matchers that checks if a catalog has an expected count of commands.
 *
 * @version $Id$
 */
public class HasCommandCount
        extends TypeSafeDiagnosingMatcher<Catalog<String, Object, Context<String, Object>>> {

    /** expected */
    private final int expected;

    /**
     * @param count int
     */
    public HasCommandCount(final int count) {
        this.expected = count;
    }

    /**
     * @param expectedCount int
     * @return Matcher
     */
    @Factory
    public static Matcher<Catalog<String, Object, Context<String, Object>>>
            hasCommandCount(final int expectedCount) {
        return new HasCommandCount(expectedCount);
    }

    /**
     * @see org.hamcrest.TypeSafeDiagnosingMatcher
     * #matchesSafely(java.lang.Object, org.hamcrest.Description)
     */
    @Override
    protected boolean matchesSafely(final Catalog<String, Object, Context<String, Object>> catalog,
            final Description mismatchDescription) {
        long actual = countCommands(catalog);
        if (actual != this.expected) {
            mismatchDescription.appendText(" catalog contains ").
                    appendValue(Long.valueOf(actual)).appendText(" commands ");
            return false;
        }
        return true;
    }

    /**
     * countCommands
     * @param catalog Catalog
     * @return long
     */
    private long countCommands(final Catalog<String, Object, Context<String, Object>> catalog) {
        Stream<String> names = catalog.getNames();
        names.forEach(name -> checkExists(catalog, name));
        return catalog.getNames().count();
    }

    /**
     * checkExists
     * @param catalog Catalog
     * @param name Name
     */
    private void checkExists(final Catalog<String, Object, Context<String, Object>> catalog,
            final String name) {
        if (catalog.getCommand(name) == null) {
            String msg = String.format(
                    "Catalog contains a command with name %s but getCommand(%s) returned null!",
                    name, name);
            throw new IllegalStateException(msg);
        }
    }

    /**
     * @see org.hamcrest.SelfDescribing#describeTo(org.hamcrest.Description)
     */
    @Override
    public void describeTo(final Description description) {
        description.appendText(" catalog contains ").
                appendValue(Integer.valueOf(this.expected)).appendText(" commands ");
    }

}
