/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.chain2.web.servlet;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.commons.chain2.web.MapEntry;

/**
 * <p>Private implementation of <code>Map</code> for HTTP session
 * attributes.</p>
 * @version $Id$
 */
final class ServletSessionScopeMap implements Map<String, Object>, Serializable {
    /** serialVersionUID */ private static final long serialVersionUID = 4102556809380028675L;
    public ServletSessionScopeMap(HttpServletRequest request) {
        this.request = request;
        sessionExists();
    }

    private transient HttpSession session = null;

    private transient HttpServletRequest request = null;

    @Override public void clear() {
        if (sessionExists()) {
            for (String key : keySet()) {
                session.removeAttribute(key);
            }
        }
    }

    @Override public boolean containsKey(Object key) {
        return sessionExists() && session.getAttribute(key(key)) != null;
    }

    @Override public boolean containsValue(Object value) {
        if (value == null || !sessionExists()) {
            return false;
        }
        @SuppressWarnings( "unchecked" ) // it is known that attribute names are String
        Enumeration<String> keys = session.getAttributeNames();
        while (keys.hasMoreElements()) {
            Object next = session.getAttribute(keys.nextElement());
            if (value.equals(next)) {
                return true;
            }
        }
        return false;
    }

    @Override public Set<Entry<String, Object>> entrySet() {
        Set<Entry<String, Object>> set = new HashSet<>();
        if (sessionExists()) {
            @SuppressWarnings( "unchecked" ) // it is known that attribute names are String
            Enumeration<String> keys = session.getAttributeNames();
            String key;
            while (keys.hasMoreElements()) {
                key = keys.nextElement();
                set.add(new MapEntry<>(key, session.getAttribute(key), true));
            }
        }
        return set;
    }

    @Override
    public boolean equals(Object o) {
        if (o != this) {
            if (!ServletSessionScopeMap.class.isInstance(o)) {
                return false;
            }
            ServletSessionScopeMap map = (ServletSessionScopeMap) o;
            return isSame(session, map.session)
                    && isSame(request, map.request);
        }
        return true;
    }

    @Override public Object get(Object key) {
        if (sessionExists()) {
            return session.getAttribute(key(key));
        }
        return null;
    }

    @Override
    public int hashCode() {
        if (sessionExists()) {
            return session.hashCode();
        }
        return 0;
    }

    @Override public boolean isEmpty() {
        return !sessionExists() || !session.getAttributeNames().hasMoreElements();
    }

    @Override public Set<String> keySet() {
        Set<String> set = new HashSet<>();
        if (sessionExists()) {
            @SuppressWarnings( "unchecked" ) // it is known that attribute names are String
            Enumeration<String> keys = session.getAttributeNames();
            while (keys.hasMoreElements()) {
                set.add(keys.nextElement());
            }
        }
        return set;
    }

    @Override public Object put(String key, Object value) {
        if (value == null) {
            return remove(key);
        }

        // Ensure the Session is created, if it
        // doesn't exist
        if (session == null) {
            session = request.getSession();
            request = null;
        }

        Object previous = session.getAttribute(key);
        session.setAttribute(key, value);
        return previous;
    }

    @Override public void putAll(Map<? extends String, ? extends Object> map) {
        for (Entry<? extends String, ? extends Object> entry : map.entrySet()) {
            put(key(entry.getKey()), entry.getValue());
        }
    }

    @Override public Object remove(Object key) {
        if (sessionExists()) {
            String skey = key(key);
            Object previous = session.getAttribute(skey);
            session.removeAttribute(skey);
            return previous;
        }
        return null;
    }

    @Override public int size() {
        int n = 0;
        if (sessionExists()) {
            @SuppressWarnings( "unchecked" ) // it is known that attribute names are String
            Enumeration<String> keys = session.getAttributeNames();
            while (keys.hasMoreElements()) {
                keys.nextElement();
                n++;
            }
        }
        return n;
    }

    @Override public Collection<Object> values() {
        List<Object> list = new ArrayList<>();
        if (sessionExists()) {
            @SuppressWarnings( "unchecked" ) // it is known that attribute names are String
            Enumeration<String> keys = session.getAttributeNames();
            while (keys.hasMoreElements()) {
                list.add(session.getAttribute(keys.nextElement()));
            }
        }
        return list;
    }

    private String key(Object key) {
        if (key == null) {
            throw new IllegalArgumentException();
        }
        if (key instanceof String) {
            return (String) key;
        }
        return key.toString();
    }

    private boolean sessionExists() {
        if (session == null) {
            session = request.getSession(false);
            if (session != null) {
                request = null;
            }
        }
        return session != null;
    }

    private boolean isSame(Object o1, Object o2) {
        return (o1 == o2 || (o1 != null && o1.equals(o2)));
    }

}
