package core.file;

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;

import common.db.JdbcSource;

import core.util.MojiUtil;

/**
 * テキストファイル書込クラス
 *
 * @author Tadashi Nakayama
 * @version 1.0.0
 */
public class EncodeOutputStreamWriter extends OutputStreamWriter {

	/** マッピング */
	private final Charset charset;

	/**
	 * コンストラクタ
	 *
	 * @param out 出力ストリーム
	 */
	public EncodeOutputStreamWriter(final OutputStream out) {
		super(out);
		this.charset = JdbcSource.getCharset();
	}

	/**
	 * コンストラクタ
	 *
	 * @param out 出力ストリーム
	 * @param cs キャラセット
	 */
	public EncodeOutputStreamWriter(final OutputStream out, final Charset cs) {
		super(out, cs);
		this.charset = cs;
	}

	/**
	 * コンストラクタ
	 *
	 * @param out 出力ストリーム
	 * @param enc キャラセットエンコーダ
	 */
	public EncodeOutputStreamWriter(final OutputStream out, final CharsetEncoder enc) {
		super(out, enc);
		this.charset = enc.charset();
	}

	/**
	 * @see java.io.OutputStreamWriter#write(int)
	 */
	@Override
	public void write(final int c) throws IOException {
		super.write(MojiUtil.correctGarbled(c, this.charset));
	}

	/**
	 * @see java.io.OutputStreamWriter#write(char[], int, int)
	 */
	@Override
	public void write(final char[] cbuf, final int off, final int len) throws IOException {
		final char[] c = new char[len];
		for (int i = 0; i < len; i++) {
			c[i] = (char)MojiUtil.correctGarbled(cbuf[off + i], this.charset);
		}
		super.write(c, 0, len);
	}

	/**
	 * @see java.io.OutputStreamWriter#write(java.lang.String, int, int)
	 */
	@Override
	public void write(final String str, final int off, final int len) throws IOException {
		if (str != null) {
			final String ret = MojiUtil.correctGarbled(str.substring(off, off + len), this.charset);
			super.write(ret, 0, len);
		}
	}
}
