/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package textkeymatcher.ui.model;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.jdesktop.application.AbstractBean;
import textkeymatcher.entity.KeyMatcher;

/**
 * メニューのチェックボックス付きメニューアイテムでキーマッチングを切り替える場合に便利なように
 * バイントサポートします.<br>
 * setText/setTextIgnoreCase/setNumericでは、引数enableがtrueである場合のみ設定されます.<br>
 * 引数がfalseの場合は単に無視して解除しません.別のタイプがsetでenableされたときに変更されます.<br>
 * @author seraphy
 */
public class KeyMatcherBinder extends AbstractBean {

    /**
     * ロガー
     */
    private static final Logger logger = Logger.getLogger(KeyMatcherBinder.class.getName());
    
    /**
     * キーマッチャをもつ実体オブジェクト
     */
    private DataViewTableModel tableModel;
    
    
    /**
     * 実体オブジェクトを設定します.<br>
     * プロパティ変更リスナーがセットアップされます.<br>
     * @param tableModel 
     */
    public void bindDataViewTableModel(DataViewTableModel tableModel) {
        if (tableModel == null) {
            throw new IllegalArgumentException();
        }
        this.tableModel = tableModel;
        this.tableModel.addPropertyChangeListener(new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent pce) {
                tableModelPropertyChanged(pce);
            }
        });
    }

    /**
     * キーマッチャが変更されたことを通知される.<br>
     * @param pce プロパティ変更イベント
     */
    protected void tableModelPropertyChanged(PropertyChangeEvent pce) {
        String propertyName = pce.getPropertyName();
        if (DataViewTableModel.PROPERTY_KEY_MATCHER.equals(propertyName)) {
            changeKeyMatcher(
                    (KeyMatcher) pce.getOldValue(),
                    (KeyMatcher) pce.getNewValue()
                    );
        }
    }
    
    protected KeyMatcher getKeyMatcher() {
        if (tableModel == null) {
            throw new IllegalStateException("先にsetDataViewTableModelで設定しておく必要があります.");
        }
        return tableModel.getKeyMatcher();
    }
    
    protected void setKeyMatcher(KeyMatcher keyMatcher) {
        if (tableModel == null) {
            throw new IllegalStateException("先にsetDataViewTableModelで設定しておく必要があります.");
        }
        tableModel.setKeyMatcher(keyMatcher); // thisにも変更通知がくることを想定
    }
    
    protected void changeKeyMatcher(KeyMatcher oldValue, KeyMatcher newValue) {
        logger.log(Level.INFO, ">change: {0}->{1}", new Object[] {oldValue, newValue});

        boolean oldText = oldValue == KeyMatcher.TEXT;
        boolean oldTextIC = oldValue == KeyMatcher.TEXT_IGNORECASE;
        boolean oldNumeric = oldValue == KeyMatcher.NUMERIC;
        
        boolean newText = newValue == KeyMatcher.TEXT;
        boolean newTextIC = newValue == KeyMatcher.TEXT_IGNORECASE;
        boolean newNumeric = newValue == KeyMatcher.NUMERIC;
        
        firePropertyChange("text", oldText, newText);
        firePropertyChange("textIgnoreCase", oldTextIC, newTextIC);
        firePropertyChange("numeric", oldNumeric, newNumeric);
    }
    
    public boolean isText() {
        return getKeyMatcher() == KeyMatcher.TEXT;
    }
    
    public boolean isTextIgnoreCase() {
        return getKeyMatcher() == KeyMatcher.TEXT_IGNORECASE;
    }
    
    public boolean isNumeric() {
        return getKeyMatcher() == KeyMatcher.NUMERIC;
    }
    
    public void setText(boolean enable) {
        logger.log(Level.INFO, "setText: {0}", enable);
        if (enable) {
            setKeyMatcher(KeyMatcher.TEXT);
        }
    }
    
    public void setTextIgnoreCase(boolean enable) {
        logger.log(Level.INFO, "setTextIgnoreCase: {0}", enable);
        if (enable) {
            setKeyMatcher(KeyMatcher.TEXT_IGNORECASE);
        }
    }
    
    public void setNumeric(boolean enable) {
        logger.log(Level.INFO, "setNumeric: {0}", enable);
        if (enable) {
            setKeyMatcher(KeyMatcher.NUMERIC);
        }
    }
}
