/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package textkeymatcher.io;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

/**
 * プロトコルバッファ形式での読み書きを行うクラス
 * @author seraphy
 */
public class ProtocolBufferesDocArchiveStrategy implements DocArchiveStrategy {
    
    /**
     * データ識別子.
     */
    private static final byte[] SIG;
    
    /**
     * クラスイニシャライザ
     */
    static {
        try {
            SIG = "TextKeyMatcher:1.0".getBytes("UTF-8");
        
        } catch (UnsupportedEncodingException ex) {
            // UTF-8エンコーディングは、すべてのJavaRuntimeに存在するので発生し得ない.
            throw new RuntimeException(ex);
        }
    }
    
    /**
     * プロトコルバッファ形式での書き込み
     * @param os 出力先
     * @param doc ドキュメント
     * @throws IOException 失敗
     */
    @Override
    public void write(OutputStream os, TextKeyMatcherDoc doc) throws IOException {
        if (os == null || doc == null) {
            throw new IllegalArgumentException();
        }

        // シグネチャの書き込み
        os.write(SIG);
        
        // プロトコルバッファの書き込み(GZIP圧縮)
        GZIPOutputStream gzipOutStm = new GZIPOutputStream(os);
        try {
            ProtocolBuffersWriter pbWriter = new ProtocolBuffersWriter();

            pbWriter.setTextKetMatcherDoc(doc);

            pbWriter.write(gzipOutStm);
            
        } finally {
            gzipOutStm.close();
        }
    }

    /**
     * プロトコルバッファ形式での読み込み
     * @param is 入力元
     * @return ドキュメント
     * @throws IOException 失敗
     */
    @Override
    public TextKeyMatcherDoc load(InputStream is) throws IOException {
        if (is == null) {
            throw new IllegalArgumentException();
        }

        // シグネチャの判定
        byte[] sig = new byte[SIG.length];
        int rd = is.read(sig);
        if (rd != SIG.length) {
            throw new IOException("Invalid format.");
        }
        if ( !Arrays.equals(sig, SIG)) {
            throw new IOException("Invalid format.");
        }
        
        // プロトコルバッファの読み込み(GZIP展開)
        TextKeyMatcherDoc doc;
        GZIPInputStream gzipInpStm = new GZIPInputStream(is);
        try {
            ProtocolBuffersReader pbReader = new ProtocolBuffersReader();
            pbReader.read(gzipInpStm);
            doc = pbReader;
            
        } finally {
            gzipInpStm.close();
        }
        
        return doc;
    }
    
}
