/* This is a program to generate calendars 
// for novels about Bobbie, Karel, Dan, and Kristie's world,
// specifically those in the Economics 101 trilogy.
//
// By Joel Matthew Rees, March 2017, 
// Copyright 2017, all rights reserved.
//
// Fair use acknowledged.
// If you want to use it for purposes other than personal entertainment, 
// rewriting it yourself is not that hard, 
// you'll understand the math and the science much better that way,
// and the result will satisfy your needs much more effectively.
//
// See
// http://joel-rees-economics.blogspot.com/2017/03/soc500-03-08-calendar-math.html
// http://joel-rees-economics.blogspot.com/2017/03/soc500-03-09-computer-calendar.html
// for more explanation of the code.
//
// Novel here:
// http://joel-rees-economics.blogspot.com/2017/01/soc500-00-00-toc.html
//
// From the intermediate text of the second draft of Sociology 500, a Novel,
// chapter 3 part 7, Family Games:
// *************************
// Uhm, lemmesee. Where did I put those notes? Oh, here they are.
// 353 days per year in about five years out of seven
// and 352 in the other two.
// So they have two skip years out of seven
// where we have one leap year out of four.
//
// But it isn't exact, of course.
// They have to adjust that again every 98 years
// and then once again every 343 years.
// Already too much detail, and you're falling asleep?
//
// But they have two moons. Did I mention that?
// The smaller moon orbits their earth
// in just under seven and an eighth days,
// and their larger moon orbits it
// in about twenty-eight and seven eighths days.
// About forty-nine and a half lunar weeks a year
// and about twelve and two fifths lunar months each year.
// *************************
//
// From the tool, econmonths.bc:
// *************************
// # If you want to do something similar, 
// # for looking at how leap years in your world  
// # match the actual orbits and revolutions
// # of your world and its moon,
// # replace isskip() with an appropriate isleap().
// # Left as an exercise for the reader.
// 
// define isskip( y, m ) {
//   if ( m != 0 ) {
//     return 0;
//   }
//   mem = scale;
//   scale = 0;
//   diff7 = y % 7;
//   diff98 = y % 98;
//   diff343 = y % 343;
//   scale = mem;
//   if ( diff98 == 48 ) {
//     return 1;
//   }
//   if ( ( diff7 != 1 ) && ( diff7 != 4 ) ) {
//     return 0;
//   }
//   if ( diff343 == 186 ) {
//     return 0;
//   }
//   return 1;
// }
// 
// 
// # Note that we are treating the first year as year 0 
// # and the first month as month 0.
// # For your earth, you will need to adjust the year and month input and output.
// *************************
//
// Note also that, in this calendar system,
// the day in the month and the day in the year start with day 0.
//
//
// Their two moons were at near zenith on the night their Christ was born, 
// in addition to the new star, but that's all I'm telling for now.
//
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <limits.h>
#if INT_MAX < 0x8000L /* sloppy 8 bit thinking */ 
#  define INT_BITS 16
#elif INT_MAX < 0x80000000L
#  define INT_BITS 32
#elif INT_MAX < 0x8000000000000000L
#  define INT_BITS 64
#else 
#  warning "INT_BITS greater than 64!!"
#endif


typedef unsigned char ubyte_t;
typedef signed char sbyte_t;


#define SPACE ' '


#define SKIPyears_per_short    2
#define FULLyears_per_short    5
#define YEARcycle_short        7
#define YEARcycle_medium       98  /* 7^2 * 2 */
#define YEARcycle_long         343 /* 7^3 */
#define YEARcycle_dbllong      686 /* 7^3 * 2 */

#define FULLYEARS_dbllongcycle \
( ( YEARcycle_dbllong / YEARcycle_short ) * FULLyears_per_short \
- ( YEARcycle_dbllong / YEARcycle_medium ) * 1 + ( YEARcycle_dbllong / YEARcycle_long ) * 1 )

#define SKIPYEARS_dbllongcycle ( YEARcycle_dbllong - FULLYEARS_dbllongcycle )


/* C double is (barely) sufficient resolution to avoid most of the tricks. */
#define YEARfraction    ( (double) FULLYEARS_dbllongcycle / YEARcycle_dbllong )
#define DAYSperYEARint  352
#define DAYSperYEAR     ( (double) DAYSperYEARint + YEARfraction )
#define DAYSperWEEK     7
#define MONTHSperYEAR   12

/* Do it in integers. 
*/
#define daysUPTOyear( year )	\
( tdays = (year) * DAYSperYEARint + ( ( year ) * FULLYEARS_dbllongcycle ) / YEARcycle_dbllong )

#define remainderUPTOyear( year ) \
( ( (year) * FULLYEARS_dbllongcycle ) % YEARcycle_dbllong )


#define SKIPmonth       0
#define SKIPyear_short_1    1
#define SKIPyear_short_2    4
#define SKIPyear_medium     48
#define LEAPyear_long    186	/* Must be short1 or short2 within the seven year cycle. */


/* Since skipyears are the exception, we test for skipyears instead of leapyears.
// Calendar system starts with year 0, not year 1.
// Would need to check and adjust if the calendar started with year 1.
*/
int isskipyr( long year )
{  int rem = year % YEARcycle_medium;
   if ( rem == SKIPyear_medium )
   {  return 1;
   }
   rem = year % YEARcycle_short;
   if ( ( rem != SKIPyear_small_1 ) && ( rem != SKIPyear_small_2 ) )
   {  return 0;
   }
   rem = year % YEARcycle_long;
   if ( rem == LEAPyear_long )	/* Must be short1 or short2 within the seven year cycle. */
   {  return 0;
   }
   return 1;
}


ubyte_t daysInMonths[ MONTHSperYEAR ] =
{  30, 
   29,
   30,
   29,
   29,
   30,
   29,
   30,
   29,
   29,
   30,
   29
};

#define daysInMonth( month, year ) \
( ( month != SKIPmonth ) ? daysInMonths[ (month) ] : daysInMonths[ (month) ] - ( isskipyr( (year) ) ? 1 : 0 ) )

/* months are 0 to MONTHSperYEAR - 1 here.
// If months were 1 to MONTHSperYEAR, would need to adjust here.
*/
int yearDaysToMonth( int month, long year )
{  unsigned total;
   int i;
/* */
   if ( month >= MONTHSperYEAR )
   {  month = MONTHSperYEAR - 1;
   }
   for ( total = 0, i = 0; i < month; ++i )
   {  total += daysInMonth( i, year );
   }
   return total;
}


/* day in month is 0 to max - 1 here.
// If day in month were 1 to max, would need to adjust here.
*/
unsigned long daysUpTo( long year, int month, int day, unsigned * remainderp )
{  unsigned long days = daysUPTOyear( year );
   unsigned partial = remainderUPTOyear( year );
/* */
   days += 
   * remainderp = partial;
   return days;
}


#define PLANETname "Bokadakr"

typedef char * str0_t; /* It looks like the pedants have attacked again. */
typedef str0_t str0array_t[]; /* Or I'm losing my memory again. */
typedef str0array_t * str0arrayptr_t; /* Wonder which. */

str0array_t weekDayNames =
{  "Sunday", 
   "Slowmoonday",
   "Aegisday",
   "Gefnday",
   "Freyday",
   "Tewesday", 
   "Vensday"
};

str0array_t fakeDayNames =
{  "Sunday", 
   "Monday",
   "Tuesday",
   "Wednesday",
   "Thursday",
   "Friday", 
   "Saturday"
};

str0array_t fakeMonthNames =
{  "January",
   "February",
   "March",
   "April",
   "May",
   "June",
   "July",
   "August",
   "September",
   "October",
   "November",
   "December",
};

str0array_t realMonthNames =
{  "Time-division",
   "Deep-winter",
   "War-time",
   "Thaw-time",
   "Rebirth",
   "Brides-month",
   "Imperious",
   "Senatorious",
   "False-summer",
   "Harvest",
   "Gratitude",
   "Winter-month",
};


void centeredName( char name[], int field )
{  int i;
   size_t length = strlen( name );
   int partial;
/* */
   if ( field < 1 ) 
   {  return;
   }
   if ( length > field )
   {  length = field;
   }
   for ( partial = field; partial > length; partial -= 2 )
   {  putchar( SPACE );
   }
   for ( i = 0; i < length; ++i )
   {  putchar( name[ i ] );
   }
   for ( partial = field - 1; partial > length; partial -= 2 ) /* weight right */
   {  putchar( SPACE );
   }   
}

void headerNames( str0array_t * daynames, int offset, int field )
{  int tracer;
/* */
   if ( offset >= DAYSperWEEK )
   {  offset = 0;
   }
   tracer = offset; 
   putchar( '|' );
   do 
   {  centeredName( (* daynames )[ tracer++ ], field );
      putchar( '|' );
      if ( tracer >= DAYSperWEEK )
      {  tracer = 0;
      }
   } while ( tracer != offset );
}

int intlog10( int number )
{  int result = 1;
   int temp = number;
/* */
   while ( temp > 0 )
   {  temp /= 10;
      ++result; 
   }
   return result;
}

void header( long year, int month, str0array_t * daynames, str0array_t * monthNames, int dayoffset, int terminalwidth )
{  int rightfield = ( terminalwidth - intlog10( year ) - intlog10( month ) - 4 - strlen( PLANETname ) ) / 2;
/* */    
   while ( rightfield-- > 0 )
   {  putchar ( SPACE );
   }
   printf( "%s: %ld %s", PLANETname, year, (* monthNames )[ month ] );
   putchar( '\n' );
   headerNames( daynames, dayoffset, ( ( terminalwidth - 1 ) / DAYSperWEEK ) - 1 ); 
   putchar( '\n' );
}

/* buffer must have room for field plus trailing NUL!
// ** That's one more than field! **
// Oh, for a dedicated parameter stack and returnable ad-hoc vectors.
// Returns the starting point for the conversion
*/
char * ulongtostring( char buffer[], unsigned long * numberp, int field, unsigned base )
{
   char * inpoint = buffer + field; /* Going to store a NUL here! */
   unsigned long number = * numberp;
/* */
   * inpoint = '\0';
   * --inpoint = '0';
   if ( number > 0 )
   {  ++inpoint;
      while ( ( number > 0 ) && ( inpoint > buffer ) )
      {  * --inpoint = '0' + number % base;
         number /= base;
      }
   }
   * numberp = number; /* leave any leftover. */
   return inpoint;
}

#define LPAREN '('
#define RPAREN ')' /* Balance to avoid confusing balancing editors, as well. */

char * longtostring( char buffer[], signed long * numberp, int field, unsigned base, char sign )
{  signed long number = * numberp;
   unsigned long value = ( number < 0 ) ? -number : number;
   char * converted = ulongtostring( buffer, &value, ( ( number < 0 ) && ( sign == LPAREN ) ) ? field - 1 : field, base );
/* */
   if ( ( number < 0 ) && ( converted > buffer ) )
   {  * --converted = sign;
      if ( sign == LPAREN ) /* sign is parenthesis */
      {  buffer[ field - 1 ] = RPAREN;
         buffer[ field ] = '\0';
      }
   }
   * numberp = ( number < 0 ) ? -value : value; /* leave any leftover. */
   return converted;
}

void centeredNumber( int number, int field, int base, char sign, char overfill )
{  int rightfield;
   int i;
   signed long value = number;
   char buffer[ field + 1 ]; /* one more for NUL */
   char * converted = longtostring( buffer, &value, field, base, sign );
   int numberwidth = field - ( converted - buffer );
/* */
/* printf( "\nnumber: %d, final value: %lu, converted: %s, field: %d\n", number, value, converted, field ); */
   if ( ( value > 0 ) || ( ( number < 0 ) && !( converted > buffer ) ) )
   {  char * p;
      for ( p = buffer + field; p > buffer; * --p = overfill ) 
      {  /* empty loop */ 
      }
   }
   rightfield = ( field - numberwidth ) / 2; /* bias left */
   for ( i = 0; i < rightfield; ++i ) 
   {  putchar( SPACE ); 
   }
   fputs ( converted, stdout );
   for ( i = rightfield + numberwidth; i < field; ++i )
   {  putchar( SPACE );
   }
}

int printOneWeek( int month, int day, int field, int offset )
{  int column;
   int endday = daysInMonths[ ( month > 0 ) ? month - 1 : MONTHSperYEAR - 1 ] - offset + 1;
/* */
   for ( column = 0; column < offset; ++column )
   {  putchar( '|' );
      centeredNumber( -( endday++ ), field - 1, 10, LPAREN, '*' );  
   }
/* */
   for ( /* as is */; ( column < DAYSperWEEK ) && ( day < daysInMonths[ month ] ); ++column )
   {  putchar( '|' );
      centeredNumber( day++, field - 1, 10, LPAREN, '*' );
   } 
/* */
   endday = 0;
   for ( /* as is */; column < DAYSperWEEK; ++column )
   {  putchar( '|' );
      centeredNumber( -( endday++ ), field - 1, 10, LPAREN, '*' );
   } 
/* */
   putchar( '|' );
   putchar( '\n' );
   return ( day < daysInMonths[ month ] ) ? day : -1;
}

void printOneMonth( long year, int month, int day, char fake, int wkstart, int terminalWidth )
{  long daysfromzero = 
/* */
   str0array_t * daynames = ( fake = 'f' ) ? &fakeDayNames : &weekDayNames;
   str0array_t * monthNames = ( fake = 'f' ) ? &fakeMonthNames : &realMonthNames;
/* */
   header( year, month, daynames, monthNames, wkstart, terminalWidth );
   day = printOneWeek( month, day, terminalWidth / 7, 2 );
   day = printOneWeek( month, day, terminalWidth / 7, 0 );
   day = printOneWeek( month, day, terminalWidth / 7, 0 );
   day = printOneWeek( month, day, terminalWidth / 7, 0 );
   day = printOneWeek( month, day, terminalWidth / 7, 0 );
   putchar( '\n' );
}


int setlong( long * target, char * source )
{  int validity = 0;
   char * parsept = source;
   long temp = strtol( source, &parsept, 0 );
/* */
   if ( parsept > source )
   {  validity = 1;
      * target = temp;
   }
/* */
/* printf( "setlong: %s:%s %ld: %d\n", source, parsept, *target, validity ); */
   return validity;
}


int setlongLimited( long * target, char * source, long low, long high )
{  long temp;
   int validity = setlong( &temp, source  );
/* */ 
   validity = validity && ( temp >= low ) && ( temp <= high );
   if ( validity )
   {  * target = temp;
   }
/* printf( "setlongLimited: %s %ld %ld %ld %ld: %d\n", source, low, temp, high, *target, validity ); */
   return validity;
}


int main( int argc, char * argv[] )
{
   long year = -1;
   long month = -1;
   int day = 0;
   long offset = 0;
   int startDay = 0;
   int argx = 1;
   char fake = 'f';
/* */
   long terminalWidth = 80;
/* */
   int valid = 1;
/* */
   while ( valid && ( argx < argc ) )
   {  
      if ( argv[ argx ][ 0 ] == '-' )
      {   switch ( argv[ argx ][ 1 ] )
          {
          case 'y':
             valid = setlong( &year, argv[ ++argx ] );
             break;
          case 'm':
             valid = setlongLimited( &month, argv[ ++argx ], 0, MONTHSperYEAR - 1 );
             break;
          case 'o':
             valid = setlongLimited( &offset, argv[ ++argx ], 0, DAYSperWEEK - 1 );
             break;
          case 'w':
             valid = setlongLimited( &terminalWidth, argv[ ++argx ], 0, LONG_MAX );
             break;
          case 'f':
          case 'r':
             fake = argv[ argx ][ 1 ];
             break;
          }
      }
      else
      {  if ( year < 0 )
         {  valid = setlong( &year, argv[ argx ] );
         }
         else if ( month < 0 )
         {  valid = setlongLimited( &month, argv[ argx ], 0, MONTHSperYEAR - 1 );
         }
      } 
      ++argx;
   }

   if ( !valid ) /* Not really quite accurate summary. */
   {  printf( "%s { [ <year> [ <month> ] ] }\n", 
              argv[ 0 ] );
      puts( "\t | { [ -f(ake) | -r(eal) ] [ -y <year> ] [ -m <month> ] [ -o <weekday-offset> ] [ -w <terminal-width> ] }" );
      return EXIT_FAILURE;
   }


   printOneMonth( year, (int) month, day, fake, offset, terminalWidth );

   return EXIT_SUCCESS;   
}
